﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/err/err_ErrorInfo.h>
#include <nn/nn_Log.h>
#include <nn/settings.h>
#include <nn/settings/system/settings_Language.h>
#include <nn/util/util_FormatString.h>

#include "testErr_Util.h"

using namespace nn;

/*
    DummyErrorMessage をインストールした状態で実行することを前提にしたテスト。
    Windows 版を実行する場合は C:\Temp\ErrorMessage 以下にメッセージデータをコピーする。
*/

namespace {

    // 指定された言語を、使用するエラーメッセージの言語に変換する（"エラーメッセージの存在する言語の数 <= 本体で設定できる言語の数" であるため）。
    // 言語の増減があった場合は Programs\Eris\Sources\Libraries\err\detail\err_SystemData.cpp に合わせて修正すること。
    settings::LanguageCode ConvertToErrorMessageLanguageCode(const settings::LanguageCode& languageCode)
    {
        if( false
            || languageCode == settings::Language_Japanese              //!< 日本語 (ja)
            || languageCode == settings::Language_AmericanEnglish       //!< 米国英語 (en-US)
            || languageCode == settings::Language_French                //!< フランス語 (fr)
            || languageCode == settings::Language_German                //!< ドイツ語 (de)
            || languageCode == settings::Language_Italian               //!< イタリア語 (it)
            || languageCode == settings::Language_Spanish               //!< スペイン語 (es)
         // || languageCode == settings::Language_Chinese               //!< 中国語 (zh-CN)
         // || languageCode == settings::Language_Korean                //!< 韓国語 (ko)
            || languageCode == settings::Language_Dutch                 //!< オランダ語 (nl)
            || languageCode == settings::Language_Portuguese            //!< ポルトガル語 (pt)
            || languageCode == settings::Language_Russian               //!< ロシア語 (ru)
         // || languageCode == settings::Language_Taiwanese             //!< 台湾語 (zh-TW)
            || languageCode == settings::Language_BritishEnglish        //!< 英国英語 (en-GB)
            || languageCode == settings::Language_CanadianFrench        //!< カナダフランス語 (fr-CA)
            || languageCode == settings::Language_LatinAmericanSpanish  //!< 中南米スペイン語 (es-419)
         // || languageCode == settings::Language_SimplifiedChinese     //!< 簡体字中国語 (zh-Hans)
         // || languageCode == settings::Language_TraditionalChinese    //!< 繁体字中国語 (zh-Hant)
            )
        {
            return languageCode;
        }
        else
        {
            // エラーメッセージが存在しない言語は一律で英語にフォールバック。
            return settings::LanguageCode::Make(settings::Language_AmericanEnglish);
        }
    }

    // 全言語分のメッセージをチェックする。
    // メッセージの実体を持つエラーコードの場合は actualErrorCode と messageErrorCode に同じものを渡す。
    void CheckErrorInfo(const err::ErrorCode& actualErrorCode, const err::ErrorCode& messageErrorCode, bool isServerError)
    {
        const int LanguageCodeCountMax = 24;
        int availableLanguageCodeCount = settings::GetAvailableLanguageCodeCount();
        ASSERT_GE(LanguageCodeCountMax, availableLanguageCodeCount);

        settings::LanguageCode languageCodes[LanguageCodeCountMax];
        auto actualLanguageCodeCount = settings::GetAvailableLanguageCodes(languageCodes, LanguageCodeCountMax);

        for( int i = 0; i < actualLanguageCodeCount; i++ )
        {
            auto languageCode = ConvertToErrorMessageLanguageCode(languageCodes[i]);
            settings::system::SetLanguageCode(languageCode);

            err::ErrorInfo errorInfo(actualErrorCode);

            EXPECT_TRUE(errorInfo.IsValid());
            // nnt::err::PrintErrorInfo(errorInfo);

            // エラーコードは常に実際に指定されたもの。
            char errorCodeStrUtf8[err::ErrorCode::StringLengthMax];
            util::SNPrintf(errorCodeStrUtf8, NN_ARRAY_SIZE(errorCodeStrUtf8), "%04u-%04u", actualErrorCode.category, actualErrorCode.number);
            char16_t errorCodeStrUtf16[err::ErrorCode::StringLengthMax];
            errorInfo.GetErrorCodeString(errorCodeStrUtf16, err::ErrorCode::StringLengthMax);
            EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(errorCodeStrUtf8, errorCodeStrUtf16));

            // その他メッセージは参照先のもの。
            // DummyErrorMessage のフォーマットは "(エラーコード) : (言語コード) : 種類"。
            char message[64];

            util::SNPrintf(message, NN_ARRAY_SIZE(message), "%04u-%04u : %s : DlgMsg", messageErrorCode.category, messageErrorCode.number, languageCode.string);
            EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(message, errorInfo.GetDialogViewUiSettings().message));
            for( int j = 0; j < 3; j++ )
            {
                util::SNPrintf(message, NN_ARRAY_SIZE(message), "%04u-%04u : %s : DlgB%d", messageErrorCode.category, messageErrorCode.number, languageCode.string, j);
                EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(message, errorInfo.GetDialogViewUiSettings().buttonMessage[j]));
            }
            util::SNPrintf(message, NN_ARRAY_SIZE(message), "%04u-%04u : %s : FlvMsg", messageErrorCode.category, messageErrorCode.number, languageCode.string);
            EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(message, errorInfo.GetFullScreenViewUiSettings().message));
            for( int j = 0; j < 3; j++ )
            {
                util::SNPrintf(message, NN_ARRAY_SIZE(message), "%04u-%04u : %s : FlvB%d", messageErrorCode.category, messageErrorCode.number, languageCode.string, j);
                EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(message, errorInfo.GetFullScreenViewUiSettings().buttonMessage[j]));
            }

            if( isServerError )
            {
                EXPECT_TRUE(errorInfo.IsServerError());
                // DummyErrorMessage のサーバーコードのフォーマットはハイフン抜きのエラーコード（8文字までなので）
                char serverCode[9];
                util::SNPrintf(serverCode, NN_ARRAY_SIZE(serverCode), "%04u%04u", actualErrorCode.category, actualErrorCode.number);
                EXPECT_STREQ(serverCode, errorInfo.GetServerCode());
            }
            else
            {
                EXPECT_FALSE(errorInfo.IsServerError());
            }
        }
    }
}

// システムデータにメッセージの実体があるエラーコード
TEST(ErrorInfo, SystemDataMessage)
{
    err::ErrorCode errorCode{ 2001, 0000 };
    CheckErrorInfo(errorCode, errorCode, true);
}

// システムデータの他のメッセージを参照しているエラーコード
TEST(ErrorInfo, SystemDataMessageReference)
{
    err::ErrorCode actualErrorCode{ 2001, 0001 };
    err::ErrorCode messageErrorCode{ 2001, 0000 };
    CheckErrorInfo(actualErrorCode, messageErrorCode, true);
}

// システムデータにメッセージの実体がなく、カテゴリのデフォルトがあるエラーコード
TEST(ErrorInfo, SystemDataCategoryDefault)
{
    err::ErrorCode actualErrorCode{ 2001, 0002 };
    err::ErrorCode messageErrorCode{ 2001, 9999 };
    CheckErrorInfo(actualErrorCode, messageErrorCode, false);
}

// システムデータにメッセージの実体がなく、カテゴリのデフォルトもないエラーコード
TEST(ErrorInfo, SystemDataDefault)
{
    // カテゴリが存在するが、デフォルトが存在しない。
    {
        err::ErrorCode actualErrorCode{ 2002, 0001 };
        err::ErrorCode messageErrorCode{ 2999, 9999 };
        CheckErrorInfo(actualErrorCode, messageErrorCode, false);
    }

    // カテゴリ自体が存在しない
    {
        err::ErrorCode actualErrorCode{ 2003, 0000 };
        err::ErrorCode messageErrorCode{ 2999, 9999 };
        CheckErrorInfo(actualErrorCode, messageErrorCode, false);
    }
}

// メッセージ文字列直接指定
TEST(ErrorInfo, Message)
{
    auto dialogMessage = "DialogMessage";
    auto fullScreenMessage = u8"全画面メッセージ";
    err::ErrorCode errorCode{ 2345, 6789 };
    err::ErrorInfo errorInfo(errorCode, dialogMessage, fullScreenMessage, settings::LanguageCode::Make(settings::Language_Japanese));
    // nnt::err::PrintErrorInfo(errorInfo);

    EXPECT_TRUE(errorInfo.IsValid());

    char16_t errorCodeStr[err::ErrorCode::StringLengthMax];
    errorInfo.GetErrorCodeString(errorCodeStr, err::ErrorCode::StringLengthMax);
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(u8"2345-6789", errorCodeStr));

    // メッセージは指定したもの、ボタンのメッセージは指定した言語のもの（TODO: 他言語のテスト）
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(dialogMessage, errorInfo.GetDialogViewUiSettings().message));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(u8"くわしく", errorInfo.GetDialogViewUiSettings().buttonMessage[0]));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(u8"とじる", errorInfo.GetDialogViewUiSettings().buttonMessage[1]));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(fullScreenMessage, errorInfo.GetFullScreenViewUiSettings().message));
    EXPECT_TRUE(nnt::err::CompareUtf8AndUtf16(u8"OK", errorInfo.GetFullScreenViewUiSettings().buttonMessage[0]));

    // メッセージ指定にサーバーコードはない
    EXPECT_FALSE(errorInfo.IsServerError());
}

#if 0
// 内部で ABORT に引っかからないかどうかのテスト。超時間がかかる。
TEST(ErrorInfo, DISABLED_RunThroughAllErrorCode)
{
    for( err::ErrorCodeCategory c = 2001; c <= 2999; c++ )
    {
        for( err::ErrorCodeNumber n = 0; n <= 9999; n++ )
        {
            NN_LOG("%04u-%04u\n", c, n);
            err::ErrorCode errorCode{ c, n };
            err::ErrorInfo errorInfo(errorCode);
            // nnt::err::PrintErrorInfo(errorInfo);
        }
    }
}
#endif
