﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ae/ae.h>
#include <nn/ae/ae_Types.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/err_ApiForApplet.h>
#include <nn/err/err_ApiForErrorViewer.h>
#include <nn/err/err_ErrorInfo.h>
#include <nn/fs.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_SdkLog.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/la/la_Api.h>
#include <nn/la/la_CommonArgumentsReader.h>
#include <nn/gfx.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/time.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "testErr_Util.h"

using namespace nn;

NN_ALIGNAS(4096) uint8_t  g_MallocBuffer[1 * 1024 * 1024];
extern "C" void nninitStartup()
{
    nn::init::InitializeAllocator(g_MallocBuffer, sizeof(g_MallocBuffer));
}

namespace
{
    nn::Bit8 errorViewerAppletParam[err::ErrorViewerStartupParamSizeMax];
    nn::Bit8 recordErrorWorkBuffer[err::RequiredBufferSizeForRecordError];
    err::EulaData eulaData; // 本体更新後のEULA用。
    ns::ApplicationControlProperty applicationControlProperty;
}

void LibraryAppletMenuMain(const nn::ae::LibraryAppletSelfInfo& info) NN_NOEXCEPT
{
    NN_LOG("\n====== Dummy ErrorViewer ======\n");

    nn::time::Initialize();
    NN_UTIL_SCOPE_EXIT{ nn::time::Finalize(); };
    nn::ns::Initialize();


#if 0
    // TORIAEZU : エラービューア以外の LA 用の API のテストをここで実行。
    NN_LOG("\n====== Test APIs for LA (Start) ======\n");
    {
        err::ErrorCode errorCode = { 2345, 6789 };
        const char* message = "RecordErrorForTelemetry";
        err::RecordErrorForTelemetry(errorCode, message, recordErrorWorkBuffer, sizeof(recordErrorWorkBuffer));
    }
    {
        err::ErrorCode errorCode = { 2345, 9876 };
        const char* message = "RecordError";
        err::RecordError(errorCode, message, recordErrorWorkBuffer, sizeof(recordErrorWorkBuffer));
    }
    {
        // err::ErrorContextType::Http
        err::ErrorCode errorCode = { 2900, 0000 };
        err::ErrorContext errorContext;
        errorContext.type = err::ErrorContextType::Http;
        util::Strlcpy(errorContext.http.fqdn, "fqdn", static_cast<int>(sizeof(errorContext.http.fqdn)));
        util::Strlcpy(errorContext.http.ip, "ip", static_cast<int>(sizeof(errorContext.http.ip)));
        ae::ReportVisibleError(errorCode, errorContext);
    }
    {
        // err::ErrorContextType::FileSystem
        err::ErrorCode errorCode = { 2900, 0001 };
        err::ErrorContext errorContext;
        errorContext.type = err::ErrorContextType::FileSystem;
        std::strncpy(errorContext.fileSystem.path, "mount:/directory/subdirectory/subsubdirectory/file", sizeof(errorContext.fileSystem.path));
        errorContext.fileSystem.fsResultValue = fs::ResultPathNotFound().GetInnerValueForDebug();
        ae::ReportVisibleError(errorCode, errorContext);
    }
    {
        // err::ErrorContextType::WebMediaPlayer
        err::ErrorCode errorCode = { 2900, 0002 };
        err::ErrorContext errorContext;
        errorContext.type = err::ErrorContextType::WebMediaPlayer;
        std::strncpy(errorContext.webMediaPlayer.openUrl, "https://somewhere/where/webmedia/opened?param0=x&param1=y", sizeof(errorContext.webMediaPlayer.openUrl));
        errorContext.webMediaPlayer.lastSocketErrors[0] = 1;
        errorContext.webMediaPlayer.lastSocketErrors[1] = 2;
        errorContext.webMediaPlayer.lastSocketErrors[2] = 3;
        errorContext.webMediaPlayer.lastSocketErrors[3] = 4;
        ae::ReportVisibleError(errorCode, errorContext);
    }
    NN_LOG("\n====== Test APIs for LA (Done) ======\n");
#endif

    // ライブラリアプレットの共通パラメータ
    la::CommonArgumentsReader reader;
    NN_ABORT_UNLESS(reader.TryPopFromInChannel(), "CommonArgumentsReader");

    // エラービューアアプレット用の共通パラメータ
    nn::applet::StorageHandle handle;
    NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&handle), "ReadErrorViewerStartUpParam failed to get input data.\n");
    auto outBufferSize = nn::applet::GetStorageSize(handle);
    NN_SDK_ASSERT(outBufferSize <= 1024 * 5);
    NN_UNUSED(outBufferSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(handle, 0, errorViewerAppletParam, outBufferSize));
    nn::applet::ReleaseStorage(handle);

    auto type = nn::err::GetErrorType(errorViewerAppletParam);

    switch( type )
    {
        case nn::err::ErrorType::SystemData:
        {
            NN_LOG("ErrorType : SystemData\n");
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam);
            nnt::err::PrintErrorInfo(errorInfo);
            nn::err::ProcessErrorInfoAttributes(errorInfo);
            nn::err::RecordError(errorViewerAppletParam, recordErrorWorkBuffer, err::RequiredBufferSizeForRecordError);
        }
        break;

    case nn::err::ErrorType::SystemError:
        {
            NN_LOG("ErrorType : SystemError\n");
            auto lang = nn::err::GetMessageLanguageCode(errorViewerAppletParam);
            NN_LOG("MessageLanguage : %s\n", lang.string);
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam);
            nnt::err::PrintErrorInfo(errorInfo);
            nn::err::RecordError(errorViewerAppletParam, recordErrorWorkBuffer, err::RequiredBufferSizeForRecordError);
        }
        break;

    case nn::err::ErrorType::ApplicationError:
        {
            NN_LOG("ErrorType : ApplicationError\n");
            auto lang = nn::err::GetMessageLanguageCode(errorViewerAppletParam);
            NN_LOG("MessageLanguage : %s\n", lang.string);
            NN_ABORT_UNLESS_RESULT_SUCCESS(ae::GetMainAppletApplicationControlProperty(&applicationControlProperty));
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam, applicationControlProperty.applicationErrorCodeCategory);
            nnt::err::PrintErrorInfo(errorInfo);
            nn::err::RecordError(errorViewerAppletParam, recordErrorWorkBuffer, err::RequiredBufferSizeForRecordError);
        }
        break;

    case nn::err::ErrorType::Eula:
        {
            NN_LOG("ErrorType : Eula\n");
            NN_LOG("Region : %d\n", nn::err::GetEulaRegionCode(errorViewerAppletParam));
        }
        break;

    case nn::err::ErrorType::ParentalControl:
        {
            auto result = nn::err::GetParentalControlResultRestricted(errorViewerAppletParam);
            NN_LOG("ErrorType : ParentaoControl\n");
            NN_LOG("Restricted Feature : %03d-%04d (0x%08x)\n", result.GetModule(), result.GetDescription(), result.GetInnerValueForDebug());
        }
        break;

    case nn::err::ErrorType::RecordedSystemData:
        {
            NN_LOG("ErrorType : RecordedSystemData\n");
            nnt::err::PrintTimeOfOccurrence(errorViewerAppletParam);
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam);
            nnt::err::PrintErrorInfo(errorInfo);
        }
        break;

    case nn::err::ErrorType::RecordedSystemError:
        {
            NN_LOG("ErrorType : RecordedSystemError\n");
            nnt::err::PrintTimeOfOccurrence(errorViewerAppletParam);
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam);
            nnt::err::PrintErrorInfo(errorInfo);
        }
        break;

    case nn::err::ErrorType::RecordedApplicationError:
        {
            NN_LOG("ErrorType : RecordedApplicationError\n");
            nnt::err::PrintTimeOfOccurrence(errorViewerAppletParam);
            nn::err::ErrorInfo errorInfo(errorViewerAppletParam);
            nnt::err::PrintErrorInfo(errorInfo);
        }
        break;

    case nn::err::ErrorType::SystemUpdateEula:
        {
            NN_LOG("ErrorType : SystemUpdateEula\n");
            NN_LOG("Region : %d\n", nn::err::GetEulaRegionCode(errorViewerAppletParam));

            nn::applet::StorageHandle handle;
            NN_ABORT_UNLESS(nn::ae::TryPopFromInChannel(&handle), "TryPop for SystemUpdateEula.\n");
            auto outBufferSize = nn::applet::GetStorageSize(handle);
            NN_UNUSED(outBufferSize);
            NN_SDK_ASSERT(outBufferSize == sizeof(err::EulaData));
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::applet::ReadFromStorage(handle, 0, &eulaData, sizeof(eulaData)));
            nn::applet::ReleaseStorage(handle);

            NN_LOG("DataCount : %d\n", eulaData.dataCount);
            for(int i = 0; i < eulaData.dataCount; i++)
            {
                NN_LOG("Language = %d, Size = %llu\n", eulaData.data[i].language, eulaData.data[i].size);
            }
        }
        break;

    default:
        NN_UNEXPECTED_DEFAULT;

    }
} // NOLINT(impl/function_size)

extern "C" void nnMain()
{
    nn::ae::InvokeLibraryAppletMain(LibraryAppletMenuMain);
}
