﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testErr_Util.h"

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/err/err_ApiForErrorViewer.h>
#include <nn/err/err_ErrorInfo.h>
#include <nn/err/err_ErrorViewerAppletParam.h>
#include <nn/err/detail/err_SystemData.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/time.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>

using namespace nn;

namespace nnt { namespace err {

namespace {
    const int Utf8BufferLength = 1024 * 4;
    char s_Utf8Buffer[Utf8BufferLength];
}

void NnLogUtf16(const char16_t* utf16str) NN_NOEXCEPT
{
    auto result = util::ConvertStringUtf16NativeToUtf8(s_Utf8Buffer, Utf8BufferLength, reinterpret_cast<const uint16_t*>(utf16str));
    if( result != util::CharacterEncodingResult_Success )
    {
        NN_LOG("Failed to convert : %d\n", result);
        return;
    }
    NN_LOG("%s", s_Utf8Buffer);
}

bool CompareUtf8AndUtf16(const char* utf8str, const char16_t* utf16str) NN_NOEXCEPT
{
    auto result = util::ConvertStringUtf16NativeToUtf8(s_Utf8Buffer, Utf8BufferLength, reinterpret_cast<const uint16_t*>(utf16str));
    if( result != util::CharacterEncodingResult_Success )
    {
        NN_LOG("CompareUtf8AndUtf16 : Failed to convert to utf8 (CharacterEncodingResult = %d)\n", result);
        return false;
    }
    if( strncmp(utf8str, s_Utf8Buffer, sizeof(s_Utf8Buffer)) != 0 )
    {
        NN_LOG("CompareUtf8AndUtf16 : %s vs %s\n", utf8str, s_Utf8Buffer);
        return false;
    }
    return true;
}

void DumpErrorMessage(const char16_t* message, int messageLength) NN_NOEXCEPT
{
    char utf8[4]{};
    uint16_t utf16[2]{};
    for( int i = 0; i < messageLength; i++ )
    {
        utf16[0] = static_cast<uint16_t>(message[i]); // 2つ使うものは未対応。
        auto result = util::ConvertCharacterUtf16NativeToUtf8(utf8, utf16);
        if( result == util::CharacterEncodingResult_Success )
        {
            NN_LOG("%04X(%s) ", message[i], message[i] == 0x000A ? "\\n" : utf8);
        }
        else
        {
            NN_LOG("%04X(--) ", message[i]);
        }
        if( i % 8 == 7 )
        {
            NN_LOG("\n");
        }
    }
}

void PrintUiSettings(const nn::err::ErrorInfo::UiSettings& uiSettings) NN_NOEXCEPT
{
    NN_LOG("MessageLength : %d\n", uiSettings.messageLength);
    NnLogUtf16(uiSettings.message); // MessageStudio 用のタグなどで途中に 0x0000 がきた場合はそこまでしか表示されない。
    NN_LOG("\n");
    DumpErrorMessage(uiSettings.message, uiSettings.messageLength);
    NN_LOG("\n\nButton Count = %d\n", uiSettings.buttonCount);
    for( uint8_t i = 0; i < uiSettings.buttonCount; i++ )
    {
        NN_LOG("    * ");
        NnLogUtf16(uiSettings.buttonMessage[i]);
        NN_LOG(" (%d)\n", uiSettings.buttonAction[i]);
    }
}

void PrintErrorInfo(const nn::err::ErrorInfo& info) NN_NOEXCEPT
{
    if( !info.IsValid() )
    {
        NN_LOG("Invalid ErrorInfo\n");
        return;
    }
    char16_t errorCodeString[nn::err::ErrorCode::StringLengthMax];
    info.GetErrorCodeString(errorCodeString, static_cast<size_t>(nn::err::ErrorCode::StringLengthMax));
    NN_LOG("================================\n");
    NN_LOG("ErrorCode : ");
    NnLogUtf16(errorCodeString);
    NN_LOG("\n");
    NN_LOG("--------------------------------\n");
    NN_LOG("[Dialog]\n");
    if( info.IsDialogViewSupported() )
    {
        PrintUiSettings(info.GetDialogViewUiSettings());
    }
    else
    {
        NN_LOG("Not Supported.\n");
    }
    NN_LOG("--------------------------------\n");
    NN_LOG("[FullScreen]\n");
    if( info.IsFullScreenViewSupported() )
    {
        PrintUiSettings(info.GetFullScreenViewUiSettings());
    }
    else
    {
        NN_LOG("Not Supported.\n");
    }
    if( info.IsServerError() )
    {
        NN_LOG("--------------------------------\n");
        NN_LOG("\nServerCode : %s\n", info.GetServerCode());
    }
    NN_LOG("--------------------------------\n");
    NN_LOG("AttributeFlatSet\n");
    nn::err::detail::AttributeFlagSet flatSet{ {info.GetAttributeFlagSetValue()} };
    NN_LOG("    UnacceptableApplicationVersion : %s\n",
        flatSet.Test<nn::err::detail::AttributeFlag::UnacceptableApplicationVersion>() ? "On" : "Off");

    NN_LOG("================================\n");
}

void PrintTimeOfOccurrence(const void* param) NN_NOEXCEPT
{
    if( nn::err::IsErrorRecordDisplayRequested(param) )
    {
        nn::time::CalendarTime calendarTime;
        nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, nn::err::GetTimeOfOccurrence(param)));
        NN_LOG("Recorded at (%s): %04d/%02d/%02d %02d:%02d:%02d\n",
            calendarAdditionalInfo.timeZone.standardTimeName,
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);
    }
    else
    {
        NN_LOG("TimeOfOccurrence is not available.\n");
    }
}

}} // nnt::err
