﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "testEns_TaskMultiple.h"

namespace nnt { namespace ens {

// リンクエラー対策
const int TaskMultiple::SubTaskCount;

TaskMultiple::TaskMultiple() NN_NOEXCEPT
    : m_pOutCounter(0)
    , m_Span(0)
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_TaskSimples); i++)
    {
        m_TaskSimples[i].RegisterAsyncContext(&m_Contexts[i]);
    }
}

TaskMultiple::~TaskMultiple() NN_NOEXCEPT
{
}

void TaskMultiple::UnregisterAsyncContext() NN_NOEXCEPT
{
    // UnregisterAsyncContext() を呼ぶと非同期コンテキストのイベントがシグナルされない状態でタスクの実行が完了してしまう場合がある。
    // その場合、結果取得時に ASSERT に引っ掛かる。
    for (int i = 0; i < NN_ARRAY_SIZE(m_TaskSimples); i++)
    {
        m_TaskSimples[i].Cancel();
    }

    Task::UnregisterAsyncContext();
}

void TaskMultiple::Cancel() NN_NOEXCEPT
{
    for (int i = 0; i < NN_ARRAY_SIZE(m_TaskSimples); i++)
    {
        m_TaskSimples[i].Cancel();
    }

    Task::Cancel();
}

void TaskMultiple::SetParameter(int* pOutCounter, nn::TimeSpan span) NN_NOEXCEPT
{
    m_pOutCounter = pOutCounter;
    m_Span = span;
}

nn::Result TaskMultiple::Main() NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), nn::ens::ResultInvalidState());

        *m_pOutCounter = 0;
    }

    for (int i = 0; i < NN_ARRAY_SIZE(m_TaskSimples); i++)
    {
        NN_RESULT_DO(StepDoSimple(i));
    }

    NN_RESULT_SUCCESS;
}

nn::Result TaskMultiple::StepDoSimple(int index) NN_NOEXCEPT
{
    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), nn::ens::ResultInvalidState());

        m_TaskSimples[index].SetParameter(m_Span);
    }

    m_TaskSimples[index].SetWorkBuffer(GetWorkBuffer(), GetWorkBufferSize());

    m_TaskSimples[index].Run();

    SetErrorDetail(m_Contexts[index].GetErrorDetail());
    NN_RESULT_DO(m_Contexts[index].GetResult());

    {
        NN_UTIL_LOCK_GUARD(GetParameterLock());

        NN_RESULT_THROW_UNLESS(IsAsyncContextRegistered(), nn::ens::ResultInvalidState());

        *m_pOutCounter = index + 1;
    }

    NN_RESULT_SUCCESS;
}

}}
