﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include "testEns_TaskSimple.h"
#include "testEns_TaskMultiple.h"

namespace
{
    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void CreateSimpleTask(nn::ens::AsyncContext* pOutContext, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskSimple);

        pTask->SetParameter(span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }

    void CreateMultipleTask(nn::ens::AsyncContext* pOutContext, int* pOutCounter, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskMultiple);

        pTask->SetParameter(pOutCounter, span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }
}

class EnsTask : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsTask, SimpleUnregisterAging)
{
    for (int i = 0; i < 10000; i++)
    {
        nn::ens::AsyncContext context;

        CreateSimpleTask(&context, nn::TimeSpan::FromSeconds(1));

        context.GetEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(i % 100));
    }
}

TEST_F(EnsTask, SimpleCancelAging)
{
    for (int i = 0; i < 10000; i++)
    {
        nn::ens::AsyncContext context;

        CreateSimpleTask(&context, nn::TimeSpan::FromSeconds(1));

        context.GetEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(i % 100));
        context.Cancel();

        EXPECT_TRUE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(50)));

        context.GetEvent().Wait();
    }
}

TEST_F(EnsTask, Multiple)
{
    nn::ens::AsyncContext context;
    int counter = 0;

    CreateMultipleTask(&context, &counter, nn::TimeSpan::FromMilliSeconds(1));

    context.GetEvent().Wait();

    NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

    EXPECT_EQ(counter, nnt::ens::TaskMultiple::SubTaskCount);
}

TEST_F(EnsTask, MultipleUnregisterAging)
{
    for (int i = 0; i < 10000; i++)
    {
        nn::ens::AsyncContext context;
        int counter = 0;

        CreateMultipleTask(&context, &counter, nn::TimeSpan::FromMicroSeconds(30));

        context.GetEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(i % 100));
    }

    int counters[nnt::ens::TaskMultiple::SubTaskCount] = {};

    for (int i = 0; i < nnt::ens::TaskMultiple::SubTaskCount; i++)
    {
        nn::ens::AsyncContext context;

        CreateMultipleTask(&context, &counters[i], nn::TimeSpan::FromMilliSeconds(300));

        // サブタスクが 1 つ完了した時点で非同期コンテキストのイベントがシグナルされないことを確認する。
        EXPECT_FALSE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(i * 300 + 50)));

        // サブタスクが 1 つ完了する度に値は更新されている。
        EXPECT_EQ(counters[i], i);
    }

    // 「最後に登録されたタスクが完了する＝タスクマネージャに登録されたタスクはすべて実行された」となる。
    {
        nn::ens::AsyncContext context;

        CreateSimpleTask(&context, nn::TimeSpan::FromMilliSeconds(1));

        context.GetEvent().Wait();
    }

    for (int i = 0; i < nnt::ens::TaskMultiple::SubTaskCount; i++)
    {
        // コンテキストを破棄した後に値が更新されていないかを確認する。
        EXPECT_EQ(counters[i], i);
    }
}

TEST_F(EnsTask, MultipleCancelAging)
{
    for (int i = 0; i < 10000; i++)
    {
        nn::ens::AsyncContext context;
        int counter = 0;

        CreateMultipleTask(&context, &counter, nn::TimeSpan::FromMicroSeconds(30));

        context.GetEvent().TimedWait(nn::TimeSpan::FromMicroSeconds(i % 100));
        context.Cancel();

        EXPECT_TRUE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(50)));

        context.GetEvent().Wait();
    }

    for (int i = 0; i < nnt::ens::TaskMultiple::SubTaskCount; i++)
    {
        nn::ens::AsyncContext context;
        int counter = 0;

        CreateMultipleTask(&context, &counter, nn::TimeSpan::FromMilliSeconds(300));

        EXPECT_FALSE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(i * 300 + 50)));
        context.Cancel();

        EXPECT_TRUE(context.GetEvent().TimedWait(nn::TimeSpan::FromMilliSeconds(50)));

        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_FAILURE(nn::ens::ResultCanceledByUser, context.GetResult());

        // 処理結果に関わらず実行したタスク数は記録される。
        EXPECT_EQ(counter, i);
    }
}
