﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

#include "testEns_TaskSimple.h"
#include "testEns_TaskThreadStatistics.h"

namespace
{
    const int ParallelCount = 4;
}

namespace
{
    nn::ens::ServiceThreadContext g_ServiceThreadContexts[ParallelCount];

    nn::os::TlsSlot g_TlsSlot;

    nn::os::ThreadType g_Threads[ParallelCount];
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStacks[ParallelCount][64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_RequiredMemory[nn::ens::RequiredMemorySizeMin];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemorys[ParallelCount][4 * 1024 * 1024];
}

namespace
{
    void WorkerThread(void* pParam) NN_NOEXCEPT
    {
        nn::ens::ServiceThreadContext* pContext = reinterpret_cast<nn::ens::ServiceThreadContext*>(pParam);

        nn::ens::detail::core::StartServiceLoop(*pContext);
    }

    void WorkerThreadForStatistics(void* pParam) NN_NOEXCEPT
    {
        nn::ens::ServiceThreadContext* pContext = reinterpret_cast<nn::ens::ServiceThreadContext*>(pParam);

        nn::ens::detail::core::StartServiceLoop(*pContext);

        int count = static_cast<int>(nn::os::GetTlsValue(g_TlsSlot));

        // 1 スレッドにつき約 250 回呼ばれるはず。
        NN_LOG("Name = %s, ExecutionCount = %d\n", nn::os::GetThreadNamePointer(nn::os::GetCurrentThread()), count);

        NN_ABORT_UNLESS_GREATER(count, 0);
    }

    void CreateAndStartThreads(nn::os::ThreadFunction pFunction) NN_NOEXCEPT
    {
        for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
        {
            g_ServiceThreadContexts[i].SetBuffer(g_ServiceWorkMemorys[i], sizeof (g_ServiceWorkMemorys[i]));

            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Threads[i], pFunction, &g_ServiceThreadContexts[i],
                g_ThreadStacks[i], sizeof (g_ThreadStacks[i]), nn::os::DefaultThreadPriority + 1));

            char name[nn::os::ThreadNameLengthMax] = {};
            nn::util::SNPrintf(name, sizeof (name), "Thread_%d", i);

            nn::os::SetThreadName(&g_Threads[i], name);

            nn::os::StartThread(&g_Threads[i]);
        }
    }

    void StopAndDestroyThreads() NN_NOEXCEPT
    {
        for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
        {
            nn::ens::detail::core::StopServiceLoop(g_ServiceThreadContexts[i]);
        }

        for (int i = 0; i < NN_ARRAY_SIZE(g_Threads); i++)
        {
            nn::os::DestroyThread(&g_Threads[i]);
        }
    }

    void CreateSimpleTask(nn::ens::AsyncContext* pOutContext, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskSimple);

        pTask->SetParameter(span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }

    void CreateThreadStatisticsTask(nn::ens::AsyncContext* pOutContext, nn::TimeSpan span) NN_NOEXCEPT
    {
        NN_DETAIL_ENS_CREATE_TASK(pOutContext, pTask, nnt::ens::TaskThreadStatistics);

        pTask->SetParameter(&g_TlsSlot, span);

        nn::ens::detail::core::RegisterTask(pTask, pOutContext);
    }
}

class EnsParallelExecution : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::AllocateTlsSlot(&g_TlsSlot, nullptr));

        nn::ens::detail::core::SetServerKey("");
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        FreeTlsSlot(g_TlsSlot);
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        nn::ens::detail::core::CreateAllocator(g_RequiredMemory, sizeof (g_RequiredMemory));
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::ens::detail::core::UnregisterTaskAll();

        nn::ens::detail::core::DestroyAllocator();
    }
};

TEST_F(EnsParallelExecution, Cancel1)
{
    CreateAndStartThreads(WorkerThread);

    NN_UTIL_SCOPE_EXIT
    {
        StopAndDestroyThreads();
    };

    int patterns[][4] =
    {
        {0, 1, 2, 3}, {0, 1, 3, 2}, {0, 2, 1, 3}, {0, 2, 3, 1}, {0, 3, 1, 2}, {0, 3, 2, 1},
        {1, 0, 2, 3}, {1, 0, 3, 2}, {1, 2, 0, 3}, {1, 2, 3, 0}, {1, 3, 0, 2}, {1, 3, 2, 0},
        {2, 0, 1, 3}, {2, 0, 3, 1}, {2, 1, 0, 3}, {2, 1, 3, 0}, {2, 3, 0, 1}, {2, 3, 1, 0},
        {3, 0, 1, 2}, {3, 0, 2, 1}, {3, 1, 0, 2}, {3, 1, 2, 0}, {3, 2, 0, 1}, {3, 2, 1, 0},
    };

    for (int n = 0; n < NN_ARRAY_SIZE(patterns); n++)
    {
        nn::ens::AsyncContext contexts[NN_ARRAY_SIZE(patterns[n])];

        for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
        {
            CreateSimpleTask(&contexts[i], nn::TimeSpan::FromSeconds(10));
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

        for (int i = 0; i < NN_ARRAY_SIZE(patterns[n]); i++)
        {
            contexts[patterns[n][i]].Cancel();
        }

        for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
        {
            contexts[i].GetEvent().Wait();
        }
    }
}

TEST_F(EnsParallelExecution, Cancel2)
{
    CreateAndStartThreads(WorkerThread);

    NN_UTIL_SCOPE_EXIT
    {
        StopAndDestroyThreads();
    };

    nn::ens::AsyncContext contexts[10];

    for (int i = 0; i < NN_ARRAY_SIZE(contexts); i++)
    {
        CreateSimpleTask(&contexts[i], nn::TimeSpan::FromSeconds(10));
    }

    nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(50));

    // タスクマネージャ内
    contexts[4].Cancel();
    contexts[7].Cancel();
    contexts[9].Cancel();
    // 実行中
    contexts[0].Cancel();
    contexts[2].Cancel();

    // その他は StopServiceLoop でキャンセルする。

    contexts[0].GetEvent().Wait();
    contexts[2].GetEvent().Wait();
    contexts[4].GetEvent().Wait();
    contexts[7].GetEvent().Wait();
    contexts[9].GetEvent().Wait();
}

TEST_F(EnsParallelExecution, ThreadStatistics)
{
    CreateAndStartThreads(WorkerThreadForStatistics);

    NN_UTIL_SCOPE_EXIT
    {
        StopAndDestroyThreads();
    };

    static nn::ens::AsyncContext s_Contexts[1000];

    for (int i = 0; i < NN_ARRAY_SIZE(s_Contexts); i++)
    {
        CreateThreadStatisticsTask(&s_Contexts[i], nn::TimeSpan::FromMilliSeconds(std::rand() % 30));
    }

    for (int i = 0; i < NN_ARRAY_SIZE(s_Contexts); i++)
    {
        s_Contexts[i].GetEvent().Wait();
    }
}
