﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../Common/testEns_Common.h"

TEST(EnsTypes, UserId)
{
    const nn::ens::UserId invalidUserId = nn::ens::InvalidUserId;
    EXPECT_EQ(invalidUserId, nn::ens::InvalidUserId);

    const nn::ens::UserId userId1 = {1ull};
    const nn::ens::UserId userId2 = {1ull};
    const nn::ens::UserId userId3 = {2ull};
    EXPECT_EQ(userId1, userId2);
    EXPECT_NE(userId1, userId3);
    EXPECT_NE(userId1, nn::ens::InvalidUserId);
}

TEST(EnsTypes, Credential)
{
    const nn::ens::Credential credential = nn::ens::InvalidCredential;
    EXPECT_FALSE(credential.IsValid());

    const nn::ens::Credential credential1 = {{1ull}, "fedcba9876543210"};
    const nn::ens::Credential credential2 = {{2ull}, "fedcba9876543210"};
    EXPECT_TRUE(credential1.IsValid());
    EXPECT_TRUE(credential2.IsValid());
}

TEST(EnsTypes, Digest)
{
    nn::ens::Digest digest = {};
    char buffer[nn::ens::Digest::StringLength + 1] = {};

    EXPECT_TRUE(nn::ens::Digest::FromString(&digest, "0000000000000000000000000000000000000000"));
    EXPECT_STREQ(digest.ToString(buffer, sizeof (buffer)), "0000000000000000000000000000000000000000");

    // pValue[40] は NULL 終端でなくてもよい。
    EXPECT_TRUE(nn::ens::Digest::FromString(&digest, "00112233445566778899001122334455667788990"));
    EXPECT_STREQ(digest.ToString(buffer, sizeof (buffer)), "0011223344556677889900112233445566778899");
    EXPECT_TRUE(nn::ens::Digest::FromString(&digest, "0011223344556677889900112233445566778899g"));
    EXPECT_STREQ(digest.ToString(buffer, sizeof (buffer)), "0011223344556677889900112233445566778899");

    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, ""));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "0"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "00"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "00112233445566778899001122334455667788"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "001122334455667788990011223344556677889"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "@"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "001122334455667788990011223344556677889@"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "G"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "001122334455667788990011223344556677889G"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "`"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "001122334455667788990011223344556677889`"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "g"));
    EXPECT_FALSE(nn::ens::Digest::FromString(&digest, "001122334455667788990011223344556677889g"));

    nn::ens::Digest expectDigest =
    {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99,
        0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, 0x00, 0x11, 0x22, 0x33,
    };

    EXPECT_TRUE(nn::ens::Digest::FromString(&digest, "00112233445566778899aabbccddeeff00112233"));
    EXPECT_EQ(digest, expectDigest);
    EXPECT_STREQ(digest.ToString(buffer, sizeof (buffer)), "00112233445566778899aabbccddeeff00112233");

    EXPECT_TRUE(nn::ens::Digest::FromString(&digest, "00112233445566778899AABBCCDDEEFF00112233"));
    EXPECT_EQ(digest, expectDigest);
    // ToString は小文字に変換する。
    EXPECT_STREQ(digest.ToString(buffer, sizeof (buffer)), "00112233445566778899aabbccddeeff00112233");
}
