﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../Common/testEns_Common.h"
#include "../../../Common/testEns_AccountUtility.h"
#include "../../../Common/testEns_NetworkUtility.h"
#include "../AcbaaCommon/testEns_ApiForAcbaa.h"

#include <nn/ens/ens_ApiForAcbaa.h>
#include <nn/ens/ens_SearchQueryBuilder.h>

namespace
{
    const size_t MetadataBufferSize = 4096;
}

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    nn::os::ThreadType g_Thread;
    NN_OS_ALIGNAS_THREAD_STACK nn::Bit8 g_ThreadStack[64 * 1024];

    NN_ALIGNAS(4096) nn::Bit8 g_ServiceWorkMemory[nn::ens::RequiredMemorySizeMin + 4 * 1024 * 1024];

    nn::ens::Credential g_Credentials[2] = {nn::ens::InvalidCredential, nn::ens::InvalidCredential};
    bool g_IsGenerated = false;

    nn::ens::MyDesignHeader g_Headers[2] = {};
    char g_MetadataBuffers[NN_ARRAY_SIZE(g_Headers)][MetadataBufferSize];
    bool g_IsListed = false;

    char g_MetadataFixtureBuffer[MetadataBufferSize];
    nn::ens::ReceiveBuffer g_MetadataFixture = {g_MetadataFixtureBuffer, sizeof (g_MetadataFixtureBuffer)};

    nn::ens::MyDesignAuthorId g_MyDesignAuthorId = nn::ens::InvalidMyDesignAuthorId;
}

namespace
{
    void WorkerThread(void*) NN_NOEXCEPT
    {
        nn::ens::StartServiceLoop("acbaa", g_ServiceWorkMemory, sizeof (g_ServiceWorkMemory));
    }

    void TestGetMyDesignHeaderList(int expectedCount, int expectedTotalCount,
        nn::ens::MyDesignHeader pOutHeaderList[], int count, int offset,
        const nn::ens::MyDesignAuthorId& authorId,
        const nn::ens::Credential& credential, const nnt::ens::NsaIdTokenGetter& token) NN_NOEXCEPT
    {
        nn::ens::AsyncContext context;
        int actualCount;
        int actualTotalCount;

        nn::ens::GetMyDesignHeaderList(&context, &actualCount, &actualTotalCount,
            pOutHeaderList, count, offset, authorId, credential, token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

        EXPECT_EQ(expectedCount, actualCount);
        EXPECT_EQ(expectedTotalCount, actualTotalCount);
    }

    void DumpMyDesignHeader(const nn::ens::MyDesignHeader& header) NN_NOEXCEPT
    {
        auto c = nn::time::ToCalendarTimeInUtc(header.updatedAt);

        char digest[nn::ens::Digest::StringLength + 1] = {};

        NN_LOG("    MyDesignId: %llu\n", header.id.value);
        NN_LOG("    AuthorName: %s\n", header.authorName.value);
        NN_LOG("    UpdatedAt:  (UTC) %04d/%02d/%02d %02d:%02d:%02d\n", c.year, c.month, c.day, c.hour, c.minute, c.second);
        NN_LOG("    Digest:     %s\n", header.digest.ToString(digest, sizeof (digest)));
        NN_LOG("    MetaSize:   %zu\n", header.metadata.receivedSize);
    }

    void PostMyDesignWithName(const char* pDesignName,
        const nn::ens::Credential& credential, const nnt::ens::NsaIdTokenGetter& token) NN_NOEXCEPT
    {
        char metadataFixtureBuffer[MetadataBufferSize];
        nn::ens::ReceiveBuffer metadataFixture = {metadataFixtureBuffer, sizeof (metadataFixtureBuffer)};

        {
            nn::ens::AsyncContext context;

            nnt::ens::GetMyDesignMetadataFixture(&context, &metadataFixture, pDesignName);

            context.GetEvent().Wait();

            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }
        {
            nn::ens::SendBuffer metadata = {metadataFixture.pBuffer, metadataFixture.receivedSize};
            nn::ens::SendBuffer body = {"body", 4};

            nn::ens::AsyncContext context;
            nn::ens::MyDesignId id = {};

            nn::ens::PostMyDesign(&context, &id, metadata, body, credential, token.Get());

            context.GetEvent().Wait();

            NNT_EXPECT_RESULT_SUCCESS(context.GetResult());
        }
    }

    void SearchMyDesign(const char* pDesignName,
        const nn::ens::Credential& credential, const nnt::ens::NsaIdTokenGetter& token) NN_NOEXCEPT
    {
        char searchQuery[nn::ens::SearchQueryLengthMax + 1] = {};
        nn::ens::SearchQueryBuilder builder(searchQuery, sizeof (searchQuery));

        ASSERT_TRUE(builder.Set("name", pDesignName));

        nn::ens::MyDesignHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        nn::ens::AsyncContext context;
        int count;
        int total;

        nn::ens::GetMyDesignHeaderList(&context, &count, &total,
            &header, 1, 0, builder.GetString(), credential, token.Get());

        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("Search(%s) => %d\n", pDesignName, total);
    }
}

class EnsMyDesign : public testing::Test
{
protected:
    static void SetUpTestCase() NN_NOEXCEPT
    {
        nn::account::Initialize();
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfig));

        nnt::ens::ReserveSockets();
        nnt::ens::InitializeLibcurl();
        nnt::ens::EnableCommunicationLogDump();

        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            g_Headers[i].metadata.pBuffer = g_MetadataBuffers[i];
            g_Headers[i].metadata.bufferSize = sizeof (g_MetadataBuffers[i]);
        }
    }

    static void TearDownTestCase() NN_NOEXCEPT
    {
        nnt::ens::FinalizeLibcurl();
        nnt::ens::CancelSocketsReservation();

        nn::socket::Finalize();
    }

    virtual void SetUp() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&g_Thread, WorkerThread, nullptr,
            g_ThreadStack, sizeof (g_ThreadStack), nn::os::DefaultThreadPriority + 1));

        nn::os::StartThread(&g_Thread);

        nn::nifm::SubmitNetworkRequestAndWait();
    }

    virtual void TearDown() NN_NOEXCEPT
    {
        nn::nifm::CancelNetworkRequest();

        nn::ens::StopServiceLoop();

        nn::os::DestroyThread(&g_Thread);
    }
};

TEST_F(EnsMyDesign, GenerateCredential)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    bool isGenerated = true;

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        nn::ens::AsyncContext context;

        nn::ens::GenerateCredential(&context, &g_Credentials[i], token.Get());

        context.GetEvent().Wait();

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

        // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
        if (context.GetResult().IsFailure() &&
            !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
        {
            NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
        }

#endif

        if (context.GetResult().IsSuccess())
        {
            NN_LOG("UserId[%d] = %llu\n", i, g_Credentials[i].userId.value);
        }
        else
        {
            isGenerated = false;
        }
    }

    g_IsGenerated = isGenerated;
}

TEST_F(EnsMyDesign, GetMetadataFixture)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    nn::ens::AsyncContext context;

    nnt::ens::GetMyDesignMetadataFixture(&context, &g_MetadataFixture, nullptr);

    context.GetEvent().Wait();

    if (context.GetResult().IsFailure())
    {
        g_MetadataFixture.receivedSize = 0;
    }

#if defined(NN_BUILD_CONFIG_OS_HORIZON)

    NNT_ASSERT_RESULT_SUCCESS(context.GetResult());

#else

    // Generic ではローカルサーバ起動が必要なので、接続タイムアウトのみ除外する。
    if (context.GetResult().IsFailure() &&
        !nn::ens::ResultHttpErrorOperationTimedout::Includes(context.GetResult()))
    {
        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }

#endif
}

TEST_F(EnsMyDesign, PostBeforeRegisterProfile)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
    nn::ens::SendBuffer body = {"body", 4};

    nn::ens::AsyncContext context;
    nn::ens::MyDesignId id = {};

    nn::ens::PostMyDesign(&context, &id, metadata, body, g_Credentials[0], token.Get());

    context.GetEvent().Wait();

    NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultApplicationSpecificError, context.GetResult());
    EXPECT_EQ(context.GetErrorDetail(), 1001);
}

TEST_F(EnsMyDesign, RegisterProfile)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        nn::ens::MyDesignAuthorId id1 = {};
        nn::ens::MyDesignAuthorId id2 = {};

        {
            nn::ens::AsyncContext context;

            nn::ens::MyDesignAuthorName name = {};
            nn::util::SNPrintf(name.value, sizeof (name.value), "name_%llu", g_Credentials[i].userId.value);

            nn::ens::RegisterMyDesignAuthorProfile(&context, &id1, name, g_Credentials[i], token.Get());

            context.GetEvent().Wait();

            NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

            NN_LOG("MyDesignAuthorId = %llu\n", id1.value);
        }
        // 2 回目（API は成功するが、名前は更新されない）
        {
            nn::ens::AsyncContext context;

            nn::ens::MyDesignAuthorName name = {};
            nn::util::SNPrintf(name.value, sizeof (name.value), "name");

            nn::ens::RegisterMyDesignAuthorProfile(&context, &id2, name, g_Credentials[i], token.Get());

            context.GetEvent().Wait();

            NNT_EXPECT_RESULT_SUCCESS(context.GetResult());
        }

        ASSERT_EQ(id1, id2);

        if (i == 0)
        {
            g_MyDesignAuthorId = id1;
        }
    }
}

TEST_F(EnsMyDesign, PostAfterRegisterProfile)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 送信 1
    {
        nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
        nn::ens::SendBuffer body = {"body 1", 6};

        nn::ens::AsyncContext context;
        nn::ens::MyDesignId id = {};

        nn::ens::PostMyDesign(&context, &id, metadata, body, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("MyDesignId1 = %llu\n", id.value);
    }

    // 次に投稿するマイデザインの更新時刻をずらすため、スリープを挟む。
    nn::os::SleepThread(nn::TimeSpan::FromSeconds(2));

    // 送信 2
    {
        nn::ens::SendBuffer metadata = {g_MetadataFixture.pBuffer, g_MetadataFixture.receivedSize};
        nn::ens::SendBuffer body = {"body 2", 6};

        nn::ens::AsyncContext context;
        nn::ens::MyDesignId id = {};

        nn::ens::PostMyDesign(&context, &id, metadata, body, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

        NN_LOG("MyDesignId2 = %llu\n", id.value);
    }
}

TEST_F(EnsMyDesign, GetList)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }
    if (g_MyDesignAuthorId == nn::ens::InvalidMyDesignAuthorId)
    {
        NN_LOG("My design author profile was not registered. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    for (int i = 0; i < NN_ARRAY_SIZE(g_Credentials); i++)
    {
        ASSERT_NO_FATAL_FAILURE(TestGetMyDesignHeaderList(NN_ARRAY_SIZE(g_Headers), NN_ARRAY_SIZE(g_Headers),
            g_Headers, NN_ARRAY_SIZE(g_Headers), 0, g_MyDesignAuthorId, g_Credentials[0], token));

        for (int j = 0; j < NN_ARRAY_SIZE(g_Headers); j++)
        {
            NN_LOG("MyDesignHeader[%d]\n", j);

            DumpMyDesignHeader(g_Headers[j]);
        }

        g_IsListed = true;
    }

    // 更新時刻の降順に取得される。
    EXPECT_GT(g_Headers[0].updatedAt, g_Headers[1].updatedAt);

    nn::ens::MyDesignAuthorName name = {};
    nn::util::SNPrintf(name.value, sizeof (name.value), "name_%llu", g_Credentials[0].userId.value);

    // 送信 2
    EXPECT_GT(g_Headers[0].updatedAt.value, 0);
    EXPECT_STREQ(g_Headers[0].authorName.value, name.value);
    EXPECT_EQ(g_Headers[0].metadata.receivedSize, g_MetadataFixture.receivedSize);
    EXPECT_EQ(std::memcmp(g_Headers[0].metadata.pBuffer, g_MetadataFixture.pBuffer, g_Headers[0].metadata.receivedSize), 0);

    // 送信 1
    EXPECT_GT(g_Headers[1].updatedAt.value, 0);
    EXPECT_STREQ(g_Headers[1].authorName.value, name.value);
    EXPECT_EQ(g_Headers[1].metadata.receivedSize, g_MetadataFixture.receivedSize);
    EXPECT_EQ(std::memcmp(g_Headers[1].metadata.pBuffer, g_MetadataFixture.pBuffer, g_Headers[1].metadata.receivedSize), 0);

    // ヘッダリスト受信（オフセットあり）
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 2;

        nn::ens::MyDesignHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestGetMyDesignHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 1, g_MyDesignAuthorId, g_Credentials[0], token));

        // g_Headers[1] と同じはず
        EXPECT_EQ(header.id, g_Headers[1].id);
        EXPECT_STREQ(header.authorName.value, g_Headers[1].authorName.value);
        EXPECT_EQ(header.updatedAt, g_Headers[1].updatedAt);
        EXPECT_EQ(header.metadata.receivedSize, g_Headers[1].metadata.receivedSize);
        EXPECT_EQ(std::memcmp(header.metadata.pBuffer, g_Headers[1].metadata.pBuffer, header.metadata.receivedSize), 0);
    }
}

TEST_F(EnsMyDesign, GetListWhole)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }
    if (g_MyDesignAuthorId == nn::ens::InvalidMyDesignAuthorId)
    {
        NN_LOG("My design author profile was not registered. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    static nn::ens::MyDesignHeader s_Headers[100] = {};
    static char s_MetadataBuffers[NN_ARRAY_SIZE(s_Headers)][MetadataBufferSize];

    for (int i = 0; i < NN_ARRAY_SIZE(s_Headers); i++)
    {
        s_Headers[i].metadata.pBuffer = s_MetadataBuffers[i];
        s_Headers[i].metadata.bufferSize = sizeof (s_MetadataBuffers[i]);
    }

    {
        nn::ens::AsyncContext context;
        int count;
        int total;

        nn::ens::GetMyDesignHeaderList(&context, &count, &total,
            s_Headers, NN_ARRAY_SIZE(s_Headers), 0, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

        for (int i = 1; i < count; i++)
        {
            // 更新時刻の降順に取得される。
            EXPECT_GE(s_Headers[i - 1].updatedAt, s_Headers[i].updatedAt);
        }
    }
}

TEST_F(EnsMyDesign, GetBody)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsListed)
    {
        NN_LOG("Headers were not listed. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 自分と他人、両方からアクセスできるはず。
    for (int c = 0; c < NN_ARRAY_SIZE(g_Credentials); c++)
    {
        for (int i = 0; i < NN_ARRAY_SIZE(g_Headers); i++)
        {
            nn::ens::ReceiveBuffer body;
            char buffer[512];

            body.pBuffer = buffer;
            body.bufferSize = sizeof (buffer);
            body.receivedSize = 0;

            nn::ens::AsyncContext context;

            nn::ens::GetMyDesignBody(&context, &body, g_Headers[i].id, g_Credentials[c], token.Get());

            context.GetEvent().Wait();

            NNT_EXPECT_RESULT_SUCCESS(context.GetResult());

            if (i == 0)
            {
                EXPECT_EQ(body.receivedSize, 6);
                EXPECT_EQ(std::memcmp(body.pBuffer, "body 2", 6), 0);
            }
            else
            {
                EXPECT_EQ(body.receivedSize, 6);
                EXPECT_EQ(std::memcmp(body.pBuffer, "body 1", 6), 0);
            }
        }
    }
}

TEST_F(EnsMyDesign, Delete)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsListed)
    {
        NN_LOG("Headers were not listed. This test is skipped.\n");
        return;
    }
    if (g_MyDesignAuthorId == nn::ens::InvalidMyDesignAuthorId)
    {
        NN_LOG("My design author profile was not registered. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    // 他人が投稿したマイデザインの削除要求
    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMyDesign(&context, g_Headers[0].id, g_Credentials[1], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_FAILURE(nn::ens::ResultNotFound, context.GetResult());
    }
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 2;

        nn::ens::MyDesignHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestGetMyDesignHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 0, g_MyDesignAuthorId, g_Credentials[0], token));
    }

    // 自分が投稿したマイデザインの削除要求
    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMyDesign(&context, g_Headers[0].id, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
    {
        const int ExpectedCount = 1;
        const int ExpectedTotalCount = 1;

        nn::ens::MyDesignHeader header = {};
        char buffer[MetadataBufferSize];

        header.metadata.pBuffer = buffer;
        header.metadata.bufferSize = sizeof (buffer);
        header.metadata.receivedSize = 0;

        ASSERT_NO_FATAL_FAILURE(TestGetMyDesignHeaderList(ExpectedCount, ExpectedTotalCount,
            &header, 1, 0, g_MyDesignAuthorId, g_Credentials[0], token));
    }

    // 再削除
    {
        nn::ens::AsyncContext context;

        nn::ens::DeleteMyDesign(&context, g_Headers[0].id, g_Credentials[0], token.Get());

        context.GetEvent().Wait();

        NNT_ASSERT_RESULT_SUCCESS(context.GetResult());
    }
}

TEST_F(EnsMyDesign, PostSearchSamples)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("=", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("&", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("~", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("!", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("+", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("-", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("e", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName("en", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName(u8"ほげほげ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(PostMyDesignWithName(u8"ほげほげ ぴよぴよ", g_Credentials[0], token));
}

TEST_F(EnsMyDesign, Search)
{
    ASSERT_TRUE(nn::nifm::IsNetworkAvailable());

    if (!g_IsGenerated)
    {
        NN_LOG("Credentials were not generated. This test is skipped.\n");
        return;
    }

    NNT_ENS_ENSURE_NSA_ID_TOKEN(token);

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("for", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("ens", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("for ens", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("Ens", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("ENS", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ｅｎｓ", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("e", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("en", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("ens2", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(" ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("=", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("&", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("~", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("!", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("+", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign("-", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ほげ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ほげほ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ほげほげ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ぴよぴよ", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ほげ ぴよ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ほげほげ ぴよぴよ", g_Credentials[0], token));

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ホゲほげ", g_Credentials[0], token));
    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(u8"ホゲホゲ", g_Credentials[0], token));

    char longName[nn::ens::SearchQueryLengthMax + 1 - (sizeof ("&q%5Bname%5D=") - 1)] = {};

    for (int i = 0; i < sizeof (longName) - 1; i++)
    {
        longName[i] = 'A';
    }

    ASSERT_NO_FATAL_FAILURE(SearchMyDesign(longName, g_Credentials[0], token));
}
