﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "LcdConfig.h"
#include <cstdio>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/lbl/lbl.h>

namespace nns {

namespace {

const Color DefaultBackGroundColor = Color(0.0f, 0.0f, 0.5f, 1.0f);

}

NN_IMPLICIT LcdConfig::LcdConfig(bool isAutoBrightnessControl, bool isVrrEnabled,
                                 float brightness, int vfp, int performanceInterval) NN_NOEXCEPT
:
m_Video(android::ISurfaceComposer::eDisplayIdMain),
m_IsSceneSwitched(false),
m_IsSwitchingSceneAscending(true)
{
    m_Setting.clear();
    m_Setting.reserve(1024);
    std::vector<SettingContext::State> settingState;

    settingState.clear();
    // 入力された引数の小数点以下を反映させる。
    float floatMod = CalculateFloatMod(brightness);
    for (int i1=0; i1<=100; ++i1)
    {
        settingState.push_back({nullptr, 0.01f * i1 + floatMod});
    }
    int initialIndex = FindSettingValueIndexFromLevel(settingState, brightness);
    m_Setting.push_back({&LcdConfig::SwitchCurrentBrightness, &LcdConfig::GetAppliedBrightness, "Brightness", std::move(settingState), initialIndex});

    settingState.clear();
    settingState.push_back({"OFF", 0.0f});
    settingState.push_back({"ON", 1.0f});
    initialIndex = FindSettingStateIndexFromLevel(settingState, static_cast<int>(isAutoBrightnessControl));
    m_Setting.push_back({&LcdConfig::SwitchAutoBrightnessControl, nullptr, "Auto Control", std::move(settingState), initialIndex});

    settingState.clear();
    // No setting では VFP を指定しない。
    settingState.push_back({"No setting", -1.0f});
    int integerMod = (0 <= vfp) ? vfp % 10 : 0;
    for (int i1=10; i1<=300; i1+=10)
    {
        settingState.push_back({nullptr, static_cast<float>(i1 + integerMod)});
    }
    initialIndex = FindSettingValueIndexFromLevel(settingState, static_cast<float>(vfp));
    if (0 <= vfp && 0 == initialIndex)
    {
        // 選択できる範囲外の値が入力されたら、Specific VFP として扱う。
        initialIndex = settingState.size();
        settingState.push_back({"Specific VFP", static_cast<float>(vfp)});
    }
    m_Setting.push_back({&LcdConfig::SwitchVfp, nullptr, "VFP", std::move(settingState), initialIndex});

    settingState.clear();
    settingState.push_back({"OFF", 0.0f});
    settingState.push_back({"ON", 1.0f});
    initialIndex = FindSettingStateIndexFromLevel(settingState, static_cast<int>(isVrrEnabled));
    m_Setting.push_back({&LcdConfig::SwitchVrr, nullptr, "VRR", std::move(settingState), initialIndex});

    settingState.clear();
    settingState.push_back({":", 0.0f});
    m_Setting.push_back({nullptr, &LcdConfig::GetLcdPanelVendorId, "Panel ID", std::move(settingState), 0});

    settingState.clear();
    settingState.push_back({"->", 0.0f});
    m_Setting.push_back({&LcdConfig::SwitchSceneAscending, nullptr, "Go to next", std::move(settingState), 0});

    settingState.clear();
    settingState.push_back({"<-", 0.0f});
    m_Setting.push_back({&LcdConfig::SwitchSceneDecending, nullptr, "Go to prev", std::move(settingState), 0});

    // パフォーマンス計測の設定。
    m_Video.TrackPerformance(performanceInterval);
    if (m_Video.GetLcdPanelVendor(m_LcdPanelId))
    {
        NN_LOG("LCD panel vender ID:[0x%X,0x%X,0x%X]\n", m_LcdPanelId[0], m_LcdPanelId[1], m_LcdPanelId[2]);
        NN_LOG("=============================================\n");
    }
}

LcdConfig::~LcdConfig() NN_NOEXCEPT
{
}

void LcdConfig::ApplySelectedLabel() NN_NOEXCEPT
{
    for (auto& setting : m_Setting)
    {
        if (1 < setting.state.size() && nullptr != setting.switchState)
        {
            (this->*(setting.switchState))(setting.state[setting.stateIndex].level);
        }
    }
}

int LcdConfig::GetConfigCount() NN_NOEXCEPT
{
    return static_cast<int>(m_Setting.size());
}

const char* LcdConfig::GetConfigCaption(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    return target.caption;
}

const char* LcdConfig::GetConfigStatus(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    if (nullptr == target.getStatus)
    {
        return "";
    }
    return (this->*(target.getStatus))();
}

const char* LcdConfig::GetConfigLabel(int index) NN_NOEXCEPT
{
    const SettingContext& target = m_Setting[index];
    if (0 < target.state.size())
    {
        const char* label = target.state[target.stateIndex].label;
        if (nullptr != label)
        {
            return label;
        }
        std::snprintf(m_LabelString, LabelStringLength, "%.3f", target.state[target.stateIndex].level);
        return m_LabelString;
    }
    return "None";
}

void LcdConfig::SetConfig(int index, bool isAscending) NN_NOEXCEPT
{
    SettingContext& target = m_Setting[index];
    //NN_LOG("[%d] %s() stateIndex:[%d][%d]\n", __LINE__, __FUNCTION__, target.stateIndex, target.state.size());
    if (0 < target.state.size())
    {
        if (isAscending)
        {
            target.stateIndex = (target.stateIndex + 1) % target.state.size();
        }
        else
        {
            target.stateIndex = (target.stateIndex <= 0) ? target.state.size() - 1 : target.stateIndex - 1;
        }
        if (nullptr != target.switchState)
        {
            (this->*(target.switchState))(target.state[target.stateIndex].level);
        }
    }
}

void LcdConfig::UpdateConfig() NN_NOEXCEPT
{
}

bool LcdConfig::IsSceneSwitched() NN_NOEXCEPT
{
    bool isSceneSwitched = m_IsSceneSwitched;
    m_IsSceneSwitched = false;
    return isSceneSwitched;
}

const Color& LcdConfig::GetBackGroundColor() NN_NOEXCEPT
{
    return DefaultBackGroundColor;
}

bool LcdConfig::IsSwitchingSceneAscending() NN_NOEXCEPT
{
    return m_IsSwitchingSceneAscending;
}

void LcdConfig::SwitchAutoBrightnessControl(float isEnabledValue) NN_NOEXCEPT
{
    bool isEnabled = 0.01f < isEnabledValue;
    if (isEnabled)
    {
        nn::lbl::EnableAutoBrightnessControl();
    }
    else
    {
        nn::lbl::DisableAutoBrightnessControl();
    }
    NN_LOG("Set brightness control setting:[%s]\n", isEnabled ? "auto" : "manual");
    NN_LOG("=============================================\n");
}

void LcdConfig::SwitchCurrentBrightness(float brightness) NN_NOEXCEPT
{
    NN_LOG("Set brightness:[%f]\n", brightness);
    nn::lbl::SetCurrentBrightnessSetting(brightness);
}

void LcdConfig::SwitchVrr(float isEnabledValue) NN_NOEXCEPT
{
    bool isEnabled = 0.01f < isEnabledValue;
    NN_LOG("%s:[%s]\n", m_Video.SetVrrEnabled(isEnabled) ?
        "Set VRR setting" : "Cannot set VRR setting", isEnabled ? "enabled" : "disabled");
    NN_LOG("=============================================\n");
}

void LcdConfig::SwitchVfp(float vfpValue) NN_NOEXCEPT
{
    if (0.0f <= vfpValue)
    {
        int vfp = static_cast<int>(vfpValue);
        NN_LOG("%s:[%d]\n", m_Video.SetVfp(vfp) ? "Set VFP" : "Cannot set VFP", vfp);
        NN_LOG("=============================================\n");
    }
}

void LcdConfig::SwitchSceneAscending(float dummy) NN_NOEXCEPT
{
    NN_UNUSED(dummy);
    m_IsSceneSwitched = true;
    m_IsSwitchingSceneAscending = true;
}

void LcdConfig::SwitchSceneDecending(float dummy) NN_NOEXCEPT
{
    NN_UNUSED(dummy);
    m_IsSceneSwitched = true;
    m_IsSwitchingSceneAscending = false;
}

const char* LcdConfig::GetAppliedBrightness() NN_NOEXCEPT
{
    float brightness = nn::lbl::GetBrightnessSettingAppliedToBacklight();
    std::snprintf(m_LabelString, LabelStringLength, "(%.3f)", brightness);
    return m_LabelString;
}

const char* LcdConfig::GetLcdPanelVendorId() NN_NOEXCEPT
{
    std::snprintf(m_LabelString, LabelStringLength,
        "[0x%02X,0x%02X,0x%02X]", m_LcdPanelId[0], m_LcdPanelId[1], m_LcdPanelId[2]);
    return m_LabelString;
}

int LcdConfig::FindSettingStateIndexFromLevel(std::vector<SettingContext::State> settingState, int level) NN_NOEXCEPT
{
    for (int i1=0; i1<settingState.size(); ++i1)
    {
        if (level == settingState[i1].level)
        {
            return i1;
        }
    }
    // 見つからなかったら便宜上0を返す。
    return 0;
}

int LcdConfig::FindSettingValueIndexFromLevel(std::vector<SettingContext::State> settingState, float level) NN_NOEXCEPT
{
    float epsilon = level * 0.0001f;
    for (int i1=0; i1<settingState.size(); ++i1)
    {
        if (level - epsilon <= settingState[i1].level && settingState[i1].level < level + epsilon)
        {
            return i1;
        }
    }
    // 見つからなかったら便宜上0を返す。
    return 0;
}

float LcdConfig::CalculateFloatMod(float value) NN_NOEXCEPT
{
    auto fatValue = value * 100.0f;
    return (fatValue - static_cast<int>(fatValue)) / 100.0f;
}

}
