﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "BrightnessListener.h"

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/lbl/lbl.h>

#if !defined( NN_BUILD_CONFIG_OS_SUPPORTS_WIN32 )
#include <unistd.h>
#endif

namespace nns {

const int ThreadStackSize = 16 * 1024;
const int SetBrightnessDegrees = 1000;
NN_ALIGNAS(4096) uint8_t s_LogThreadStack[ThreadStackSize];
NN_ALIGNAS(4096) uint8_t s_SetThreadStack[ThreadStackSize];
volatile bool s_IsLogListenerRunning;
volatile bool s_IsSetListenerRunning;

void LogAppliedBrighthnessListener(void* arg)
{
    while (s_IsLogListenerRunning)
    {
        float brightness = nn::lbl::GetBrightnessSettingAppliedToBacklight();
        NN_LOG("Get applied brightness:[%5.3f].\n", brightness);
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(reinterpret_cast<uint64_t>(arg)));
    }
}

void SetBrighthnessListener(void* arg)
{
    uint64_t intervalMSec = reinterpret_cast<uint64_t>(arg);
    uint64_t turningMSec = intervalMSec * 10;
    while (s_IsSetListenerRunning)
    {
        for (int i1=1; i1<=SetBrightnessDegrees && s_IsSetListenerRunning; ++i1)
        {
            nn::lbl::SetCurrentBrightnessSetting(static_cast<float>(i1) / SetBrightnessDegrees);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalMSec));
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(turningMSec));
        for (int i1=SetBrightnessDegrees - 1; i1>=0 && s_IsSetListenerRunning; --i1)
        {
            nn::lbl::SetCurrentBrightnessSetting(static_cast<float>(i1) / SetBrightnessDegrees);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(intervalMSec));
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(turningMSec));
    }
}

NN_IMPLICIT BrightnessListener::BrightnessListener(uint64_t logIntervalMSec, uint64_t setIntervalMSec)
{
    nn::lbl::Initialize();

    // 現在設定された輝度レベルのログを出す。
    if (0 < logIntervalMSec)
    {
        nn::Result result = nn::os::CreateThread(
            &m_LogThread, LogAppliedBrighthnessListener, reinterpret_cast<void*>(logIntervalMSec),
            s_LogThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        s_IsLogListenerRunning = result.IsSuccess();
        if (s_IsLogListenerRunning)
        {
            NN_LOG("Start applied brightness log listener.\n");
            nn::os::StartThread(&m_LogThread);
        }
        else
        {
            NN_LOG("Cannot create listener to listen to applied brightness.\n");
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }

    // 輝度を一定時間で変える。
    if (0 < setIntervalMSec)
    {
        nn::Result result = nn::os::CreateThread(
            &m_SetThread, SetBrighthnessListener, reinterpret_cast<void*>(setIntervalMSec),
            s_SetThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        s_IsSetListenerRunning = result.IsSuccess();
        if (s_IsSetListenerRunning)
        {
            NN_LOG("Start listener to set brightness.\n");
            nn::os::StartThread(&m_SetThread);
        }
        else
        {
            NN_LOG("Cannot create listener to set brightness.\n");
        }
    }
}

BrightnessListener::~BrightnessListener()
{
    if (s_IsLogListenerRunning)
    {
        s_IsLogListenerRunning = false;
        nn::os::WaitThread(&m_LogThread);
        nn::os::DestroyThread(&m_LogThread);
        NN_LOG("Stop applied brightness log listener.\n");
    }
    if (s_IsSetListenerRunning)
    {
        s_IsSetListenerRunning = false;
        nn::os::WaitThread(&m_SetThread);
        nn::os::DestroyThread(&m_SetThread);
        NN_LOG("Stop listener to set brightness.\n");
    }
    nn::lbl::Finalize();
}

}
