﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HdcpManager.h"
#include <cstdio>
#include <cstdlib>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os.h>

#include <nn/settings/fwdbg/settings_SettingsCommon.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>

namespace nns {

const int NvConfigStringLength = 256;

NN_IMPLICIT HdcpManager::HdcpManager() NN_NOEXCEPT
:
m_AuthenticationTimeout(12000)
{
    m_IsNvhdcpAccessible = GetIsStubEmulation();
    if (m_IsNvhdcpAccessible)
    {
        HDCP_RET_ERROR error = hdcp_open(&m_Handle, 1);
        if (HDCP_RET_SUCCESS != error) {
            m_IsNvhdcpAccessible = false;
            SetIsStubEmulation(false);
        }
        nn::os::InitializeEvent(&m_TimeoutEvent, false, nn::os::EventClearMode_AutoClear);
        auto f = [](void* context)
        {
            auto pManager = reinterpret_cast<HdcpManager*>(context);
            while (pManager->m_IsTimeoutThreadRunning)
            {
                if (HDCP_STATE_UNAUTHENTICATED == pManager->m_CurrentNvState && !pManager->m_IsAuthenticationTimeout)
                {
                    if (!nn::os::TimedWaitEvent(&pManager->m_TimeoutEvent,
                        nn::TimeSpan::FromMilliSeconds(pManager->m_AuthenticationTimeout)))
                    {
                        pManager->m_IsAuthenticationTimeout = true;
                        auto pType = reinterpret_cast<StateTransitionType*>(pManager->pStateTransitionType);
                        if (pType->disabledStateHandler)
                        {
                            // 認証タイムアウトを通知。
                            pType->disabledStateHandler(pType->context);
                        }
                    }
                }
                else
                {
                    nn::os::WaitEvent(&pManager->m_TimeoutEvent);
                }
            }
        };
        m_IsAuthenticationTimeout = false;
        static NN_ALIGNAS(4096) char s_Stack[8 * 1024];
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_TimeoutThread, f, this, s_Stack, sizeof(s_Stack), nn::os::DefaultThreadPriority));
        m_IsTimeoutThreadRunning = true;
        nn::os::StartThread(&m_TimeoutThread);
    }
}

HdcpManager::~HdcpManager() NN_NOEXCEPT
{
    if (m_IsNvhdcpAccessible)
    {
        m_IsAuthenticationTimeout = false;
        m_IsTimeoutThreadRunning = false;
        nn::os::SignalEvent(&m_TimeoutEvent);
        nn::os::DestroyThread(&m_TimeoutThread);
        nn::os::ClearEvent(&m_TimeoutEvent);
        nn::os::FinalizeEvent(&m_TimeoutEvent);
        HDCP_RET_ERROR error = hdcp_close(m_Handle);
        NN_ASSERT(HDCP_RET_SUCCESS == error);
    }
}

void HdcpManager::SetHdcpListener(StateTransitionType* pType) NN_NOEXCEPT
{
    if (m_IsNvhdcpAccessible)
    {
        auto f = [](HDCP_CLIENT_HANDLE handle, NvU32 state, void* context)
        {
            if (context)
            {
                HDCP_RET_ERROR status = hdcp_status(handle);
                auto pManager = reinterpret_cast<HdcpManager*>(context);
                pManager->m_CurrentNvState = state;
                pManager->m_IsAuthenticationTimeout = false;
                auto pType = reinterpret_cast<StateTransitionType*>(pManager->pStateTransitionType);
                if (HDCP_RET_AUTH_IN_PROCESS == status)
                {
                    // Signal for authentication timeout.
                    NN_LOG("[%d] %s() HDCP_RET_AUTH_IN_PROCESS state:[%d]\n", __LINE__, __FUNCTION__, state);
                    if (pType->progressingStateHandler)
                    {
                        pType->progressingStateHandler(pType->context);
                    }
                }
                else if (HDCP_RET_SUCCESS == status)
                {
                    NN_LOG("[%d] %s() HDCP_RET_SUCCESS state:[%d]\n", __LINE__, __FUNCTION__, state);
                    if (pType->enabledStateHandler)
                    {
                        pType->enabledStateHandler(pType->context);
                    }
                }
                else if (HDCP_RET_NOT_PLUGGED == status)
                {
                    NN_LOG("[%d] %s() HDCP_RET_NOT_PLUGGED state:[%d]\n", __LINE__, __FUNCTION__, state);
                    if (pType->dockoutStateHandler)
                    {
                        pType->dockoutStateHandler(pType->context);
                    }
                }
                else
                {
                    NN_LOG("[%d] %s() HDCP_RET_FAILED - status:[%d] state:[%d]\n", __LINE__, __FUNCTION__, status, state);
                    if (pType->disabledStateHandler)
                    {
                        pType->disabledStateHandler(pType->context);
                    }
                }
                nn::os::SignalEvent(&pManager->m_TimeoutEvent);
            }
        };
        pStateTransitionType = pType;
        HDCP_RET_ERROR error = hdcp_event_state_transit(m_Handle, f, this);
        NN_ASSERT(HDCP_RET_SUCCESS == error);
        error = hdcp_init_asyncevents(m_Handle);
        NN_ASSERT(HDCP_RET_SUCCESS == error);
    }
}

bool HdcpManager::SwitchHdcp(bool isEnabled) NN_NOEXCEPT
{
    if (m_IsNvhdcpAccessible)
    {
        if (HDCP_RET_SUCCESS == hdcp_enable(m_Handle, isEnabled))
        {
            m_IsEnabled = isEnabled;
            return true;
        }
    }
    return false;
}

int HdcpManager::GetAuthenticationTimeout() NN_NOEXCEPT
{
    return m_AuthenticationTimeout;
}

bool HdcpManager::GetIsStubEmulation() NN_NOEXCEPT
{
    bool isEnabled = false;
    auto typeSize = sizeof(isEnabled);
    nn::settings::fwdbg::GetSettingsItemValue(&isEnabled, typeSize, "hdcp", "stub_emulation");
    return isEnabled;
}

void HdcpManager::SetAuthenticationTimeout(int msec) NN_NOEXCEPT
{
    m_AuthenticationTimeout = msec;
}

void HdcpManager::SetIsStubEmulation(bool isEnabled) NN_NOEXCEPT
{
    auto typeSize = sizeof(isEnabled);
    nn::settings::fwdbg::SetSettingsItemValue("hdcp", "stub_emulation", &isEnabled, typeSize);
}

}
