﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{Ui2dSimple.cpp,PageSampleUi2dSimple}
 *
 * @brief
 *  Ui2d ライブラリの使い方を示したサンプルプログラム
 */

/**
 * @page PageSampleUi2dSimple Ui2dSimple
 * @tableofcontents
 *
 * @image html Applications\Ui2dSimple\ui2dsimple.png
 *
 * @brief
 *  Ui2d ライブラリの使い方を示したサンプルプログラムです。
 *
 * @section PageSampleUi2dSimple_SectionBrief 概要
 *  ここでは、Ui2dSimple の説明をします。
 *
 * @section PageSampleUi2dSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/Ui2dSimple Samples/Sources/Applications/Ui2dSimple @endlink 以下にあります。
 *
 * @section PageSampleUi2dSimple_SectionNecessaryEnvironment 必要な環境
 *  特にありません。
 *
 * @section PageSampleUi2dSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleUi2dSimple_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleUi2dSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。g_SampleMode の値を変更することで、実行するサンプルを切り替えることができます。
 *
 * @section PageSampleUi2dSimple_SectionDetail 解説
 *  初期化処理、メインループ処理、終了処理を行うだけのシンプルな流れになっています。
 *
 * g_SampleMode を切り替えることで、以下の機能を確認できます。
 *
 * - SampleMode_Simple              最も単純なサンプルです。
 * - SampleMode_MultiArcResource    複数のアーカイブリソースやリソースアクセサを扱うサンプルです。
 * - SampleMode_TextureChange       ペインのテクスチャを自ら読み込んだテクスチャに変更します。
 */
#include "Ui2dSimple.h"
#include <cstdlib>

#include <nn/vi.h>

#include <nn/gfx.h>

#include <nn/nn_Assert.h>

#include <nn/ui2d.h>
#include <nn/font.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>
#include <nn/util/util_Vector.h>
#include <nn/util/util_Matrix.h>
#include <nn/mem.h>
#include <nn/os.h>

#if defined(NN_BUILD_CONFIG_OS_WIN)
#include <filesystem>
#define WIN32_LEAN_AND_MEAN
#define NOMINMAX
#include <nn/nn_Windows.h>
#endif

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
#include <nn/init.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
#include <nv/nv_MemoryManagement.h>
#endif
#include <Common/SetupGraphics.h>

namespace nns {

// サンプルの動作モードです。
enum SampleMode
{
    SampleMode_Simple,               // 最も単純なサンプル
    SampleMode_MultiArcResource,     // 複数のアーカイブリソースやリソースアクセサを扱うサンプル
    SampleMode_TextureChange,        // ペインのテクスチャを自ら読み込んだテクスチャに変更します。
};

//------------------------------------------------------------------------------
//  変数の宣言
//------------------------------------------------------------------------------
static SampleMode g_SampleMode = SampleMode_TextureChange;

static int g_AllocationCount = 0;

// アプリケーションヒープとして確保されるメモリ領域の大きさ(byte)
static const size_t ApplicationHeapSize = 256 * 1024 * 1024;
// メモリプールとして確保されるメモリ領域の大きさ(byte)
static const size_t MemoryPoolSize          = 64 * 1024 * 1024;   // 64k の倍数である必要がある
static const size_t MemoryPoolSizeInvisible = 64 * 1024 * 1024;   // 64k の倍数である必要がある
// コマンドバッファで使用するメモリ領域の大きさ(byte)
static const size_t CommandBufferMemorySize = 32 * 1024 * 1024;
static const size_t CommandBufferControlMemorySize = 256 * 1024;

// 各種ディスクリプタプールの最大数
static const int DescriptorPoolSlotCount = 2048;

// 画面サイズ
static uint32_t ScreenWidth = 1280;
static uint32_t ScreenHeight = 720;

// アプリケーションヒープ
static nn::util::BytePtr g_pMemoryHeap(NULL);
static nn::mem::StandardAllocator g_ApplicationHeapHandle;

// コマンドバッファコントロール用メモリ
static void* g_pCommandBufferControlMemory = NULL;

// メモリプール
static void* g_MemoryPoolMemory;
static nn::gfx::MemoryPool g_MemoryPool;
static ptrdiff_t g_MemoryPoolOffset;

static void* g_MemoryPoolInvisibleMemory;
static nn::gfx::MemoryPool g_MemoryPoolInvisible;
static ptrdiff_t g_MemoryPoolInvisibleOffset;

// vi ハンドル
static nn::vi::Display* g_pDisplay;
// vi レイヤハンドル
static nn::vi::Layer* g_pLayer;

// gfx デバイス
static nn::gfx::Device g_Device;
// gfx スワップチェイン
static nn::gfx::SwapChain g_SwapChain;
// gfx キュー
static nn::gfx::Queue g_Queue;
// gfx コマンドバッファ
static nn::gfx::CommandBuffer g_CommandBuffer;
static ptrdiff_t g_CommandBufferStartOffset;
// gfx ビューポートシザーステート
static nn::gfx::ViewportScissorState g_ViewportScissor;
// gfx ラスタライザステート
static nn::gfx::RasterizerState g_RasterizerState;
// gfx デプスステンシルステート
static nn::gfx::DepthStencilState g_DepthStencilState;
// gfx ブレンドステート
static nn::gfx::BlendState  g_BlendState;
// gfx カラーバッファ用テクスチャ
static nn::gfx::Texture g_ColorBuffer;
// gfx カラーターゲットビュー
static nn::gfx::ColorTargetView g_ColorBufferView;
// gfx デプスステンシル用テクスチャ
static nn::gfx::Texture g_DepthStencilBuffer;
// gfx デプスステンシルビュー
static nn::gfx::DepthStencilView g_DepthStencilView;

// ui2d レイアウト
static nn::ui2d::Layout* g_pLayout = NULL;
static nn::ui2d::Layout* g_pLayout2 = NULL; // SampleMode_MultiArcResource用
static nn::ui2d::Layout* g_pLayout3 = NULL; // SampleMode_MultiArcResource用

// ui2d アーカイブリソースアクセッサ
static nn::ui2d::ArcResourceAccessor* g_pArcResourceAccessor = NULL;
static nn::ui2d::ArchiveHandle* g_pArchiveHandle = NULL; // SampleMode_MultiArcResource用
static nn::ui2d::ArchiveHandle* g_pArchiveHandle2 = NULL; // SampleMode_MultiArcResource用
static nn::ui2d::MultiArcResourceAccessor* g_pMultiArcResourceAccessor1 = NULL; // SampleMode_MultiArcResource用
static nn::ui2d::MultiArcResourceAccessor* g_pMultiArcResourceAccessor2 = NULL; // SampleMode_MultiArcResource用
// レイアウトアーカイブバイナリデータ
static void* g_pLayoutArchiveBinary;
static void* g_pLayoutArchiveBinary2; // SampleMode_MultiArcResource用
// テクスチャ情報（テクスチャを差し替える際に利用します）
static nn::ui2d::PlacementTextureInfo g_PlacementTextureInfo; // SampleMode_TextureChange用
// レイアウトで使用するフォントリソース
static nn::font::ResFont* g_pFont = NULL;
// ui2d が描画に使用するグラフィックスリソース
static nn::ui2d::GraphicsResource g_GraphicsResource;
// GPU バッファのためのメモリプールポインタ
static nn::util::BytePtr    g_pUi2dDrawMemoryPoolPtr( NULL );
// フォント描画の際に使用する描画用 GPU バッファのためのメモリプール
static nn::gfx::MemoryPool  g_Ui2dDrawMemoryPool;
// ui2d 描画の際に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dConstantBuffer;
// フォントの描画に使用するコンスタントバッファ
static nn::font::GpuBuffer  g_Ui2dFontConstantBuffer;
// ui2d のコンスタントバッファのダブルバッファリング用インデックス
static uint32_t     g_Ui2dConstantBufferIndex = 0;
// ui2d 描画情報
nn::ui2d::DrawInfo         g_DrawInfo;

//------------------------------------------------------------------------------
// DescriptorPool 関連
const  int                             DescriptorPoolStartSlot = 256;
static nn::gfx::DescriptorPool         g_TextureDescriptorPool;
static nn::gfx::DescriptorPool         g_SamplerDescriptorPool;
static int                             g_SamplerSlotCount = DescriptorPoolStartSlot;
static int                             g_TextureSlotCount = DescriptorPoolStartSlot;

nn::gfx::Device* g_pDevice;
nns::ConfigSwitcher* g_pConfigSwitcher;

//------------------------------------------------------------------------------
//  アプリケーションヒープからメモリを確保する
//------------------------------------------------------------------------------
static void* AllocateFromApplicationHeap(size_t size, size_t alignment)
{
    void* ptr = g_ApplicationHeapHandle.Allocate(size, alignment);
    if (ptr != NULL)
    {
        g_AllocationCount++;
    }
    return ptr;
}

static void* AllocateFromApplicationHeap(size_t size)
{
    return AllocateFromApplicationHeap(size, 4);
}

//------------------------------------------------------------------------------
//  アプリケーションヒープから確保したメモリを開放する
//------------------------------------------------------------------------------
static void DeallocateApplicationHeap(void* ptr)
{
    if (ptr == NULL)
    {
        return;
    }
    g_AllocationCount--;
    g_ApplicationHeapHandle.Free(ptr);
}

template <typename TObject>
TObject* AllocAndConstruct()
{
    return new(AllocateFromApplicationHeap(sizeof(TObject))) TObject();
}

template <typename TObject>
void DestructAndFree(TObject* ptr)
{
    ptr->TObject::~TObject();
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// ui2d 用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* Ui2dAllocateFunction(size_t size, size_t alignment, void* pUserData)
{
    NN_UNUSED(pUserData);
    return AllocateFromApplicationHeap(size, alignment);
}

static void Ui2dDeallocateFunction(void* ptr, void* pUserData)
{
    NN_UNUSED(pUserData);
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// nn::fs用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
static void* FsAllocateFunction( size_t size )
{
    return AllocateFromApplicationHeap(size);
}

static void FsDeallocateFunction( void* ptr, size_t size )
{
    NN_UNUSED( size );
    DeallocateApplicationHeap(ptr);
}

//------------------------------------------------------------------------------
// オフセットの切り上げ
//------------------------------------------------------------------------------
static ptrdiff_t AlignupOffset(ptrdiff_t offset, size_t alignment)
{
    return ((offset + alignment - 1) / alignment) * alignment;
}

//------------------------------------------------------------------------------
// gfx で使用するメモリプール初期化関数
//------------------------------------------------------------------------------
void InitializeMemoryPool()
{
    // 単純さのため全体で 2種類のメモリプールを使います
    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );
        info.SetPoolMemory( g_MemoryPoolMemory, MemoryPoolSize );

        g_MemoryPool.Initialize( g_pDevice, info );
        g_MemoryPoolOffset = 0;
    }

    {
        nn::gfx::MemoryPool::InfoType info;
        info.SetDefault();
        info.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuInvisible | nn::gfx::MemoryPoolProperty_GpuCached | nn::gfx::MemoryPoolProperty_Compressible);
        info.SetPoolMemory( g_MemoryPoolInvisibleMemory, MemoryPoolSizeInvisible );

        g_MemoryPoolInvisible.Initialize( g_pDevice, info );
        g_MemoryPoolInvisibleOffset = 0;
    }
}

//-----------------------------------------------------------------------------
//  gfx のディスクリプタプール初期化関数
//-----------------------------------------------------------------------------
static void InitialzieGfxDescriptorPool_()
{
    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_TextureView );
        info.SetSlotCount( DescriptorPoolSlotCount );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( g_pDevice, info );

        g_TextureDescriptorPool.Initialize( g_pDevice, info, &g_MemoryPool, g_MemoryPoolOffset, size );
        g_MemoryPoolOffset += size;
    }

    {
        nn::gfx::DescriptorPool::InfoType info;
        info.SetDefault();
        info.SetDescriptorPoolType( nn::gfx::DescriptorPoolType_Sampler);
        info.SetSlotCount( DescriptorPoolSlotCount );

        size_t size = nn::gfx::DescriptorPool::CalculateDescriptorPoolSize( g_pDevice, info );

        g_SamplerDescriptorPool.Initialize( g_pDevice, info, &g_MemoryPool, g_MemoryPoolOffset, size );
        g_MemoryPoolOffset += size;
    }
}

//------------------------------------------------------------------------------

bool RegisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_TextureSlotCount;
    NN_ASSERT(g_TextureSlotCount < DescriptorPoolSlotCount);

    g_TextureDescriptorPool.BeginUpdate();
    {
        g_TextureDescriptorPool.SetTextureView( slot, &textureView);
        g_TextureSlotCount++;
    }
    g_TextureDescriptorPool.EndUpdate();

    g_TextureDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

bool RegisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    NN_UNUSED(pUserData);

    const int slot = g_SamplerSlotCount;
    NN_ASSERT(g_SamplerSlotCount < DescriptorPoolSlotCount);
    g_SamplerDescriptorPool.BeginUpdate();
    {
        g_SamplerDescriptorPool.SetSampler( slot, &sampler);
        g_SamplerSlotCount++;
    }
    g_SamplerDescriptorPool.EndUpdate();

    g_SamplerDescriptorPool.GetDescriptorSlot(pDstSlot, slot);

    return true;
}

//------------------------------------------------------------------------------

void UnregisterSlotForTexture_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::TextureView& textureView, void* pUserData)
{
    // このサンプルではなにも行いません。
    // ユーザーがアプリケーションに適した方法で、ディスクリタプールの解放処理を行ってください。
    NN_UNUSED(pUserData);
    NN_UNUSED(pDstSlot);
    NN_UNUSED(textureView);
}

//------------------------------------------------------------------------------

void UnregisterSlotForSampler_(nn::gfx::DescriptorSlot* pDstSlot, const nn::gfx::Sampler& sampler, void* pUserData)
{
    // このサンプルではなにも行いません。
    // ユーザーがアプリケーションに適した方法で、ディスクリタプールの解放処理を行ってください。
    NN_UNUSED(pUserData);
    NN_UNUSED(pDstSlot);
    NN_UNUSED(sampler);
}

//------------------------------------------------------------------------------
// メモリを確保してファイルを読み込みます
//------------------------------------------------------------------------------
void* ReadFileWithAllocate(const char* pFileName, size_t alignment)
{
    nn::Result result;
    nn::fs::FileHandle fileHandle;
    int64_t fileSize;

    result = nn::fs::OpenFile(&fileHandle, pFileName, nn::fs::OpenMode_Read);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    result = nn::fs::GetFileSize(&fileSize, fileHandle);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    void* binary = AllocateFromApplicationHeap(static_cast<size_t>(fileSize), alignment);
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, fileHandle, 0, binary, static_cast<size_t>(fileSize));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    nn::fs::CloseFile(fileHandle);

    return binary;
}

//------------------------------------------------------------------------------
// gfx の一部のクラスで共通するメモリ割り当て処理
//------------------------------------------------------------------------------
template <typename TGfxClass, typename TGfxInfo>
static void
SetGfxMemoryTmpl_(TGfxClass * pGfxObject, TGfxInfo& info )
{
    size_t memorySize = TGfxClass::GetRequiredMemorySize( info );
    void* pWork = AllocateFromApplicationHeap(memorySize);

    pGfxObject->SetMemory( pWork, memorySize );
}

//-----------------------------------------------------------------------------
// コマンドバッファーのコールバック処理
//-----------------------------------------------------------------------------
void OutOfMemoryEventCallback( nn::gfx::CommandBuffer* pCommandBuffer,
    const nn::gfx::OutOfMemoryEventArg& arg )
{
    NN_UNUSED( pCommandBuffer );
    NN_UNUSED( arg );

    // このサンプルでは予めコマンドバッファ用に十分な大きさのメモリを確保しているためメモリが不足することはありません。
    // ユーザーがアプリケーションに適した方法で追加のメモリを確保し、AddCommandMemory(...) や AddControlMemory(...) を使って
    // コマンドバッファにメモリを追加してください。
    NN_ABORT("CommandMemory shortage!\n");
}

#if defined( NN_BUILD_CONFIG_OS_WIN )
//------------------------------------------------------------------------------
static void InitializeGfxWin_(HWND hwnd, int width, int height )
{
    // 2015/4/1現在、gfxではウインドウサイズが設定出来ないようなので
    // Win32APIを呼び出して強制的にウインドウサイズを設定
    // 将来的には削除予定
    {
        RECT rw, rc;
        ::GetWindowRect(hwnd, &rw);
        ::GetClientRect(hwnd, &rc);

        int new_width = (rw.right - rw.left) - (rc.right - rc.left) + width;
        int new_height = (rw.bottom - rw.top) - (rc.bottom - rc.top) + height;

        SetWindowPos(hwnd, NULL, 0, 0, new_width, new_height, SWP_NOMOVE | SWP_NOZORDER);
        SetWindowText(hwnd, L"Ui2dSimple");
    }
}
#endif

//-----------------------------------------------------------------------------
// スワップチェーンを初期化
//-----------------------------------------------------------------------------
void InitializeSwapChain()
{
    nn::gfx::SwapChain::InfoType info;
    info.SetDefault();
    info.SetLayer(g_pLayer);
    info.SetWidth(ScreenWidth);
    info.SetHeight(ScreenHeight);
    info.SetFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);

    size_t size = g_SwapChain.CalculateScanBufferSize(g_pDevice, info);
    g_MemoryPoolInvisibleOffset = AlignupOffset(
                                    g_MemoryPoolInvisibleOffset,
                                    nn::gfx::SwapChain::GetScanBufferAlignment(g_pDevice, info));
    g_SwapChain.Initialize(g_pDevice, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
    g_MemoryPoolInvisibleOffset += size;
}

//-----------------------------------------------------------------------------
// キューを初期化
//-----------------------------------------------------------------------------
void InitializeQueue()
{
    nn::gfx::Queue::InfoType info;
    info.SetDefault();
    info.SetCapability(nn::gfx::QueueCapability_Graphics);
    g_Queue.Initialize(g_pDevice, info);
}

//-----------------------------------------------------------------------------
// カラーターゲット関連を初期化
//-----------------------------------------------------------------------------
void InitializeColorBuffer()
{
    // カラーバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(ScreenWidth);
        info.SetHeight(ScreenHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_ColorBuffer);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetMipCount(1);
        info.SetDepth(1);

        g_MemoryPoolInvisibleOffset = AlignupOffset(
                                        g_MemoryPoolInvisibleOffset,
                                        nn::gfx::Texture::CalculateMipDataAlignment(g_pDevice, info));
        size_t size = g_ColorBuffer.CalculateMipDataSize(g_pDevice, info);
        g_ColorBuffer.Initialize(g_pDevice, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
        g_MemoryPoolInvisibleOffset += size;
    }

    // カラーバッファビューを初期化
    {
        nn::gfx::ColorTargetView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_R8_G8_B8_A8_UnormSrgb);
        info.SetTexturePtr(&g_ColorBuffer);
        g_ColorBufferView.Initialize(g_pDevice, info);
    }
}

//-----------------------------------------------------------------------------
// 深度ステンシルバッファ関連を初期化
//-----------------------------------------------------------------------------
void InitializeDepthStencilBuffer()
{
    // 深度ステンシルバッファ用のテクスチャを初期化
    {
        nn::gfx::Texture::InfoType info;
        info.SetDefault();
        info.SetWidth(ScreenWidth);
        info.SetHeight(ScreenHeight);
        info.SetGpuAccessFlags(nn::gfx::GpuAccess_DepthStencil);
        info.SetImageStorageDimension(nn::gfx::ImageStorageDimension_2d);
        info.SetImageFormat(nn::gfx::ImageFormat_D16_Unorm);
        info.SetMipCount(1);
        info.SetDepth(1);

        g_MemoryPoolInvisibleOffset = AlignupOffset(
                                        g_MemoryPoolInvisibleOffset,
                                        nn::gfx::Texture::CalculateMipDataAlignment(g_pDevice, info));
        size_t size = g_DepthStencilBuffer.CalculateMipDataSize(g_pDevice, info);
        g_DepthStencilBuffer.Initialize(g_pDevice, info, &g_MemoryPoolInvisible, g_MemoryPoolInvisibleOffset, size);
        g_MemoryPoolInvisibleOffset += size;
    }

    // 深度ステンシルビューの初期化
    {
        nn::gfx::DepthStencilView::InfoType info;
        info.SetDefault();
        info.SetImageDimension(nn::gfx::ImageDimension_2d);
        info.SetTexturePtr(&g_DepthStencilBuffer);
        g_DepthStencilView.Initialize(g_pDevice, info);
    }
}

//-----------------------------------------------------------------------------
// コマンドバッファを初期化
//-----------------------------------------------------------------------------
void InitializeCommandBuffer()
{
    nn::gfx::CommandBuffer::InfoType info;
    info.SetDefault();
    info.SetQueueCapability( nn::gfx::QueueCapability_Graphics );
    info.SetCommandBufferType( nn::gfx::CommandBufferType_Direct );
    g_CommandBuffer.Initialize(g_pDevice, info);

    // バッファ（メモリプールより確保）
    g_MemoryPoolOffset = AlignupOffset(g_MemoryPoolOffset, nn::gfx::CommandBuffer::GetCommandMemoryAlignment(g_pDevice));
    g_CommandBufferStartOffset = g_MemoryPoolOffset;
    g_MemoryPoolOffset += CommandBufferMemorySize;

    // コントロールメモリ（通常メモリより確保）
    g_pCommandBufferControlMemory = AllocateFromApplicationHeap(
                                        CommandBufferControlMemorySize,
                                        nn::gfx::CommandBuffer::GetControlMemoryAlignment(g_pDevice) );
}

//-----------------------------------------------------------------------------
// ビューポートシザーを初期化
//-----------------------------------------------------------------------------
void InitializeViewportScissorState()
{
    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled(true);

    nn::gfx::ViewportStateInfo viewportInfo;
    viewportInfo.SetDefault();
    viewportInfo.SetWidth(static_cast<float>(ScreenWidth));
    viewportInfo.SetHeight(static_cast<float>(ScreenHeight));
    info.SetViewportStateInfoArray(&viewportInfo, 1);

    nn::gfx::ScissorStateInfo scissorInfo;
    scissorInfo.SetDefault();
    scissorInfo.SetWidth(ScreenWidth);
    scissorInfo.SetHeight(ScreenHeight);
    info.SetScissorStateInfoArray(&scissorInfo, 1);

    SetGfxMemoryTmpl_(&g_ViewportScissor, info);
    g_ViewportScissor.Initialize(g_pDevice, info);
}

//-----------------------------------------------------------------------------
// ブレンドステートを初期化
//-----------------------------------------------------------------------------
void InitializeBlendState()
{
    nn::gfx::BlendState::InfoType info;
    info.SetDefault();
    nn::gfx::BlendTargetStateInfo targetInfo;
    {
        targetInfo.SetDefault();
        targetInfo.SetChannelMask( nn::gfx::ChannelMask_Red | nn::gfx::ChannelMask_Green | nn::gfx::ChannelMask_Blue );
        targetInfo.SetBlendEnabled(true);
        targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
        targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
    };
    info.SetBlendTargetStateInfoArray( &targetInfo, 1 );

    SetGfxMemoryTmpl_(&g_BlendState, info);

    g_BlendState.Initialize(g_pDevice, info);
}

//-----------------------------------------------------------------------------
// ラスタライザステートを初期化
//-----------------------------------------------------------------------------
void InitializeRasterizerState()
{
    // ui2dライブラリ内ではカリングの設定は行われないので、
    // 明示的にCullMode_BackもしくはCullMode_Noneに設定する必要があります。
    nn::gfx::RasterizerState::InfoType info;
    info.SetDefault();
    info.SetCullMode(nn::gfx::CullMode_Back);
    info.SetScissorEnabled(true);
    info.SetDepthClipEnabled(false);
    g_RasterizerState.Initialize(g_pDevice, info);
}

//-----------------------------------------------------------------------------
// 深度ステンシルステートを初期化
//-----------------------------------------------------------------------------
void InitializeDepthStencilState()
{
    nn::gfx::DepthStencilState::InfoType info;
    info.SetDefault();
    info.SetDepthTestEnabled(false);
    info.SetDepthWriteEnabled(false);
    g_DepthStencilState.Initialize(g_pDevice, info);
}

//-----------------------------------------------------------------------------
// gfx 初期化処理
//-----------------------------------------------------------------------------
void InitializeGfx()
{
    // ライブラリを初期化
    //nn::gfx::Initialize();

    // デバイスを初期化
    //InitializeDevice();

    // gfx で使用するメモリプールを初期化
    {
        nn::gfx::MemoryPoolInfo info;
        info.SetDefault();

        const size_t MemoryPoolAligment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(g_pDevice, info);
        g_MemoryPoolMemory = AllocateFromApplicationHeap(MemoryPoolSize, MemoryPoolAligment);
        g_MemoryPoolOffset = 0;

        g_MemoryPoolInvisibleMemory = AllocateFromApplicationHeap(MemoryPoolSizeInvisible, MemoryPoolAligment);
        g_MemoryPoolInvisibleOffset = 0;

        InitializeMemoryPool();
    }

#if defined( NN_BUILD_CONFIG_OS_WIN )
    HWND hwnd;
    nn::vi::GetNativeWindow((nn::vi::NativeWindowHandle*)&hwnd, g_pLayer);
    InitializeGfxWin_(hwnd, ScreenWidth, ScreenHeight);
#endif

    // スワップチェーンを初期化
    InitializeSwapChain();

    // キューを初期化
    InitializeQueue();

    // カラーバッファ関連を初期化
    InitializeColorBuffer();

    // 深度ステンシルバッファ関連を初期化
    InitializeDepthStencilBuffer();

    // ディスクリプションプールを初期化
    InitialzieGfxDescriptorPool_();

    // コマンドバッファを初期化
    InitializeCommandBuffer();

    // ビューポートシザーを初期化
    InitializeViewportScissorState();

    // ブレンドステートを初期化
    InitializeBlendState();

    // ラスタライザステートを初期化
    InitializeRasterizerState();

    // 深度ステンシルステートを初期化
    InitializeDepthStencilState();

    g_pConfigSwitcher->Initialize(g_pDevice, g_pDisplay,
        &g_TextureDescriptorPool, &g_SamplerDescriptorPool,
        DescriptorPoolSlotCount, DescriptorPoolSlotCount);
}

//-----------------------------------------------------------------------------
// gfx 終了処理
//-----------------------------------------------------------------------------
void FinalizeGfx()
{
    g_TextureDescriptorPool.Finalize( g_pDevice );
    g_SamplerDescriptorPool.Finalize( g_pDevice );

    void* pViewportScissorMemory = g_ViewportScissor.GetMemory();
    void* pBlendMemory = g_BlendState.GetMemory();

    g_DepthStencilView.Finalize(g_pDevice);
    g_DepthStencilBuffer.Finalize(g_pDevice);
    g_ColorBufferView.Finalize(g_pDevice);
    g_ColorBuffer.Finalize(g_pDevice);
    g_DepthStencilState.Finalize(g_pDevice);
    g_RasterizerState.Finalize(g_pDevice);
    g_BlendState.Finalize(g_pDevice);
    g_ViewportScissor.Finalize(g_pDevice);
    g_CommandBuffer.Finalize(g_pDevice);
    g_Queue.Finalize(g_pDevice);
    g_SwapChain.Finalize(g_pDevice);

    DeallocateApplicationHeap(pViewportScissorMemory);
    DeallocateApplicationHeap(pBlendMemory);
    DeallocateApplicationHeap(g_pCommandBufferControlMemory);

    g_MemoryPool.Finalize(g_pDevice);
    g_MemoryPoolInvisible.Finalize(g_pDevice);
    DeallocateApplicationHeap(g_MemoryPoolMemory);
    DeallocateApplicationHeap(g_MemoryPoolInvisibleMemory);

    //g_Device.Finalize();

    //nn::gfx::Finalize();
}

//------------------------------------------------------------------------------
// ui2d 描画のためのバッファ関連初期化処理
//------------------------------------------------------------------------------
void InitializeUi2dBuffers(nn::ui2d::DrawInfo& drawInfo, const nn::ui2d::BuildResultInformation& buildResultInformation)
{
    // バッファの再セットアップ
    // ui2d 描画に使用する各種 gfx バッファのためのメモリプールを作成
    nn::gfx::MemoryPoolInfo memPoolInfo;
    memPoolInfo.SetDefault();
    memPoolInfo.SetMemoryPoolProperty( nn::gfx::MemoryPoolProperty_CpuUncached | nn::gfx::MemoryPoolProperty_GpuCached );

    // BuildWithName でレイアウトデータをセットアップした際に BuildResultInformation に収集した
    // 各種バッファのサイズを満たすメモリプールを作成する。

    const size_t constantBufferTotalSize = buildResultInformation.requiredUi2dConstantBufferSize
                                            + buildResultInformation.requiredFontConstantBufferSize;

    const size_t memoryPoolGranularity = nn::gfx::MemoryPool::GetPoolMemorySizeGranularity(
                                                                g_pDevice,
                                                                memPoolInfo);

    const size_t memoryPoolSize = nn::util::align_up(
                                    constantBufferTotalSize, memoryPoolGranularity);

    const size_t MemoryPoolAlignment = nn::gfx::MemoryPool::GetPoolMemoryAlignment(
                                                                g_pDevice,
                                                                memPoolInfo);

    // ui2d 用コンスタントバッファ、font 用コンスタントバッファの順番で
    // メモリプール上に領域を確保する。

    if (memoryPoolSize > 0)
    {
        size_t  totalMemoryPoolSize = memoryPoolSize * nn::ui2d::ConstantBufferCount;
        void*   pMemoryPoolMemory = Ui2dAllocateFunction(totalMemoryPoolSize, MemoryPoolAlignment, NULL);

        g_pUi2dDrawMemoryPoolPtr.Reset(pMemoryPoolMemory);
        memPoolInfo.SetPoolMemory(g_pUi2dDrawMemoryPoolPtr.Get(), totalMemoryPoolSize);

        g_Ui2dDrawMemoryPool.Initialize(g_pDevice, memPoolInfo);
    }

    size_t  bufferOffset = 0;

    // ui2d 用のコンスタントバッファを作成する。
    if (buildResultInformation.requiredUi2dConstantBufferSize > 0)
    {
        nn::font::GpuBuffer::InitializeArg  arg;

        arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
        arg.SetBufferSize(buildResultInformation.requiredUi2dConstantBufferSize);
        arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
        arg.SetMemoryPool(&g_Ui2dDrawMemoryPool);
        arg.SetMemoryPoolOffset(bufferOffset);
        arg.SetAllocator(Ui2dAllocateFunction, NULL);
        arg.SetAllocateSyncFlag(false);

        // フォント描画のためのコンスタントバッファを作成
        g_Ui2dConstantBuffer.Initialize(g_pDevice, arg);
        drawInfo.SetUi2dConstantBuffer(&g_Ui2dConstantBuffer);
        bufferOffset += buildResultInformation.requiredUi2dConstantBufferSize * nn::ui2d::ConstantBufferCount;

    }
    else
    {
        drawInfo.SetUi2dConstantBuffer(NULL);
    }

    // ui2d 内で使用される font 用のコンスタントバッファを作成する。
    if (buildResultInformation.requiredFontConstantBufferSize > 0)
    {
        nn::font::GpuBuffer::InitializeArg  arg;

        arg.SetGpuAccessFlag(nn::gfx::GpuAccess_ConstantBuffer);
        arg.SetBufferSize(buildResultInformation.requiredFontConstantBufferSize);
        arg.SetBufferCount(nn::ui2d::ConstantBufferCount);
        arg.SetMemoryPool(&g_Ui2dDrawMemoryPool);
        arg.SetMemoryPoolOffset(bufferOffset);
        arg.SetAllocator(Ui2dAllocateFunction, NULL);
        arg.SetAllocateSyncFlag(false);

        g_Ui2dFontConstantBuffer.Initialize(g_pDevice, arg);
        drawInfo.SetFontConstantBuffer(&g_Ui2dFontConstantBuffer);
        bufferOffset += buildResultInformation.requiredFontConstantBufferSize * nn::ui2d::ConstantBufferCount;
    }
    else
    {
        drawInfo.SetFontConstantBuffer(NULL);
    }
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理(SampleMode_Simple)
//------------------------------------------------------------------------------
void InitializeUi2dRuntimeForSimple()
{
    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pArcResourceAccessor = AllocAndConstruct<nn::ui2d::ArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dSimple.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        bool    result = g_pArcResourceAccessor->Attach(g_pLayoutArchiveBinary, ".");
        NN_ASSERT(result);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(g_pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);

        g_pArcResourceAccessor->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();
    nn::ui2d::Layout::BuildOption   opt;
    opt.SetDefault();
    nn::ui2d::BuildResultInformation buildResultInformation;
    buildResultInformation.SetDefault();

    g_pLayout->BuildWithName(&buildResultInformation, g_pDevice, g_pArcResourceAccessor, NULL, NULL, opt, "simple.bflyt");

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(g_pDevice, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, NULL);

    // Ui2d の描画に使用される各種バッファを初期化して DrawInfo へ設定する
    InitializeUi2dBuffers(g_DrawInfo, buildResultInformation);

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(
            &projection,
            rect.left,
            rect.right,
            rect.bottom,
            rect.top,
            0.0f,
            300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(
            &view,
            pos,
            target,
            up);

        g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
        g_DrawInfo.SetProjectionMtx(projection);
        g_DrawInfo.SetViewMtx(view);
    }

    nn::ui2d::Animator* pAnimator = g_pLayout->CreateGroupAnimator(g_pDevice, "loop");
    pAnimator->PlayAuto(1.0f);

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pArcResourceAccessor->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理(TextureChange)
//------------------------------------------------------------------------------
void InitializeUi2dRuntimeForTextureChange()
{
    // テクスチャを読み込んで、動的に変更します。
    // 読み込むテクスチャは、アーカイブ作成の直前に ftxb 形式のファイルを、バイナリコンバーターの画像出力フォルダにコピーしています。
    // (./Resources 以下のコンバートバッチファイルを参照してください)
    const nn::ui2d::TextureInfo* pInfo = g_pArcResourceAccessor->AcquireTexture(g_pDevice, "SampleTexture");
    NN_ASSERT_NOT_NULL(pInfo);
    NN_ASSERT(pInfo->IsValid(), "Can't load texture !!!");

    g_PlacementTextureInfo.Set(*pInfo);
    g_PlacementTextureInfo.ResetTextureDescriptorSlot();

    // スロットを設定します。
    RegisterSlotForTexture_(g_PlacementTextureInfo.GetTextureDescriptorSlot(), *pInfo->GetTextureView(), NULL);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理(SampleMode_Simple)
//------------------------------------------------------------------------------
void FinalizeUi2dRuntimeForSimple()
{
    g_Ui2dConstantBuffer.Finalize(g_pDevice, Ui2dDeallocateFunction, NULL);
    g_Ui2dFontConstantBuffer.Finalize(g_pDevice, Ui2dDeallocateFunction, NULL);

    g_Ui2dDrawMemoryPool.Finalize(g_pDevice);
    DeallocateApplicationHeap(g_pUi2dDrawMemoryPoolPtr.Get());

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, NULL);
    g_GraphicsResource.Finalize(g_pDevice);
    g_pLayout->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);

    void* pFontResource = g_pFont->RemoveResource(g_pDevice);
    DeallocateApplicationHeap(pFontResource);
    g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pFont->Finalize(g_pDevice);

    DestructAndFree<nn::font::ResFont>(g_pFont);

    g_pArcResourceAccessor->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pArcResourceAccessor->Detach();
    g_pArcResourceAccessor->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::ArcResourceAccessor>(g_pArcResourceAccessor);

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    DeallocateApplicationHeap(g_pLayoutArchiveBinary);
}

//------------------------------------------------------------------------------
// レイアウトデータセットアップ処理
//------------------------------------------------------------------------------
void BuildLayoutData(
    nn::ui2d::BuildResultInformation& totalRequiredBufferInfomation,
    nn::ui2d::Layout* pLayout,
    nn::ui2d::Layout::BuildOption& option,
    nn::ui2d::ResourceAccessor* pResourceAccessor,
    const char* pFileName)
{
    // レイアウトデータをセットアップして、必要なバッファサイズを収集します。
    nn::ui2d::BuildResultInformation   buildResultInformation;
    buildResultInformation.SetDefault();

    pLayout->BuildWithName(
        &buildResultInformation,
        g_pDevice,
        pResourceAccessor,
        NULL,
        NULL,
        option,
        pFileName);

    totalRequiredBufferInfomation.requiredUi2dConstantBufferSize
        += buildResultInformation.requiredUi2dConstantBufferSize;
    totalRequiredBufferInfomation.requiredFontConstantBufferSize
        += buildResultInformation.requiredFontConstantBufferSize;
}

//------------------------------------------------------------------------------
// ui2d 関連の初期化処理(SampleMode_MultiArcResource)
//------------------------------------------------------------------------------
void InitializeUi2dRuntimeForMultiArcResource()
{
    // レイアウトライブラリの初期化
    nn::ui2d::Initialize(Ui2dAllocateFunction, Ui2dDeallocateFunction, NULL);

    // リソースアクセサの初期化
    g_pMultiArcResourceAccessor1 = AllocAndConstruct<nn::ui2d::MultiArcResourceAccessor>();
    g_pMultiArcResourceAccessor2 = AllocAndConstruct<nn::ui2d::MultiArcResourceAccessor>();

    // レイアウトアーカイブの読み込み
    g_pLayoutArchiveBinary = ReadFileWithAllocate("Contents:/Ui2dSimple.arc", nn::ui2d::ArchiveResourceAlignment);
    g_pLayoutArchiveBinary2 = ReadFileWithAllocate("Contents:/Ui2dSimple2.arc", nn::ui2d::ArchiveResourceAlignment);
    {
        // アーカイブリソースをg_pArchiveHandleに構築して、複数のMultiArcResourceAccessorで共有させる
        g_pArchiveHandle = AllocAndConstruct<nn::ui2d::ArchiveHandle>();
        g_pArchiveHandle2 = AllocAndConstruct<nn::ui2d::ArchiveHandle>();

        g_pArchiveHandle->Initialize(g_pLayoutArchiveBinary, ".");
        g_pArchiveHandle2->Initialize(g_pLayoutArchiveBinary2, ".");

        g_pMultiArcResourceAccessor1->Attach(g_pArchiveHandle);
        g_pMultiArcResourceAccessor1->Attach(g_pArchiveHandle2);
        // g_pArchiveHandle は Accessor1 と Accessor2 双方にアタッチします。共有参照されている状態です。
        g_pMultiArcResourceAccessor2->Attach(g_pArchiveHandle);
    }

    // フォントの初期化
    g_pFont = AllocAndConstruct<nn::font::ResFont>();

    // フォントの読み込み
    {
        void* pFont = ReadFileWithAllocate("Contents:/sample.bffnt", nn::font::ResourceAlignment);
        bool    result = g_pFont->SetResource(g_pDevice, pFont);
        NN_ASSERT(result);
        g_pFont->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);

        g_pMultiArcResourceAccessor1->RegisterFont("sample.bffnt", g_pFont);
        g_pMultiArcResourceAccessor2->RegisterFont("sample.bffnt", g_pFont);
    }

    // レイアウトの初期化
    g_pLayout = AllocAndConstruct<nn::ui2d::Layout>();
    g_pLayout2 = AllocAndConstruct<nn::ui2d::Layout>();
    g_pLayout3 = AllocAndConstruct<nn::ui2d::Layout>();

    nn::ui2d::Layout::BuildOption   opt;
    opt.SetDefault();

    // 描画に必要なコンスタントバッファサイズを計算します。
    // 同時に表示するレイアウト分のバッファサイズを積算します。
    nn::ui2d::BuildResultInformation totalRequiredBufferInformation;
    totalRequiredBufferInformation.SetDefault();

    BuildLayoutData(
        totalRequiredBufferInformation,
        g_pLayout,
        opt,
        g_pMultiArcResourceAccessor1,
        "simple.bflyt");

    BuildLayoutData(
        totalRequiredBufferInformation,
        g_pLayout2,
        opt,
        g_pMultiArcResourceAccessor2,
        "simple.bflyt");

    BuildLayoutData(
        totalRequiredBufferInformation,
        g_pLayout3,
        opt,
        g_pMultiArcResourceAccessor1,
        "multiTexture.bflyt");

    // グラフィックスリソースの設定
    g_GraphicsResource.Setup(g_pDevice, 512);
    g_GraphicsResource.RegisterCommonSamplerSlot(RegisterSlotForSampler_, NULL);

    // Ui2d の描画に使用されるコンスタントバッファを初期化して DrawInfo へ設定する
    InitializeUi2dBuffers(g_DrawInfo, totalRequiredBufferInformation);

    // 表示位置・サイズの調整
    {
        nn::util::Float2 scale = NN_UTIL_FLOAT_2_INITIALIZER(0.2f, 0.2f);

        nn::util::Float2 t1 = NN_UTIL_FLOAT_2_INITIALIZER(-60.f, 50.f);
        g_pLayout->GetRootPane()->SetScale(scale);
        g_pLayout->GetRootPane()->SetTranslate(t1);

        nn::util::Float2 t2 = NN_UTIL_FLOAT_2_INITIALIZER( 60.f, 50.f);
        g_pLayout2->GetRootPane()->SetScale(scale);
        g_pLayout2->GetRootPane()->SetTranslate(t2);

        nn::util::Float2 t3 = NN_UTIL_FLOAT_2_INITIALIZER( 0.f, -50.f);
        g_pLayout3->GetRootPane()->SetScale(scale);
        g_pLayout3->GetRootPane()->SetTranslate(t3);
    }

    // 描画に使用する情報の設定
    {
        nn::util::MatrixT4x4fType   projection;
        nn::font::Rectangle rect = g_pLayout->GetLayoutRect();
        nn::util::MatrixOrthographicOffCenterRightHanded(
            &projection,
            rect.left,
            rect.right,
            rect.bottom,
            rect.top,
            0.0f,
            300.0f);
        nn::util::MatrixT4x3fType   view;
        nn::util::Vector3fType  pos;
        nn::util::Vector3fType  up;
        nn::util::Vector3fType  target;
        nn::util::VectorSet(&pos, 0.0f, 0.0f, 1.0f);
        nn::util::VectorSet(&up, 0.0f, 1.0f, 0.0f);
        nn::util::VectorSet(&target, 0.0f, 0.0f, 0.0f);
        nn::util::MatrixLookAtRightHanded(
            &view,
            pos,
            target,
            up);

        g_DrawInfo.SetGraphicsResource(&g_GraphicsResource);
        g_DrawInfo.SetProjectionMtx(projection);
        g_DrawInfo.SetViewMtx(view);
    }

    // アニメーションの作成と再生
    nn::ui2d::GroupAnimator* pAnimator = g_pLayout->CreateGroupAnimator(g_pDevice, "loop");
    pAnimator->PlayAuto(1.0f);

    // テクスチャパターンアニメーションに利用されるテクスチャが正しく登録されるように、アニメーションの初期化後に実行します。
    g_pMultiArcResourceAccessor1->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);
    g_pMultiArcResourceAccessor2->RegisterTextureViewToDescriptorPool(RegisterSlotForTexture_, NULL);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理(MultiArcResource)
//------------------------------------------------------------------------------
void FinalizeUi2dRuntimeForMultiArcResource()
{
    g_Ui2dConstantBuffer.Finalize(g_pDevice, Ui2dDeallocateFunction, NULL);
    g_Ui2dFontConstantBuffer.Finalize(g_pDevice, Ui2dDeallocateFunction, NULL);

    g_Ui2dDrawMemoryPool.Finalize(g_pDevice);
    DeallocateApplicationHeap(g_pUi2dDrawMemoryPoolPtr.Get());

    g_GraphicsResource.UnregisterCommonSamplerSlot(UnregisterSlotForSampler_, NULL);
    g_GraphicsResource.Finalize(g_pDevice);
    g_pLayout->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout);

    if (g_pLayout2 != NULL)
    {
        g_pLayout2->Finalize(g_pDevice);
        DestructAndFree<nn::ui2d::Layout>(g_pLayout2);
        g_pLayout2 = NULL;
    }

    g_pLayout3->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::Layout>(g_pLayout3);
    g_pLayout3 = NULL;

    void* pFontResource = g_pFont->RemoveResource(g_pDevice);
    DeallocateApplicationHeap(pFontResource);
    g_pFont->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pFont->Finalize(g_pDevice);

    DestructAndFree<nn::font::ResFont>(g_pFont);

    if (g_pMultiArcResourceAccessor2 != NULL)
    {
        g_pMultiArcResourceAccessor2->Detach(g_pArchiveHandle);
        g_pMultiArcResourceAccessor2->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
        g_pMultiArcResourceAccessor2->Finalize(g_pDevice);
        DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor2);
        g_pMultiArcResourceAccessor2 = NULL;
    }

    // g_pMultiArcResourceAccessor1はg_pArchiveHandleのDetachを行わずにFinalizeを呼ぶことで、
    // g_pArchiveHandleの解放が自動で行われます。
    g_pMultiArcResourceAccessor1->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
    g_pMultiArcResourceAccessor1->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor1);

    g_pArchiveHandle->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::ArchiveHandle>(g_pArchiveHandle);

    g_pArchiveHandle2->Finalize(g_pDevice);
    DestructAndFree<nn::ui2d::ArchiveHandle>(g_pArchiveHandle2);

    // ArcResourceAccessor の Finalize でアクセスされるため最後に開放する。
    DeallocateApplicationHeap(g_pLayoutArchiveBinary);
    DeallocateApplicationHeap(g_pLayoutArchiveBinary2);
}

//------------------------------------------------------------------------------
// ui2d 関連の終了処理(TextureChange)
//------------------------------------------------------------------------------
void FinalizeUi2dRuntimeForTextureChange()
{
    g_PlacementTextureInfo.Finalize(g_pDevice);
}

//------------------------------------------------------------------------------
//  SampleMode_MultiArcResourceの処理
//------------------------------------------------------------------------------
void CalculateForMultiArcResource(int frame)
{
    // リソースアクセサの1つを3秒後に解放する
    if (frame == 60 * 3)
    {
        // Layoutの1つを解放する
        if (g_pLayout2 != NULL)
        {
            g_pLayout2->Finalize(g_pDevice);
            DestructAndFree<nn::ui2d::Layout>(g_pLayout2);
            g_pLayout2 = NULL;
        }

        // MultiArcResourceAccessorの1つを解放する
        if (g_pMultiArcResourceAccessor2 != NULL)
        {
            // 複数のリソースアクセサで、1つのアーカイブリソースを共有することができます。
            // 1つのアーカイブリソースを複数のリソースアクセサにAttachした場合、
            // 1つのリソースアクセサがFinalizeするとAttachしていたアーカイブリソースが自動的に解放されてしまうため、
            // 他のリソースアクセサに影響が及んでしまいます。
            // このように1つのリソースを別のアクセサでも使っている場合には、
            // リソースを解放したくないリソースアクセサからリソースをDetachすることでFinalize時に解放しないようにできます。
            g_pMultiArcResourceAccessor2->Detach(g_pArchiveHandle);

            // Detachしたので、g_pArchiveHandle内のリソースは
            // UnregisterTextureViewFromDescriptorPoolおよびFinalizeの影響を受けません。
            g_pMultiArcResourceAccessor2->UnregisterTextureViewFromDescriptorPool(UnregisterSlotForTexture_, NULL);
            g_pMultiArcResourceAccessor2->Finalize(g_pDevice);

            DestructAndFree<nn::ui2d::MultiArcResourceAccessor>(g_pMultiArcResourceAccessor2);
            g_pMultiArcResourceAccessor2 = NULL;
        }
        NN_LOG("The second multiArcResourceAccessor was freed.\n");
    }
}

//------------------------------------------------------------------------------
//  SampleMode_TextureChange の処理
//------------------------------------------------------------------------------
void CalculateForTextureChange(int frame)
{
    if(frame == 60 * 3)
    {
        nn::ui2d::Pane* pTargetPane = g_pLayout->GetRootPane()->FindPaneByName("picture_ts");
        NN_ASSERT_NOT_NULL(pTargetPane);
        pTargetPane->GetMaterial()->SetTexMap(0, &g_PlacementTextureInfo);
    }
}

//------------------------------------------------------------------------------
// メインループ を中断するかどうかを判定
//------------------------------------------------------------------------------
bool CheckToBreakMainLoop()
{
#if defined(NN_BUILD_CONFIG_OS_WIN)
    // WindowsMessage の処理
    {
        MSG  msg;
        if ( PeekMessage( &msg, NULL, 0, 0, PM_REMOVE) )
        {
            TranslateMessage(&msg);
            if ( msg.message == WM_QUIT)
            {
                return true;
            }
            DispatchMessage(&msg);
        }
    }
#elif !defined(NN_BUILD_CONFIG_OS_HORIZON)
    if(g_Pad.IsHold(g_Pad.MASK_L|g_Pad.MASK_R))
    {
        return true;
    }
#endif

    return false;
};

//------------------------------------------------------------------------------
// メインループ (ループを中断する際は true を返します)
//------------------------------------------------------------------------------
bool MainLoop(int frame)
{
    // ループを中断する必要があるかどうか。
    if(CheckToBreakMainLoop())
    {
        return true;
    }

    // 各サンプルごとの更新処理
    {
        switch (g_SampleMode)
        {
        case SampleMode_Simple:
            break;
        case SampleMode_MultiArcResource:
            CalculateForMultiArcResource(frame);
            break;
        case SampleMode_TextureChange:
            CalculateForTextureChange(frame);
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // レイアウトの更新処理
    g_pLayout->AnimateAndUpdateAnimFrame();

    // 描画処理
    g_CommandBuffer.Reset();

    // コマンドバッファ用のメモリを設定します
    {
        g_CommandBuffer.AddCommandMemory( &g_MemoryPool, g_CommandBufferStartOffset, CommandBufferMemorySize);
        g_CommandBuffer.AddControlMemory(g_pCommandBufferControlMemory, CommandBufferControlMemorySize);
        g_CommandBuffer.SetOutOfCommandMemoryEventCallback( OutOfMemoryEventCallback );
        g_CommandBuffer.SetOutOfControlMemoryEventCallback( OutOfMemoryEventCallback );
    }

    g_CommandBuffer.Begin();
    {
        // ディスクリプタプール や シェーダーを更新した場合は、GPU のキャッシュを無効化します。
        g_CommandBuffer.InvalidateMemory(nn::gfx::GpuAccess_Descriptor | nn::gfx::GpuAccess_ShaderCode);
        // ディスクリプタプールをセットする。
        g_CommandBuffer.SetDescriptorPool( &g_TextureDescriptorPool );
        g_CommandBuffer.SetDescriptorPool( &g_SamplerDescriptorPool );

        nn::gfx::ColorTargetView* pTarget = &g_ColorBufferView;

        const nns::Color& color = g_pConfigSwitcher->GetBackGroundColorFromDisplayConfig();
        g_CommandBuffer.ClearColor(pTarget, color.rRatio, color.gRatio, color.bRatio, color.aRatio, NULL);
        g_CommandBuffer.ClearDepthStencil(&g_DepthStencilView, 1.0f, 0, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);
        g_CommandBuffer.SetRenderTargets(1, &pTarget, &g_DepthStencilView);
        g_CommandBuffer.SetViewportScissorState(&g_ViewportScissor);
        g_CommandBuffer.SetRasterizerState(&g_RasterizerState);
        g_CommandBuffer.SetDepthStencilState(&g_DepthStencilState);
        g_CommandBuffer.SetBlendState(&g_BlendState);

        // CalculateMtx
        {
            g_DrawInfo.Map(g_Ui2dConstantBufferIndex);
            {
                g_pLayout->Calculate(g_DrawInfo);
                if(g_pLayout2 != NULL)
                {
                    g_pLayout2->Calculate(g_DrawInfo);
                }
                if(g_pLayout3 != NULL)
                {
                    g_pLayout3->Calculate(g_DrawInfo);
                }
            }
            g_DrawInfo.Unmap();
            g_DrawInfo.SetGpuAccessBufferIndex(g_Ui2dConstantBufferIndex);
        }

        // Draw
        g_pLayout->Draw(g_DrawInfo, g_CommandBuffer);
        if(g_pLayout2 != NULL)
        {
            g_pLayout2->Draw(g_DrawInfo, g_CommandBuffer);
        }
        if(g_pLayout3 != NULL)
        {
            g_pLayout3->Draw(g_DrawInfo, g_CommandBuffer);
        }

        ++g_Ui2dConstantBufferIndex;
        g_Ui2dConstantBufferIndex %= nn::ui2d::ConstantBufferCount;
    }
    g_pConfigSwitcher->Draw(&g_CommandBuffer);
    g_CommandBuffer.End();

    // コマンドの実行
    g_Queue.ExecuteCommand(&g_CommandBuffer, NULL);
    g_Queue.CopyToScanBuffer(&g_SwapChain, &g_ColorBufferView);

    // Queue をフラッシュします。
    g_Queue.Flush();

    g_Queue.Present(&g_SwapChain, 1);
    g_Queue.Sync();
    g_Queue.Flush();

    return false;
}

void ShowMovingPictures(SetupGraphics* pGfxSetup, ConfigSwitcher* pSwitcher)
{
    g_pConfigSwitcher = pSwitcher;
    g_pLayer = pGfxSetup->GetLayer();
    g_pDisplay = pGfxSetup->GetDisplay();
    g_pDevice = pGfxSetup->GetDevice();

    // 初期化
    {
        // アプリケーションで使用するヒープ領域を初期化
        g_pMemoryHeap.Reset(reinterpret_cast<uint8_t*>(malloc(ApplicationHeapSize)));
        NN_ABORT_UNLESS_NOT_NULL(g_pMemoryHeap.Get());
        g_ApplicationHeapHandle.Initialize(g_pMemoryHeap.Get(), ApplicationHeapSize);

        // レイヤ を初期化
        //InitializeLayer();

        // gfx を初期化
        InitializeGfx();

        // ui2d ランタイムの初期化処理
        switch (g_SampleMode)
        {
        case SampleMode_Simple:
            InitializeUi2dRuntimeForSimple();
            break;
        case SampleMode_MultiArcResource:
            InitializeUi2dRuntimeForMultiArcResource();
            break;
        case SampleMode_TextureChange:
            InitializeUi2dRuntimeForSimple();
            InitializeUi2dRuntimeForTextureChange();
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
    }

    // メインループ
    NN_LOG("Start demo.\n");
    for(int frame = 0; frame < 60 * 1 || !g_pConfigSwitcher->IsSceneSwitchedOn(); frame++)
    {
        bool needToBreak = MainLoop(frame);
        if(needToBreak)
        {
            break;
        }
    }

    // 終了処理
    {
        switch (g_SampleMode)
        {
        case SampleMode_Simple:
            FinalizeUi2dRuntimeForSimple();
            break;
        case SampleMode_MultiArcResource:
            FinalizeUi2dRuntimeForMultiArcResource();
            break;
        case SampleMode_TextureChange:
            FinalizeUi2dRuntimeForSimple();
            FinalizeUi2dRuntimeForTextureChange();
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        g_pConfigSwitcher->Finalize();

        // gfx 終了処理
        FinalizeGfx();

        // レイヤ 終了処理
        //FinalizeLayer();

        // ヒープメモリーの終了処理
        g_ApplicationHeapHandle.Finalize();
        free(g_pMemoryHeap.Get());
    }

    NN_ASSERT(g_AllocationCount == 0);
}
} // nns
