﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ConfigSwitcher.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>

namespace nns {

namespace {
const uint8_t FontColorLight = 0xe0;
const uint8_t FontColorDark = 0x40;
const int CharCountMax = 1024;
const int LcdWidth = 1280;
const int LcdHeight = 720;
const int SceneSwitchKeepingPressBoundary = 100;
const int SceneSwitchKeepingTouchBoundary = 1400;
const int InputKeepingBoundary = 30;
const float FontWidthRaito = 20.0f / 37.0f;
const float FontHeightRaito = 1.0f;
const float CaptionLeft = 50.0f;
const float CaptionTop = 30.0f;
const float LableLeft = 560.0f;
const float FontSizeDefault = 64.0f;
const float FontSizeLarge = FontSizeDefault * 1.5f;
const float MarkSize = 72.0f;
const float TouchInfoFontSize = 32.0f;
const nn::util::Unorm8x4 MarkColor = {{0xff, 0xff, 0x00, 0xff}};
const size_t ListenerStackSize = 32 * 1024;
NN_ALIGNAS(nn::os::ThreadStackAlignment) char g_ListenerStack[ListenerStackSize];
char g_StringBuffer[CharCountMax];

void InputListener(void* arg) NN_NOEXCEPT
{
    ConfigSwitcher* pSwitcher = reinterpret_cast<ConfigSwitcher*>(arg);
    while (true)
    {
        pSwitcher->DoListeningProcedure();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

}

NN_IMPLICIT ConfigSwitcher::ConfigSwitcher(DisplayConfig* pConfig, bool isVisible) NN_NOEXCEPT
:
m_ConfigNumber(pConfig->GetConfigCount()),
m_IsVisible(isVisible),
m_IsFontColorDarkToLight(false),
m_PreTouchIndex(-1),
m_SelectedIndex(0),
m_KeepingTouchCount(0),
m_LastTouchX(0),
m_LastTouchY(0),
m_ImageWidthRatio(0.0f),
m_ImageHeightRatio(0.0f),
m_Pixels(nullptr),
m_pDebugFontHeap(nullptr),
m_pDisplayConfig(pConfig),
m_FontColor({{FontColorDark, FontColorDark, FontColorDark, 0xff}}),
m_SelectedFontColor({{FontColorLight, FontColorLight, FontColorLight, 0xff}})
{
    m_DebugPadPressCount.select = 0;
    m_DebugPadPressCount.left = 0;
    m_DebugPadPressCount.right = 0;
    m_BasicNpadPressCount.select = 0;
    m_BasicNpadPressCount.left = 0;
    m_BasicNpadPressCount.right = 0;
    nn::hid::InitializeTouchScreen();
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(
        nn::hid::NpadStyleFullKey::Mask |
        nn::hid::NpadStyleJoyDual::Mask |
        nn::hid::NpadStyleHandheld::Mask);
    // 使用する Npad を設定
    m_NpadIds[0] = nn::hid::NpadId::Handheld;
    m_NpadIds[1] = nn::hid::NpadId::No1;
    m_NpadIds[2] = nn::hid::NpadId::No2;
    nn::hid::SetSupportedNpadIdType(m_NpadIds, NpadIdCountMax);
}

ConfigSwitcher::~ConfigSwitcher() NN_NOEXCEPT
{
}

void ConfigSwitcher::Initialize(nn::gfx::Device* pDevice, const nn::vi::Display* pDisplay,
                                nn::gfx::DescriptorPool* pTextureDescriptorPool,
                                nn::gfx::DescriptorPool* pSamplerDescriptorPool,
                                int textureDescriptorIndexSlot, int samplerDescriptorIndexSlot) NN_NOEXCEPT
{
    nn::gfx::util::DebugFontTextWriterInfo info;
    info.SetDefault();
    info.SetCharCountMax(CharCountMax);
    info.SetUserMemoryPoolEnabled(false);

    size_t debugFontHeapSize = nn::gfx::util::DebugFontTextWriter::GetRequiredMemorySize(pDevice, info);
    m_pDebugFontHeap = new nn::util::BytePtr(new uint8_t[debugFontHeapSize]);

    m_TextWriter.Initialize(
        pDevice,
        info,
        m_pDebugFontHeap->Get(),
        debugFontHeapSize,
        nullptr,
        0,
        0
        );

    m_TextWriter.SetDisplayWidth(LcdWidth);
    m_TextWriter.SetDisplayHeight(LcdHeight);
    m_TextWriter.SetTextureDescriptor(pTextureDescriptorPool, textureDescriptorIndexSlot - 1);
    m_TextWriter.SetSamplerDescriptor(pSamplerDescriptorPool, samplerDescriptorIndexSlot - 1);
}

void ConfigSwitcher::Finalize() NN_NOEXCEPT
{
    m_TextWriter.Finalize();
    delete[] reinterpret_cast<uint8_t*>(m_pDebugFontHeap->Get());
    m_pDebugFontHeap->Reset(nullptr);
    delete m_pDebugFontHeap;
}

void ConfigSwitcher::Draw(nn::gfx::CommandBuffer* pCommandBuffer) NN_NOEXCEPT
{
    m_pDisplayConfig->UpdateConfig();
    UpdateFontColor();
    PutConfigList();
    m_TextWriter.Draw(pCommandBuffer);
}

void ConfigSwitcher::CreateInputListener() NN_NOEXCEPT
{
    int priority = nn::os::GetThreadCurrentPriority(nn::os::GetCurrentThread());
    auto result = nn::os::CreateThread(&m_ListenerThread, &nns::InputListener,
        reinterpret_cast<void*>(this), g_ListenerStack, sizeof(g_ListenerStack), priority);
    NN_ASSERT(result.IsSuccess());
    nn::os::StartThread(&m_ListenerThread);
}

void ConfigSwitcher::DoListeningProcedure() NN_NOEXCEPT
{
    ChangeConfigViaTouchPanel();
    ChangeConfigViaDebugPad();
    ChangeConfigViaBasicNpad();
}

bool ConfigSwitcher::IsSceneSwitchedOn() NN_NOEXCEPT
{
    return m_pDisplayConfig->IsSceneSwitched() ||
        (SceneSwitchKeepingTouchBoundary < m_KeepingTouchCount) ||
        (SceneSwitchKeepingPressBoundary < m_DebugPadPressCount.select) ||
        (SceneSwitchKeepingPressBoundary < m_BasicNpadPressCount.select);
}

void ConfigSwitcher::UpdateFontColor() NN_NOEXCEPT
{
    uint8_t variation = m_IsFontColorDarkToLight ? 1 : -1;
    uint8_t component = (m_SelectedFontColor.v[1] + variation);
    m_SelectedFontColor.v[1] = m_SelectedFontColor.v[2] = component;
    // フォントの明るさの境界にきたら変化の方向を反転させる。
    if (m_SelectedFontColor.v[2] <= FontColorDark || FontColorLight <= m_SelectedFontColor.v[2])
    {
        m_IsFontColorDarkToLight = !m_IsFontColorDarkToLight;
    }
}

void ConfigSwitcher::PutConfigList() NN_NOEXCEPT
{
    if (m_IsVisible)
    {
        m_TextWriter.SetTabWidth(2);
        m_TextWriter.SetCursorY(CaptionTop);
        // 操作できるコンフィグの種類をリストアップ。
        for (int i1=0; i1<m_ConfigNumber; ++i1)
        {
            m_TextWriter.SetCursorX(CaptionLeft);
            m_TextWriter.SetTextColor((m_SelectedIndex == i1) ? m_SelectedFontColor : m_FontColor);
            m_TextWriter.SetFontSize((m_SelectedIndex == i1) ? FontSizeLarge : FontSizeDefault);
            m_TextWriter.Print(m_pDisplayConfig->GetConfigCaption(i1));
            m_TextWriter.SetCursorX(CaptionLeft + LableLeft);
            m_TextWriter.Print(m_pDisplayConfig->GetConfigLabel(i1));
            const char* statusString = m_pDisplayConfig->GetConfigStatus(i1);
            if (std::strncmp(statusString, "", CharCountMax))
            {
                m_TextWriter.Print(statusString);
            }
            m_TextWriter.Print("\n");
        }

        // 触れた場所をマーキング。
        m_TextWriter.SetTextColor(MarkColor);
        m_TextWriter.SetFontSize(MarkSize);
        // "o" というテキストが少し沈むので少し上に持っていく。
        m_TextWriter.SetCursor(m_LastTouchX, std::max(0.0f, m_LastTouchY - 48.0f));
        m_TextWriter.Print("o");

        // 触れた場所の座標と色を表示。
        if (nullptr != m_Pixels)
        {
            m_TextWriter.SetFontSize(TouchInfoFontSize);
            m_TextWriter.SetCursor(0, LcdHeight - TouchInfoFontSize);
            float floatWidth = m_LastTouchX * m_ImageWidthRatio;
            float floatHeight = (LcdHeight - m_LastTouchY) * m_ImageHeightRatio;
            int pixelIndex = static_cast<int>(floatWidth + (floatHeight * m_ImageWidthRatio * LcdWidth)) * 4;
            std::snprintf(g_StringBuffer, CharCountMax, " (%4d, %4d):[%3d, %3d, %3d]",
                m_LastTouchX, m_LastTouchY, m_Pixels[pixelIndex + 0], m_Pixels[pixelIndex + 1], m_Pixels[pixelIndex + 2]);
            m_TextWriter.Print(g_StringBuffer);
        }
    }
    else {
        // 背景色を変えるためにコンフィグを読む。
        for (int i1 = 0; i1 < m_ConfigNumber; ++i1)
        {
            m_pDisplayConfig->GetConfigStatus(i1);
        }
    }
}

int ConfigSwitcher::GetTouchedConfigIndex(const nn::hid::TouchState& state) NN_NOEXCEPT
{
    if (0 == state.x && 0 == state.y)
    {
        // 何もタッチされていない
        return -1;
    }
    // "o" でマーキングするので、Y座標を少しずらす。
    m_LastTouchX = state.x;
    m_LastTouchY = state.y;

    int left = CaptionLeft;
    int top = CaptionTop;
    //NN_LOG("[%d] %s() (%d,%d)\n", __LINE__, __FUNCTION__, state.x, state.y);
    for (int i1=0; i1<m_ConfigNumber; ++i1)
    {
        const float fontSize = (m_SelectedIndex == i1) ? FontSizeLarge : FontSizeDefault;
        const char* label = m_pDisplayConfig->GetConfigLabel(i1);
        int labelLength = strnlen(label, CharCountMax);
        int breakCount = 1;
        for (int i1=0; i1<labelLength; ++i1)
        {
            if ('\n' == label[i1])
            {
                ++breakCount;
            }
        }

        int right = fontSize * FontWidthRaito * labelLength + left + LableLeft;
        int bottom = fontSize * FontHeightRaito * breakCount + top;
        if (left <= state.x && state.x < right && top <= state.y && state.y < bottom)
        {
            // 対象のコンフィグがタッチされている。
            return i1;
        }
        top = bottom;
    }
    // 対象外がタッチされている。
    return m_ConfigNumber;
}

void ConfigSwitcher::ChangeConfigViaTouchPanel() NN_NOEXCEPT
{
    nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> g_TouchState;
    nn::hid::GetTouchScreenState(&g_TouchState);
    // 最後にタッチされている点しか考慮しない。
    int index = GetTouchedConfigIndex(g_TouchState.touches[0]);
    bool isTouched = (0 <= index);
    bool isPreTouched = (0 <= m_PreTouchIndex);
    int releasedTouchIndex = (!isTouched && isPreTouched) ? m_PreTouchIndex : -1;
    m_PreTouchIndex = index;
    // 長押し検知用のカウント。
    m_KeepingTouchCount = (isTouched && isPreTouched) ? (m_KeepingTouchCount + 1) : 0;

    if (m_IsVisible)
    {
        if (0 <= releasedTouchIndex)
        {
            if (m_SelectedIndex == releasedTouchIndex)
            {
                // 選択されているコンフィグのリストにタッチされたら水準を操作する。
                m_pDisplayConfig->SetConfig(releasedTouchIndex, true);
            }
            else
            {
                if (m_ConfigNumber == releasedTouchIndex)
                {
                    // コンフィグのリスト以外を触られたら非表示に切り替える。
                    m_IsVisible = false;
                }
                else
                {
                    // コンフィグのリストの選択を切り替える。
                    m_SelectedIndex = releasedTouchIndex;
                }
            }
        }
    }
    else
    {
        // コンフィグのリストが表示されていないときは、表示の切り替えのみ反応させる。
        m_IsVisible = (0 <= releasedTouchIndex);
    }
}

void ConfigSwitcher::ChangeConfigViaDebugPad() NN_NOEXCEPT
{
    nn::hid::DebugPadState debugPadState[nn::hid::DebugPadStateCountMax];
    // Pad の入力を取得します。
    nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

    // Pad の状態を割り出します。
    nn::hid::DebugPadButtonSet debugPadButtonDown(debugPadState[0].buttons & ~m_DebugPadState.buttons);
    nn::hid::DebugPadButtonSet debugPadButtonHold(debugPadState[0].buttons);
    m_DebugPadState = debugPadState[0];

    if (m_IsVisible)
    {
        // 非表示に切り替える。
        m_IsVisible = !debugPadButtonDown.Test<nn::hid::DebugPadButton::Start>();
        // 上下のボタンで設定するコンフィグを選択。
        if (debugPadButtonDown.Test<nn::hid::DebugPadButton::Up>())
        {
            m_SelectedIndex = (m_SelectedIndex <= 0) ? m_ConfigNumber - 1 : m_SelectedIndex - 1;
        }
        else if (debugPadButtonDown.Test<nn::hid::DebugPadButton::Down>())
        {
            m_SelectedIndex = (m_SelectedIndex + 1) % m_ConfigNumber;
        }
        // 設定を変更。
        if (debugPadButtonDown.Test<nn::hid::DebugPadButton::A>() ||
            debugPadButtonDown.Test<nn::hid::DebugPadButton::Right>() ||
            m_DebugPadPressCount.right >= InputKeepingBoundary)
        {
            m_pDisplayConfig->SetConfig(m_SelectedIndex, true);
        }
        else if (debugPadButtonDown.Test<nn::hid::DebugPadButton::B>() ||
            debugPadButtonDown.Test<nn::hid::DebugPadButton::Left>() ||
            m_DebugPadPressCount.left >= InputKeepingBoundary)
        {
            m_pDisplayConfig->SetConfig(m_SelectedIndex, false);
        }
    }
    else
    {
        // コンフィグのリストが表示されていないときは、表示の切り替えのみ反応させる。
        m_IsVisible = debugPadButtonDown.Test<nn::hid::DebugPadButton::Start>();
    }

    // 長押し検知用のカウント。
    m_DebugPadPressCount.select = debugPadButtonHold.Test<nn::hid::DebugPadButton::Select>() ? (m_DebugPadPressCount.select + 1) : 0;
    m_DebugPadPressCount.left = debugPadButtonHold.Test<nn::hid::DebugPadButton::Left>() ? (m_DebugPadPressCount.left + 1) : 0;
    m_DebugPadPressCount.right = debugPadButtonHold.Test<nn::hid::DebugPadButton::Right>() ? (m_DebugPadPressCount.right + 1) : 0;
}

void ConfigSwitcher::ChangeConfigViaBasicNpad() NN_NOEXCEPT
{
    // いずれのコントローラー(携帯、接続先1、接続先2)からも入力できる。
    nn::hid::NpadButtonSet basicNpadButtonDown;
    nn::hid::NpadButtonSet basicNpadButtonHold;
    bool isPairing = false;
    for (int i1 = 0; i1 < NpadIdCountMax; ++i1)
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadIds[i1]);
        if (style.Test<nn::hid::NpadStyleFullKey>())
        {
            nn::hid::NpadFullKeyState fullKeyState;
            nn::hid::GetNpadState(&fullKeyState, m_NpadIds[i1]);
            if (!isPairing || fullKeyState.buttons.IsAnyOn())
            {
                basicNpadButtonDown = fullKeyState.buttons & ~m_FullKeyState.buttons;
                basicNpadButtonHold = fullKeyState.buttons;
            }
            m_FullKeyState = fullKeyState;
            isPairing = true;
        }
        else if (style.Test<nn::hid::NpadStyleJoyDual>())
        {
            nn::hid::NpadJoyDualState joyDualState;
            nn::hid::GetNpadState(&joyDualState, m_NpadIds[i1]);
            if (!isPairing || joyDualState.buttons.IsAnyOn())
            {
                basicNpadButtonDown = joyDualState.buttons & ~m_JoyDualState.buttons;
                basicNpadButtonHold = joyDualState.buttons;
            }
            m_JoyDualState = joyDualState;
            isPairing = true;
        }
        else if (style.Test<nn::hid::NpadStyleHandheld>())
        {
            nn::hid::NpadHandheldState handheldState;
            nn::hid::GetNpadState(&handheldState, m_NpadIds[i1]);
            if (!isPairing || handheldState.buttons.IsAnyOn())
            {
                basicNpadButtonDown = handheldState.buttons & ~m_HandheldState.buttons;
                basicNpadButtonHold = handheldState.buttons;
            }
            m_HandheldState = handheldState;
            isPairing = true;
        }
    }
    if (!isPairing)
    {
        // 何もコントローラーが検出されなかった
        return;
    }

    if (m_IsVisible)
    {
        // 非表示に切り替える。
        m_IsVisible = !basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Plus>();
        // 上下のボタンで設定するコンフィグを選択。
        if (basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Up>())
        {
            m_SelectedIndex = (m_SelectedIndex <= 0) ? m_ConfigNumber - 1 : m_SelectedIndex - 1;
        }
        else if (basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Down>())
        {
            m_SelectedIndex = (m_SelectedIndex + 1) % m_ConfigNumber;
        }
        // 設定を変更。
        if (basicNpadButtonDown.Test<nn::hid::NpadJoyButton::A>() ||
            basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Right>() ||
            m_BasicNpadPressCount.right >= InputKeepingBoundary)
        {
            m_pDisplayConfig->SetConfig(m_SelectedIndex, true);
        }
        else if (basicNpadButtonDown.Test<nn::hid::NpadJoyButton::B>() ||
            basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Left>() ||
            m_BasicNpadPressCount.left >= InputKeepingBoundary)
        {
            m_pDisplayConfig->SetConfig(m_SelectedIndex, false);
        }
    }
    else
    {
        // コンフィグのリストが表示されていないときは、表示の切り替えのみ反応させる。
        m_IsVisible = basicNpadButtonDown.Test<nn::hid::NpadJoyButton::Plus>();
    }

    // 長押し検知用のカウント。
    m_BasicNpadPressCount.select = basicNpadButtonHold.Test<nn::hid::NpadJoyButton::Minus>() ? (m_BasicNpadPressCount.select + 1) : 0;
    m_BasicNpadPressCount.left = basicNpadButtonHold.Test<nn::hid::NpadJoyButton::Left>() ? (m_BasicNpadPressCount.left + 1) : 0;
    m_BasicNpadPressCount.right = basicNpadButtonHold.Test<nn::hid::NpadJoyButton::Right>() ? (m_BasicNpadPressCount.right + 1) : 0;
}

void ConfigSwitcher::ReferImagePixel(const uint8_t* pixels, int imageWidth, int imageHeight) NN_NOEXCEPT
{
    m_Pixels = nullptr;
    m_ImageWidthRatio = static_cast<float>(imageWidth) / LcdWidth;
    m_ImageHeightRatio = static_cast<float>(imageHeight) / LcdHeight;
    m_Pixels = pixels;
}

const Color& ConfigSwitcher::GetBackGroundColorFromDisplayConfig() NN_NOEXCEPT
{
    return m_pDisplayConfig->GetBackGroundColor();
}

}
