﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nnt/viUtil/testVi_EdidFixture.h>
#include <nn/vi/vi_Result.h>
#include "testDisplay_DisplayFactoryFixture.h"
#include "master/detail/visrv_PlatformDisplayInfo.h"
#include "master/detail/visrv_DisplayFactory.h"
#include "master/detail/visrv_DefaultDisplay-spec.NX.h"
#include "settings/visrv_Settings.h"
#include "settings/visrv_PlatformConfig.h"
#include "settings/visrv_Strategy.h"
#include "settings/visrv_InterfaceType.h"

namespace
{
    typedef DisplayFactoryFixture DisplayFactory;

    const size_t FactoryMemorySize = 1024;
}

TEST_F(DisplayFactory, CreateDefaultDisplay_HighestPriority_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.defaultDisplayStrategy = nn::visrv::settings::Strategy::HighestPriority;
    config.dcList[0].interface = nn::visrv::settings::InterfaceType::Dsi;
    config.dcList[0].priority  = 0;
    config.dcList[1].interface = nn::visrv::settings::InterfaceType::Null;
    config.dcList[1].priority  = 5;

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Default",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Default");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());
    pDisplay->Open();

    // if we're using the null display, this should fail
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, pDisplay->SetCmuMode(nn::vi::CmuMode_Default));

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, CreateDefaultDisplay_HighestPriority_NullDefault_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.defaultDisplayStrategy = nn::visrv::settings::Strategy::HighestPriority;

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Default",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Default");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());
    pDisplay->Open();

    // if we're using the null display, this should fail
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, pDisplay->SetCmuMode(nn::vi::CmuMode_Default));

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, CreateDefaultDisplay_PriorityFallback_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.defaultDisplayStrategy = nn::visrv::settings::Strategy::PriorityFallback;
    config.dcList[0].interface = nn::visrv::settings::InterfaceType::Dsi;
    config.dcList[0].priority  = 0;
    config.dcList[1].interface = nn::visrv::settings::InterfaceType::Null;
    config.dcList[1].priority  = 5;

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Default",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Default");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());
    pDisplay->Open();

    // if we're using the null display, this should fail
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, pDisplay->SetCmuMode(nn::vi::CmuMode_Default));

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, CreateDefaultDisplay_PriorityFallback_NullDefault_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.defaultDisplayStrategy = nn::visrv::settings::Strategy::PriorityFallback;

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Default",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Default");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());
    pDisplay->Open();

    // if we're using the null display, this should fail
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, pDisplay->SetCmuMode(nn::vi::CmuMode_Default));

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, CreateNullDisplay_Any)
{
    nn::visrv::settings::PlatformConfig config;
    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Null",
                                                        0,
                                                        0,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Null");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());

    // Without RTTI, no way to check the created type is the expected type

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, CreateUnknownDisplay_Any)
{
    nn::visrv::settings::PlatformConfig config;
    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("nope",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "nope");
    ASSERT_EQ(nullptr, pDisplay);
}

TEST_F(DisplayFactory, Create_NoEdid_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.dcList[0].interface = nn::visrv::settings::InterfaceType::Null;
    config.dcList[0].priority  = 0;
    config.physicalDisplayCount = 1;
    config.displays[0].dc = 0;
    std::strncpy(config.displays[0].name, "Test", sizeof(config.displays[0].name));

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "Test");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());
    pDisplay->Open();

    // if we're using the null display, this should fail
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, pDisplay->SetCmuMode(nn::vi::CmuMode_Default));

    factory.Destroy(pDisplay);
}

TEST_F(DisplayFactory, Create_Edid_Any)
{
    nn::visrv::settings::PlatformConfig config;
    config.dcList[0].interface = nn::visrv::settings::InterfaceType::Null;
    config.dcList[0].priority  = 0;
    config.physicalDisplayCount = 1;
    config.displays[0].dc = 0;
    std::strncpy(config.displays[0].name, "Test", sizeof(config.displays[0].name));
    std::strncpy(config.displays[0].edidDisplayName, "TestEdid", sizeof(config.displays[0].edidDisplayName));

    char displayMemory[FactoryMemorySize];
    nn::visrv::master::detail::DisplayFactory factory(displayMemory, sizeof(displayMemory), &config);

    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                      0,
                                                                                      0,
                                                                                      0),
                                                        nn::visrv::master::detail::DisplaySupport{ nullptr },
                                                        false);

    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);
    nn::visrv::master::Display* pDisplay = factory.Create(infoSet, GetAllocator(), "TestEdid");
    ASSERT_NE(nullptr, pDisplay);
    EXPECT_FALSE(pDisplay->IsOpen());

    factory.Destroy(pDisplay);
}
