﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/vi/vi_Result.h>
#include <nnt/result/testResult_Assert.h>
#include "master/visrv_Display.h"
#include "testDisplay_AutoscaleFixture.h"
#include "testDisplay_MockFilter.h"
#include "testDisplay_MockPhysicalDisplay.h"
#include "master/detail/visrv_SingleDisplayStrategy.h"

namespace
{
    class DisplayFixture : public ::testing::Test
    {
    public:
        DisplayFixture() NN_NOEXCEPT
            : m_Allocator(m_AllocatorMem, sizeof(m_AllocatorMem), alignof(nn::visrv::master::Layer))
            , m_Strategy(&m_PhysicalDisplay)
            , m_Display(m_InfoSet, &m_Allocator, &m_Strategy)
        {
        }

        nn::visrv::master::Display* GetDisplay() NN_NOEXCEPT
        {
            return &m_Display;
        }

        MockPhysicalDisplay* GetPhysicalDisplay() NN_NOEXCEPT
        {
            return &m_PhysicalDisplay;
        }
    private:
        class InfoSet
        {
        public:
            InfoSet() NN_NOEXCEPT
            {
                m_Set.SetInfo(nn::vi::PolicyLevel_Standard,
                              nn::visrv::master::detail::PlatformDisplayInfo
                              (
                                  "Test",
                                  0,
                                  0,
                                  nn::visrv::master::detail::DisplayPolicy
                                  (
                                      false,
                                      0,
                                      0,
                                      0
                                  ),
                                  nn::visrv::master::detail::DisplaySupport(),
                                  false
                              ));
            }

            NN_IMPLICIT operator nn::visrv::master::detail::PlatformDisplayInfoSet() NN_NOEXCEPT
            {
                return m_Set;
            }
        private:
            nn::visrv::master::detail::PlatformDisplayInfoSet m_Set;
        };

        InfoSet m_InfoSet;

        char m_AllocatorMem[sizeof(nn::visrv::master::Layer) * 2];
        nn::visrv::master::Display::Allocator m_Allocator;

        MockPhysicalDisplay m_PhysicalDisplay;
        nn::visrv::master::detail::SingleDisplayStrategy m_Strategy;

        nn::visrv::master::Display m_Display;
    };
}

typedef DisplayFixture Display;
typedef AutoscaleFixture Autoscale;

const int AutoscaleDisplayWidth  = 1920;
const int AutoscaleDisplayHeight = 1080;

namespace nn { namespace visrv { namespace master { namespace detail {
    bool operator==(const DisplayPolicy& lhs, const DisplayPolicy& rhs) NN_NOEXCEPT
    {
        return lhs.HasLayerLimit() == rhs.HasLayerLimit() &&
               lhs.GetLayerCountMax() == rhs.GetLayerCountMax() &&
               lhs.GetZOrderCountMin() == rhs.GetZOrderCountMin() &&
               lhs.GetZOrderCountMax() == rhs.GetZOrderCountMax();
    }

    bool operator==(const DisplaySupport& lhs, const DisplaySupport& rhs) NN_NOEXCEPT
    {
        return lhs.queryScalingModeSupport == rhs.queryScalingModeSupport;
    }

    bool operator==(const PlatformDisplayInfo& lhs, const PlatformDisplayInfo& rhs) NN_NOEXCEPT
    {
        return !strcmp(lhs.GetName(), rhs.GetName()) &&
               lhs.GetLayerWidthPixelCountMax() == rhs.GetLayerWidthPixelCountMax() &&
               lhs.GetLayerHeightPixelCountMax() == rhs.GetLayerHeightPixelCountMax() &&
               lhs.GetPolicy() == rhs.GetPolicy() &&
               lhs.GetSupport() == rhs.GetSupport();
    }
}}}}

namespace nn { namespace vi {
    bool operator==(const DisplayModeInfo& lhs, const DisplayModeInfo& rhs) NN_NOEXCEPT
    {
        return lhs.width == rhs.width &&
               lhs.height == rhs.height &&
               lhs.refreshRate == rhs.refreshRate &&
               lhs.mode == rhs.mode;
    }
}}

TEST_F(Display, Ctor_Any)
{
    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        0,
                                                        0,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                                 0,
                                                                                                 0,
                                                                                                 0),
                                                        nn::visrv::master::detail::DisplaySupport(),
                                                        false);
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);

    MockPhysicalDisplay physicalDisplay;
    nn::visrv::master::detail::SingleDisplayStrategy strategy(&physicalDisplay);

    char allocatorMem[sizeof(nn::visrv::master::Layer)];
    nn::visrv::master::Display::Allocator allocator(allocatorMem, sizeof(allocatorMem), alignof(nn::visrv::master::Layer));
    nn::visrv::master::Display display(infoSet, &allocator, &strategy);

    ASSERT_FALSE(display.IsOpen());
}

TEST_F(Display, Open_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->Open());
    ASSERT_TRUE(GetDisplay()->IsOpen());
}

TEST_F(Display, Close_Any)
{
    nn::visrv::master::Display* pDisplay = GetDisplay();
    NNT_ASSERT_RESULT_SUCCESS(pDisplay->Open());
    pDisplay->Close();
    ASSERT_FALSE(pDisplay->IsOpen());
}

TEST_F(Display, GetInfo_Any)
{
    nn::visrv::master::detail::DisplaySupport support = { nullptr };
    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        0,
                                                        0,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                        0,
                                                                                        0,
                                                                                        0),
                                                        support,
                                                        false);
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);

    MockPhysicalDisplay physicalDisplay;
    nn::visrv::master::detail::SingleDisplayStrategy strategy(&physicalDisplay);

    char allocatorMem[sizeof(nn::visrv::master::Layer)];
    nn::visrv::master::Display::Allocator allocator(allocatorMem, sizeof(allocatorMem), alignof(nn::visrv::master::Layer));
    nn::visrv::master::Display display(infoSet, &allocator, &strategy);

    ASSERT_EQ(info, display.GetInfo(nn::vi::PolicyLevel_Standard));
}

TEST_P(Autoscale, Enabled_Any)
{
    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        1280,
                                                        720,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                        0,
                                                                                        0,
                                                                                        0),
                                                        GetSupport(),
                                                        false);
    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);

    MockPhysicalDisplay lcd;
    nn::vi::DisplayModeInfo mode{ 1920, 1080, 60.00f, nn::vi::StereoMode_None };
    lcd.SetMode(mode);
    nn::visrv::master::detail::SingleDisplayStrategy strategy(&lcd);
    char allocatorMem[sizeof(nn::visrv::master::Layer)];
    nn::visrv::master::Display::Allocator allocator(allocatorMem, sizeof(allocatorMem), alignof(nn::visrv::master::Layer));
    nn::visrv::master::Display display(infoSet, &allocator, &strategy);

    nn::visrv::master::Layer* pLayer;

    nn::vi::LayerSettings settings;
    settings.Reset();
    settings.Set<nn::vi::LayerFlags::Fullscreen>();

    NNT_ASSERT_RESULT_SUCCESS(display.CreateLayer(&pLayer, settings, nn::vi::PolicyLevel_Standard, nn::applet::GetAppletResourceUserId()));
    EXPECT_NE(nullptr, pLayer);
    EXPECT_EQ(MockPhysicalDisplay::DisplayLogicalWidth, pLayer->GetCurrentWidth());
    EXPECT_EQ(MockPhysicalDisplay::DisplayLogicalHeight, pLayer->GetCurrentHeight());

    // make sure this doesn't crash
    display.DestroyLayer(pLayer);
}

TEST_P(Autoscale, Disabled_Any)
{
    nn::visrv::master::detail::PlatformDisplayInfo info("Test",
                                                        1920,
                                                        1080,
                                                        nn::visrv::master::detail::DisplayPolicy(false,
                                                                                        0,
                                                                                        0,
                                                                                        0),
                                                        GetSupport(),
                                                        false);
    nn::visrv::master::detail::PlatformDisplayInfoSet infoSet;
    infoSet.SetInfo(nn::vi::PolicyLevel_Standard, info);

    MockPhysicalDisplay lcd;
    nn::visrv::master::detail::SingleDisplayStrategy strategy(&lcd);
    char allocatorMem[sizeof(nn::visrv::master::Layer)];
    nn::visrv::master::Display::Allocator allocator(allocatorMem, sizeof(allocatorMem), alignof(nn::visrv::master::Layer));
    nn::visrv::master::Display display(infoSet, &allocator, &strategy);

    nn::vi::DisplayModeInfo mode{AutoscaleDisplayWidth, AutoscaleDisplayHeight, 60.f, nn::vi::StereoMode_None};
    // overload is not resolving properly...
    static_cast<nn::visrv::master::Display*>(&display)->SetMode(mode, nn::vi::PolicyLevel_Standard);

    int layerWidth = GetParam().width;
    int layerHeight = GetParam().height;

    nn::visrv::master::Layer* pLayer;
    NNT_ASSERT_RESULT_SUCCESS(display.CreateLayer(&pLayer, nn::vi::LayerSettings().Reset(), nn::vi::PolicyLevel_Standard, nn::applet::GetAppletResourceUserId()));
    EXPECT_NE(nullptr, pLayer);

    // really degenerate cases could yield dimensions of zero
    // Ex. 1920x1 layer dimensions on 720p display
    if( pLayer->GetCurrentHeight() > 0 )
    {
        // odd case where autoscaling is disabled, but layer is too large for the display
        // scaling is applied, but aspect ratio is maintained
        EXPECT_NEAR(static_cast<float>(layerWidth) / layerHeight, static_cast<float>(pLayer->GetCurrentWidth()) / pLayer->GetCurrentHeight(), .01f);
    }

    EXPECT_GE(pLayer->GetCurrentWidth(), 0);
    EXPECT_GE(pLayer->GetCurrentHeight(), 0);

    EXPECT_LE(pLayer->GetCurrentWidth(), AutoscaleDisplayWidth);
    EXPECT_LE(pLayer->GetCurrentHeight(), AutoscaleDisplayHeight);

    display.DestroyLayer(pLayer);
}

INSTANTIATE_TEST_CASE_P(AutoscaleBoundaries,
                        Autoscale,
                        ::testing::Values(LayerSize{ AutoscaleDisplayWidth, AutoscaleDisplayHeight },
                                          LayerSize{ AutoscaleDisplayWidth + 1, AutoscaleDisplayHeight },
                                          LayerSize{ AutoscaleDisplayWidth, AutoscaleDisplayHeight + 1 },
                                          LayerSize{ AutoscaleDisplayWidth - 1, AutoscaleDisplayHeight },
                                          LayerSize{ AutoscaleDisplayWidth, AutoscaleDisplayHeight - 1 },
                                          LayerSize{ AutoscaleDisplayWidth, 1},
                                          LayerSize{ 1, AutoscaleDisplayHeight },
                                          LayerSize{ 1920, 1},
                                          LayerSize{ 1, 1080}));

TEST_F(Display, GetHotplugState_Any)
{
    nn::vi::HotplugState states[] =
    {
        nn::vi::HotplugState_Connected,
        nn::vi::HotplugState_Disconnected,
    };

    for( int i = 0; i < sizeof(states) / sizeof(states[0]); ++i )
    {
        GetPhysicalDisplay()->SetHotplugState(states[i]);

        nn::vi::HotplugStateType current;
        NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetHotplugState(&current));
        EXPECT_EQ(states[i], current);
    }
}

TEST_F(Display, GetMode_Any)
{
    nn::vi::DisplayModeInfo mode{ 1920, 1080, 60.f, nn::vi::StereoMode_None };
    GetPhysicalDisplay()->SetMode(mode);

    nn::vi::DisplayModeInfo current;
    std::memset(&current, 0, sizeof(current));

    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetMode(&current));
    EXPECT_EQ(mode, current);
}

TEST_F(Display, ListRgbRanges_Any)
{
    nn::vi::RgbRange ranges[] =
    {
        nn::vi::RgbRange_Limited,
        nn::vi::RgbRange_Full,
    };

    const int ResultSize = sizeof(ranges) / sizeof(ranges[0]);

    GetPhysicalDisplay()->SetValidRgbRanges(ranges, ResultSize);

    nn::vi::RgbRangeType current[nn::vi::RgbRange_Max];
    ASSERT_EQ(ResultSize, GetDisplay()->ListRgbRanges(current, sizeof(current) / sizeof(current[0])));

    for( int i = 0; i < ResultSize; ++i )
    {
        EXPECT_EQ(ranges[i], current[i]);
    }
}

TEST_F(Display, GetRgbRange_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetRgbRange(nn::vi::RgbRange_Full));
    nn::vi::RgbRangeType current;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetRgbRange(&current));
    EXPECT_EQ(nn::vi::RgbRange_Full, current);
}

TEST_F(Display, SetRgbRange_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetRgbRange(nn::vi::RgbRange_Full));
    nn::vi::RgbRangeType current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetRgbRange(&current));
    EXPECT_EQ(nn::vi::RgbRange_Full, current);
}

TEST_F(Display, SetUnderscan_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetUnderscan(20));
    int current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetUnderscan(&current));
    EXPECT_EQ(20, current);
}

TEST_F(Display, GetUnderscan_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetUnderscan(20));
    int current;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetUnderscan(&current));
    EXPECT_EQ(20, current);
}

TEST_F(Display, SetAlpha_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetAlpha(50.f));
    EXPECT_EQ(50.f, GetPhysicalDisplay()->GetAlpha());
}

TEST_F(Display, SetPowerState_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetPowerState(nn::vi::PowerState_On));
    nn::vi::PowerStateType current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetPowerState(&current));
    EXPECT_EQ(nn::vi::PowerState_On, current);
}

TEST_F(Display, SetLayerStack_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetLayerStack(nn::vi::LayerStack_Null));
    nn::vi::LayerStackType stack;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetLayerStack(&stack));
    EXPECT_EQ(nn::vi::LayerStack_Null, stack);
}

TEST_F(Display, SetCmuLuma_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetCmuLuma(20.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetCmuLuma(&current));
    EXPECT_EQ(20.f, current);
}

TEST_F(Display, GetCmuLuma_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetCmuLuma(20.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetCmuLuma(&current));
    EXPECT_EQ(20.f, current);
}

TEST_F(Display, IsHotplugEventSupported_Any)
{
    GetPhysicalDisplay()->SetIsHotplugEventSupported(true);
    EXPECT_TRUE(GetDisplay()->IsHotplugEventSupported());
}

TEST_F(Display, IsVsyncEventSupported_Any)
{
    GetPhysicalDisplay()->SetIsVsyncEventSupported(true);
    EXPECT_TRUE(GetDisplay()->IsVsyncEventSupported());
}

TEST_F(Display, GetLogicalResolution_Any)
{
    GetPhysicalDisplay()->SetLogicalResolution(720, 480);
    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLogicalResolution(&width, &height));
    EXPECT_EQ(720, width);
    EXPECT_EQ(480, height);
}

TEST_F(Display, GetCmuMode_Any)
{
    nn::vi::CmuMode modes[] =
    {
        nn::vi::CmuMode_Disabled,
        nn::vi::CmuMode_Default,
        nn::vi::CmuMode_Grayscale,
        nn::vi::CmuMode_HighContrast,
        nn::vi::CmuMode_InvertColor,
    };

    for( int i = 0; i < sizeof(modes) / sizeof(modes[0]); ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetCmuMode(modes[i]));
        nn::vi::CmuModeType current;
        NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetCmuMode(&current));
        EXPECT_EQ(modes[i], current);
    }
}

TEST_F(Display, SetCmuMode_Any)
{
    nn::vi::CmuMode modes[] =
    {
        nn::vi::CmuMode_Disabled,
        nn::vi::CmuMode_Default,
        nn::vi::CmuMode_Grayscale,
        nn::vi::CmuMode_HighContrast,
        nn::vi::CmuMode_InvertColor,
    };

    for( int i = 0; i < sizeof(modes) / sizeof(modes[0]); ++i )
    {
        NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetCmuMode(modes[i]));
        nn::vi::CmuModeType current;
        NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetCmuMode(&current));
        EXPECT_EQ(modes[i], current);
    }
}

TEST_F(Display, GetContrastRatio_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetContrastRatio(1.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetContrastRatio(&current));
    EXPECT_EQ(1.f, current);
}

TEST_F(Display, SetContrastRatio_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetContrastRatio(1.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetContrastRatio(&current));
    EXPECT_EQ(1.f, current);
}

TEST_F(Display, GetGamma_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetGamma(3.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetGamma(&current));
    EXPECT_EQ(3.f, current);
}

TEST_F(Display, SetGamma_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->SetGamma(3.f));
    float current;
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->GetGamma(&current));
    EXPECT_EQ(3.f, current);
}

TEST_F(Display, GetLayerStack_Any)
{
    NNT_ASSERT_RESULT_SUCCESS(GetPhysicalDisplay()->SetLayerStack(nn::vi::LayerStack_Null));
    nn::vi::LayerStackType stack;
    NNT_ASSERT_RESULT_SUCCESS(GetDisplay()->GetLayerStack(&stack));
    EXPECT_EQ(nn::vi::LayerStack_Null, stack);
}
