﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <vector>
#include <nn/vi/vi_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "master/detail/visrv_DefaultDisplayStrategy.h"
#include "testDisplay_MockPhysicalDisplay.h"

typedef std::vector<nn::visrv::master::detail::IPhysicalDisplay*> DisplayList;

TEST(DefaultDisplayStrategy, GetDisplay_Any)
{
    MockPhysicalDisplay lcd;
    MockPhysicalDisplay hdmi;
    nn::visrv::master::detail::IPhysicalDisplay* displays[] =
    {
        &hdmi,
        &lcd,
    };
    nn::visrv::master::detail::DefaultDisplayStrategy strategy;
    strategy.Initialize(displays, sizeof(displays) / sizeof(displays[0]));

    // neither display is powered
    EXPECT_EQ(&lcd, strategy.GetDisplay());

    NNT_ASSERT_RESULT_SUCCESS(lcd.SetPowerState(nn::vi::PowerState_On));
    EXPECT_EQ(&lcd, strategy.GetDisplay());

    NNT_ASSERT_RESULT_SUCCESS(hdmi.SetPowerState(nn::vi::PowerState_On));
    EXPECT_EQ(&hdmi, strategy.GetDisplay());

    NNT_ASSERT_RESULT_SUCCESS(lcd.SetPowerState(nn::vi::PowerState_Off));
    EXPECT_EQ(&hdmi, strategy.GetDisplay());
}

TEST(DefaultDisplayStrategy, Visit_Any)
{
    MockPhysicalDisplay lcd;
    MockPhysicalDisplay hdmi;
    nn::visrv::master::detail::IPhysicalDisplay* displays[] =
    {
        &hdmi,
        &lcd,
    };
    nn::visrv::master::detail::DefaultDisplayStrategy strategy;
    strategy.Initialize(displays, sizeof(displays) / sizeof(displays[0]));

    DisplayList visited;

    NNT_EXPECT_RESULT_SUCCESS(
        strategy.Visit(
            [](nn::visrv::master::detail::IPhysicalDisplay* display, void* userData) -> nn::Result
            {
                static_cast<DisplayList*>(userData)->push_back(display);
                return nn::ResultSuccess();
            },
            &visited
        )
    );

    ASSERT_EQ(2, visited.size());
    EXPECT_EQ(&hdmi, visited[0]);
    EXPECT_EQ(&lcd, visited[1]);
}

TEST(DefaultDisplayStrategy, Visit_ResultPropagation_Any)
{
    MockPhysicalDisplay lcd;
    MockPhysicalDisplay hdmi;
    nn::visrv::master::detail::IPhysicalDisplay* displays[] =
    {
        &hdmi,
        &lcd,
    };
    nn::visrv::master::detail::DefaultDisplayStrategy strategy;
    strategy.Initialize(displays, sizeof(displays) / sizeof(displays[0]));

    DisplayList visited;

    NNT_EXPECT_RESULT_FAILURE(
        nn::vi::ResultOperationFailed,
        strategy.Visit(
            [](nn::visrv::master::detail::IPhysicalDisplay* display, void* userData) -> nn::Result
            {
                static_cast<DisplayList*>(userData)->push_back(display);
                NN_RESULT_THROW(nn::vi::ResultOperationFailed());
            },
            &visited
        )
    );

    ASSERT_EQ(1, visited.size());
    EXPECT_EQ(&hdmi, visited[0]);

    visited.clear();

    NNT_EXPECT_RESULT_FAILURE(
        nn::vi::ResultOperationFailed,
        strategy.Visit(
            [](nn::visrv::master::detail::IPhysicalDisplay* display, void* userData) -> nn::Result
            {
                DisplayList* list = static_cast<DisplayList*>(userData);
                list->push_back(display);

                if( list->size() > 1)
                {
                    NN_RESULT_THROW(nn::vi::ResultOperationFailed());
                }

                NN_RESULT_SUCCESS;
            },
            &visited
        )
    );

    ASSERT_EQ(2, visited.size());
    EXPECT_EQ(&hdmi, visited[0]);
    EXPECT_EQ(&lcd, visited[1]);
}
