﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayConfig.h>
#include <nn/vi/vi_DisplayEvents.h>
#include <nn/vi/vi_DisplayEventsDebug.h>
#include <nn/vi/vi_CmuLuma.h>
#include <nnt/result/testResult_Assert.h>
#include <nnt/viUtil/testVi_Fixture.h>

typedef nnt::vi::util::Fixture Display;

// TODO: move to process with only vi:u
TEST_F(Display, DISABLED_ListDisplays_Any)
{
    nn::vi::Initialize();

    nn::vi::DisplayInfo info;
    ASSERT_EQ(1, nn::vi::ListDisplays(&info, 1));

    EXPECT_STREQ("Default", info.name);
    EXPECT_EQ(true, info.hasLayerLimit);
    EXPECT_EQ(1, info.layerCountMax);
    EXPECT_EQ(1920, info.layerWidthPixelCountMax);
    EXPECT_EQ(1080, info.layerHeightPixelCountMax);
}

TEST_F(Display, ListDisplays_UIG_Any)
{
    nn::vi::Initialize();

    nn::vi::DisplayInfo info;
    ASSERT_EQ(1, nn::vi::ListDisplays(&info, 1));

    EXPECT_STREQ("Default", info.name);
    EXPECT_EQ(true, info.hasLayerLimit);
    EXPECT_EQ(2, info.layerCountMax);
    EXPECT_EQ(1920, info.layerWidthPixelCountMax);
    EXPECT_EQ(1080, info.layerHeightPixelCountMax);
}

TEST_F(Display, ListDisplays_Zero_Any)
{
    nn::vi::Initialize();

    nn::vi::DisplayInfo info;
    ASSERT_EQ(0, nn::vi::ListDisplays(&info, 0));
}

TEST_F(Display, ListDisplays_Null_Any)
{
    nn::vi::Initialize();
    ASSERT_EQ(0, nn::vi::ListDisplays(nullptr, 1));
}

TEST_F(Display, OpenDefaultDisplay_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));
    ASSERT_NE(nullptr, pDisplay);

    // test double open
    nn::vi::Display* pDuplicate = nullptr;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDuplicate));
    ASSERT_NE(nullptr, pDuplicate);
    EXPECT_EQ(pDisplay, pDuplicate);

    nn::vi::CloseDisplay(pDisplay);
    nn::vi::CloseDisplay(pDuplicate);
}

TEST_F(Display, OpenDisplay_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay = nullptr;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDisplay(&pDisplay, "Default"));
    ASSERT_NE(nullptr, pDisplay);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, OpenDisplay_NotFound_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    nn::Result result = nn::vi::OpenDisplay(&pDisplay, "Nope");
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotFound, result);
    EXPECT_EQ(nullptr, pDisplay);
}

TEST_F(Display, GetDisplayResolution_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    int width  = -1;
    int height = -1;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayResolution(&width, &height, pDisplay));
    // this will likely vary depending on the exact environment (i.e. HDMI/DSI, PC, etc.)
    // make sure we get some interesting value back
    EXPECT_NE(-1, width);
    EXPECT_NE(-1, height);

    nn::vi::CloseDisplay(pDisplay);
}

// TODO: move to process with only vi:u
TEST_F(Display, DISABLED_GetZOrderCountMin_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    EXPECT_EQ(0, nn::vi::GetZOrderCountMin(pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetZOrderCountMin_UIG_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    EXPECT_EQ(0, nn::vi::GetZOrderCountMin(pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

// TODO: move to process with only vi:u
TEST_F(Display, DISABLED_GetZOrderCountMax_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    EXPECT_EQ(0, nn::vi::GetZOrderCountMax(pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetZOrderCountMax_UIG_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    EXPECT_EQ(255, nn::vi::GetZOrderCountMax(pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetUnderscan_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayUnderscan(pDisplay, 0));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetUnderscan_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    int underscan = 50;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayUnderscan(&underscan, pDisplay));
    // make sure no write happens in this case
    EXPECT_EQ(underscan, 50);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetDisplayAlpha_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayAlpha(pDisplay, 1.f));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetDisplayLayerStack_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayLayerStack(pDisplay, nn::vi::LayerStack_Default));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetDisplayPowerState_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::SetDisplayPowerState(pDisplay, nn::vi::PowerState_On));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetDisplayHotplugEvent_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType hotplugEvent;
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayHotplugEvent(&hotplugEvent, pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetDisplayCmuLuma_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    float values[] = { -1.f, 1.f, 0.f };

    for( float value : values )
    {
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayCmuLuma(pDisplay, value));
        float luma;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayCmuLuma(&luma, pDisplay));
        EXPECT_EQ(value, luma);
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetDisplayCmuLuma_InvalidRange_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, -1.1f));
    NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultInvalidRange, nn::vi::SetDisplayCmuLuma(pDisplay, 1.1f));

    nn::vi::CloseDisplay(pDisplay);
}

// Disabling due to SIGLO-43619
TEST_F(Display, DISABLED_GetDisplayVsyncEvent_Any)
{
    int priority = nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), nn::os::HighestThreadPriority);
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), priority);
    };

    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEventType vsync;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));
    // best effort to see if event is initialized
    ASSERT_NE(nn::os::SystemEventType::State_NotInitialized, vsync._state);

    {
        // only one event per client
        nn::os::SystemEventType vsync;
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultDenied, nn::vi::GetDisplayVsyncEvent(&vsync, pDisplay));
    }

    nn::os::SystemEventType vsyncDebug;
    NNT_EXPECT_RESULT_SUCCESS(nn::vi::GetDisplayVsyncEventForDebug(&vsyncDebug, pDisplay));
    ASSERT_NE(nn::os::SystemEventType::State_NotInitialized, vsyncDebug._state);

    {
        nn::os::SystemEventType vsyncDebug;
        NNT_EXPECT_RESULT_FAILURE(nn::vi::ResultDenied, nn::vi::GetDisplayVsyncEventForDebug(&vsyncDebug, pDisplay));
    }

    nn::os::MultiWaitHolderType vsyncHolder;
    nn::os::InitializeMultiWaitHolder(&vsyncHolder, &vsync);

    nn::os::MultiWaitHolderType vsyncDebugHolder;
    nn::os::InitializeMultiWaitHolder(&vsyncDebugHolder, &vsyncDebug);

    nn::os::MultiWaitType waiter;
    nn::os::InitializeMultiWait(&waiter);
    nn::os::LinkMultiWaitHolder(&waiter, &vsyncHolder);
    nn::os::LinkMultiWaitHolder(&waiter, &vsyncDebugHolder);

    for( int i = 0; i < 128; ++i )
    {
        nn::os::Tick start = nn::os::GetSystemTick();

        bool isVsyncSignalled = false;
        bool isVsyncDebugSignalled = false;

        while( !isVsyncSignalled || !isVsyncDebugSignalled )
        {
            nn::os::MultiWaitHolderType* signalled = nn::os::WaitAny(&waiter);

            if( signalled == &vsyncHolder )
            {
                if( isVsyncSignalled )
                {
                    // already signalled but haven't received the debug event signal yet...
                    FAIL();
                }

                isVsyncSignalled = true;
                nn::os::ClearSystemEvent(&vsync);
            }
            else
            {
                if( signalled == &vsyncDebugHolder )
                {
                    if( isVsyncDebugSignalled )
                    {
                        // already signalled but haven't received the real vsync event signal yet...
                        FAIL();
                    }

                    isVsyncDebugSignalled = true;
                    nn::os::ClearSystemEvent(&vsyncDebug);
                }
            }
        }

        nn::os::Tick end = nn::os::GetSystemTick();

        // skip first iteration
        // both events fired, but we cannot expect a full vsync duration based on when the test started
        if( i != 0 )
        {
            nn::TimeSpan diff((end - start).ToTimeSpan());
            EXPECT_LE(diff, nn::TimeSpan::FromMicroSeconds(17167));
            NN_LOG("Elapsed: %d microseconds\n", diff.GetMicroSeconds());
        }
    }

    nn::os::UnlinkMultiWaitHolder(&vsyncDebugHolder);
    nn::os::FinalizeMultiWaitHolder(&vsyncDebugHolder);
    nn::os::UnlinkMultiWaitHolder(&vsyncHolder);
    nn::os::FinalizeMultiWaitHolder(&vsyncHolder);
    nn::os::FinalizeMultiWait(&waiter);

    nn::os::DestroySystemEvent(&vsyncDebug);
    nn::os::DestroySystemEvent(&vsync);
    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, SetRgbRange_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::vi::RgbRange ranges[nn::vi::RgbRange_Max];
    int rgbCount = nn::vi::ListDisplayRgbRanges(ranges, nn::vi::RgbRange_Max, pDisplay);
    ASSERT_NE(0, rgbCount);

    bool hasFullRangeSupport = false;
    for( int i = 0; i < rgbCount; ++i )
    {
        hasFullRangeSupport = hasFullRangeSupport || ranges[i] == nn::vi::RgbRange_Full;

        NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, ranges[i]));
        nn::vi::RgbRange actualRange;
        NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));
        EXPECT_EQ(ranges[i], actualRange);
    }

    NNT_ASSERT_RESULT_SUCCESS(nn::vi::SetDisplayRgbRange(pDisplay, nn::vi::RgbRange_Auto));
    nn::vi::RgbRange actualRange;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayRgbRange(&actualRange, pDisplay));

    if( hasFullRangeSupport )
    {
        EXPECT_EQ(nn::vi::RgbRange_Full, actualRange);
    }
    else
    {
        EXPECT_EQ(nn::vi::RgbRange_Limited, actualRange);
    }

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetDisplayLogicalResolution_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    int width;
    int height;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::GetDisplayLogicalResolution(&width, &height, pDisplay));

    EXPECT_EQ(1920, width);
    EXPECT_EQ(1080, height);

    nn::vi::CloseDisplay(pDisplay);
}

TEST_F(Display, GetDisplayModeChangedEvent_Any)
{
    nn::vi::Initialize();

    nn::vi::Display* pDisplay;
    NNT_ASSERT_RESULT_SUCCESS(nn::vi::OpenDefaultDisplay(&pDisplay));

    nn::os::SystemEvent event;
    NNT_ASSERT_RESULT_FAILURE(nn::vi::ResultNotSupported, nn::vi::GetDisplayModeChangedEvent(event.GetBase(), pDisplay));

    nn::vi::CloseDisplay(pDisplay);
}
