<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Verify TV is unaffected by powering on and off the test SDEV

    .DESCRIPTION
        Powers on and off the test SDEV.  Verifies that CEC and nothing else causes the TV to turn on or change active source.
#>

param
(
    [Parameter(Mandatory=$True)]
    [Int]
    # Number of iterations
    $Iterations,
    
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the test target
    $TestAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the test target
    $TestTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the remote control
    $RemoteControlAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the remote control
    $RemoteControlTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The IP Address of the power source for the HDMI splitter
    $HDMIPowerIP,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The offset number of the power source for the HDMI splitter
    $HDMIPowerNum,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the microphone server
    $ServerAddress,
    
    [Parameter(Mandatory=$True)]
    [Boolean]
    # false: TV turns off
    # true: TV stays on
    $PowerOffTV
    
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# Note: Most of the above parameters are used in Common.ps1
. "${scriptDirectoryPath}\Common.ps1"

function Verify-Nothing-Happened
{
    Write-Host "Making sure nothing happened..."

    # Nothing is supposed to happen, so the TV may or may not be on
    if ($PowerOffTV)
    {
        # Fail if we hear the test tone or remote control tone
        & $CheckToneExe -silentTone $TestTone -silentTone $RemoteControlTone -ipAddr $ServerAddress > $null
        if ($lastexitcode -ne 0)
        {
            Cleanup
            Write-Error "xxx TEST MACHINE TURNED ON TV xxx"
            
            Write-Error "Aborting"
            exit 1
        }
    }
    else
    {
        # Fail if we don't hear the remote control tone
        # Try a few times in case there was a blip in the tone
        $attemptCounter = 1
        do
        {
            if ($attemptCounter -ge 5)
            {
                Cleanup
                Write-Error "xxx REMOTE CONTROL TONE LOST xxx"
                
                & $CheckToneExe -activeTone $TestTone -ipAddr $ServerAddress > $null
                if ($lastexitcode -eq 0)
                {
                    Write-Error "Note: TV is playing test tone (test machine should not have done anything)"
                }
                
                Write-Error "Aborting"
                exit 1
            }

            Write-Host "- Attempt #${attemptCounter} -"
            
            Start-Sleep -s 1

            & $CheckToneExe -activeTone $RemoteControlTone -silentTone $TestTone -ipAddr $ServerAddress > $null
            
            $attemptCounter = $attemptCounter + 1
        }
        while ($lastexitcode -ne 0)
    }
    
    Write-Host "Confirmed nothing happened!"
}

Initialize -initRemoteControl $True

Write-Host "Undocking test machine..."
& $ControlTargetExe disable-cradle -t $TestAddress > $null

Write-Host "Powering off test machine..."
& $ControlTargetExe power-off -t $TestAddress > $null

Get-TV-Focus-With-Remote

if ($PowerOffTV)
{
    Power-Off-TV-With-Remote($true)
}

for ($i = 1; $i -le $Iterations; ++$i)
{
    Write-Host "=== Iteration $i ==="

    Write-Host "Undocking test machine..."
    & $ControlTargetExe disable-cradle -t $TestAddress > $null

    Write-Host "Powering off test machine..."
    & $ControlTargetExe power-off -t $TestAddress > $null
    
#################################################################
    Write-Host "Docking test machine..."
    & $ControlTargetExe enable-cradle -t $TestAddress > $null
        
#################################################################
    # Verify this had no impact on the TV
    Write-Host "Waiting 30 seconds for nothing to happen..."
    Start-Sleep -s 30
    
    Verify-Nothing-Happened
    
#################################################################
    Write-Host "Powering on test machine..."
    & $ControlTargetExe power-on -t $TestAddress > $null
        
#################################################################
    # Verify this had no impact on the TV
    Write-Host "Waiting 30 seconds for nothing to happen..."
    Start-Sleep -s 30
    
    Verify-Nothing-Happened
}

Cleanup

Write-Host "Passed!"
