<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Runs various CEC tests on multiple SDEVS simultaneously

    .DESCRIPTION
        Runs several tests to ensure CEC works properly. These tests are run on multiple SDEVs simulaneously.
        Pulls the list of SDEVs and server address from ListFile (relative to script folder).
        Only uses TVs that have at least two SDEVs.
        The first SDEV listed for a TV is the test system, the second is the remote control, and any extra are ignored
#>

param
(
    [Parameter(Mandatory=$False)]
    [Int]
    # Number of iterations
    $Iterations = 5,
    [Parameter(Mandatory=$False)]
    [String]
    # The location of the list of SDEVs/TVs relative to the script folder
    $ListFile = "Display Lab.csv"
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)
$scriptRootPath      = "$env:NINTENDO_SDK_ROOT"
$toolsDirectoryPath  = "${scriptRootPath}\Tools\CommandLineTools"

if ($scriptRootPath -eq "")
{
    Write-Error "Error: NINTENDO_SDK_ROOT not set"
    Write-Error "Aborting"
    exit 1
}

$listCsv = Import-Csv "$scriptDirectoryPath\$ListFile"

$tvList = @{}

# Get the server and list of TVs
foreach ($entry in $listCsv)
{
    $type = $entry.Type.Trim()
    $identifier = $entry.Identifier.Trim()
    $name = $entry.Name.Trim()
    $parameter = $entry.Parameter.Trim()
    
    if ($type -eq "server" -and $name -eq "NAME")
    {
        $serverAddress = $parameter
        continue
    }

    if ($type -eq "TV")
    {
        if($tvList.ContainsKey($identifier))
        {
            $tv = $tvList.Item($identifier)
        }
        else
        {
            $tv = @{}
            $tvList.Add($identifier, $tv)
        }
        
        $tv.Add($name, $parameter)
    }
}

if ($tvList.Count -eq 0)
{
    Write-Error "Error: No TVs in ${ListFile}"
    exit -1
}

$tempYml = "${scriptDirectoryPath}/temp.NX.testlist.yml"
$timeout = $Iterations * 300 + 600

$newListText = "
    
---

Platforms     : [ Win32 ]
BuildTypes    : [ Develop ]

Timeout: ${timeout}

Tests:
  - Units:"

"
Platforms     : [ Win32 ]
BuildTypes    : [ Develop ]

Timeout: ${timeout}

Tests:
  - Units:" > $tempYml

Try
{

    $toneList = @(440, 499, 565, 641, 726, 823, 933, 1058, 1199, 1359, 1541, 1747)
    $testList = @("SleepWake", "DockUndock")
    $tvPowerList = @("0", "1")
    $first = $true

    foreach ($tvOff in $tvPowerList)
    {
        foreach ($test in $testList)
        {
            $counter = 0
            
            if (!$first)
            {
                $newListText >> $tempYml
            }
            
            $first = $false

            foreach ($identifier in $tvList.Keys)
            {
                $tv = $tvList.Item($identifier)
                
                # Only do TVs with at least two SDEVs and hdmi power information
                if ($tv.ContainsKey("SDEV_0") -and $tv.ContainsKey("SDEV_1") -and $tv.ContainsKey("HDMI_POWER_IP") -and $tv.ContainsKey("HDMI_POWER_NUM"))
                {
                    # There are only so many tones... if it's exceeded, we'll need to do it separately
                    if ($counter > $toneList.Length)
                    {
                        $counter = 0
                        
                        $newListText >> $tempYml
                    }

                    $rcAddress = $tv.Item("SDEV_1")
                    $testAddress = $tv.Item("SDEV_0")
                    $hdmiPowerIP = $tv.Item("HDMI_POWER_IP")
                    $hdmiPowerNum = $tv.Item("HDMI_POWER_NUM")
                    $testTone = $toneList[$counter]
                    $rcTone = $toneList[$counter + 1]
                    
                    "    - Path: ${scriptDirectoryPath}/ActiveSourceTest.ps1
      Option: -ArtifactPath `${RESULT_DIR} -Iterations ${Iterations} -ServerAddress ${serverAddress} -TestAddress ${testAddress} -RemoteControlAddress ${rcAddress} -TestTone ${testTone} -RemoteControlTone ${rcTone} -HDMIPowerIP ${hdmiPowerIP} -HDMIPowerNum ${hdmiPowerNum} -TestType ${test} -PowerOffTV ${tvOff}" >> $tempYml

                    & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $testTone -tone $rcTone
                      
                    $counter = $counter + 2
                }
            }
        }
        
        $counter = 0

        $newListText >> $tempYml

        foreach ($identifier in $tvList.Keys)
        {
            $tv = $tvList.Item($identifier)
            
            # Only do TVs with at least two SDEVs and hdmi power information
            if ($tv.ContainsKey("SDEV_0") -and $tv.ContainsKey("SDEV_1") -and $tv.ContainsKey("HDMI_POWER_IP") -and $tv.ContainsKey("HDMI_POWER_NUM"))
            {
                # There are only so many tones... if it's exceeded, we'll need to do it separately
                if ($counter > $toneList.Length)
                {
                    $counter = 0
                    
                    $newListText >> $tempYml
                }

                $rcAddress = $tv.Item("SDEV_1")
                $testAddress = $tv.Item("SDEV_0")
                $hdmiPowerIP = $tv.Item("HDMI_POWER_IP")
                $hdmiPowerNum = $tv.Item("HDMI_POWER_NUM")
                $testTone = $toneList[$counter]
                $rcTone = $toneList[$counter + 1]
                
                "    - Path: ${scriptDirectoryPath}/PowerCradleTest.ps1
      Option: -ArtifactPath `${RESULT_DIR} -Iterations ${Iterations} -ServerAddress ${serverAddress} -TestAddress ${testAddress} -RemoteControlAddress ${rcAddress} -TestTone ${testTone} -RemoteControlTone ${rcTone} -HDMIPowerIP ${hdmiPowerIP} -HDMIPowerNum ${hdmiPowerNum} -PowerOffTV ${tvOff}" >> $tempYml

                & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $testTone -tone $rcTone
                  
                $counter = $counter + 2
            }
        }
        
        $counter = 0

        $newListText >> $tempYml
                
        foreach ($identifier in $tvList.Keys)
        {
            $tv = $tvList.Item($identifier)
            
            # Only do TVs with at least two SDEVs and hdmi power information
            if ($tv.ContainsKey("SDEV_0") -and $tv.ContainsKey("SDEV_1") -and $tv.ContainsKey("HDMI_POWER_IP") -and $tv.ContainsKey("HDMI_POWER_NUM"))
            {
                # There are only so many tones... if it's exceeded, we'll need to do it separately
                if ($counter > $toneList.Length)
                {
                    $counter = 0
                    
                    $newListText >> $tempYml
                }

                $rcAddress = $tv.Item("SDEV_1")
                $testAddress = $tv.Item("SDEV_0")
                $hdmiPowerIP = $tv.Item("HDMI_POWER_IP")
                $hdmiPowerNum = $tv.Item("HDMI_POWER_NUM")
                $testTone = $toneList[$counter]
                $rcTone = $toneList[$counter + 1]
                
                "    - Path: ${scriptDirectoryPath}/HDMIDisconnectTest.ps1
      Option: -ArtifactPath `${RESULT_DIR} -Iterations ${Iterations} -ServerAddress ${serverAddress} -TestAddress ${testAddress} -RemoteControlAddress ${rcAddress} -TestTone ${testTone} -RemoteControlTone ${rcTone} -HDMIPowerIP ${hdmiPowerIP} -HDMIPowerNum ${hdmiPowerNum} -PowerOffTV ${tvOff}" >> $tempYml

                & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $testTone -tone $rcTone
                  
                $counter = $counter + 2
            }
        }
    }
        
    $counter = 0

    $newListText >> $tempYml

    foreach ($identifier in $tvList.Keys)
    {
        $tv = $tvList.Item($identifier)
        
        # Only do TVs with at least one SDEV and hdmi power information
        if ($tv.ContainsKey("SDEV_0") -and $tv.ContainsKey("HDMI_POWER_IP") -and $tv.ContainsKey("HDMI_POWER_NUM"))
        {
            # There are only so many tones... if it's exceeded, we'll need to do it separately
            if ($counter > $toneList.Length)
            {
                $counter = 0
                
                $newListText >> $tempYml
            }

            $testAddress = $tv.Item("SDEV_0")
            $hdmiPowerIP = $tv.Item("HDMI_POWER_IP")
            $hdmiPowerNum = $tv.Item("HDMI_POWER_NUM")
            $testTone = $toneList[$counter]
            
            "    - Path: ${scriptDirectoryPath}/HDCPTest.ps1
      Option: -ArtifactPath `${RESULT_DIR} -ServerAddress ${serverAddress} -TestAddress ${testAddress}  -TestTone ${testTone} -HDMIPowerIP ${hdmiPowerIP} -HDMIPowerNum ${hdmiPowerNum}" >> $tempYml

            & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $testTone
              
            $counter = $counter + 1
        }
    }
        
    $counter = 0

    $newListText >> $tempYml

    foreach ($identifier in $tvList.Keys)
    {
        $tv = $tvList.Item($identifier)
        
        # Only do TVs with at least one SDEV and hdmi power information
        if ($tv.ContainsKey("SDEV_0") -and $tv.ContainsKey("HDMI_POWER_IP") -and $tv.ContainsKey("HDMI_POWER_NUM"))
        {
            # There are only so many tones... if it's exceeded, we'll need to do it separately
            if ($counter > $toneList.Length)
            {
                $counter = 0
                
                $newListText >> $tempYml
            }

            $testAddress = $tv.Item("SDEV_0")
            $hdmiPowerIP = $tv.Item("HDMI_POWER_IP")
            $hdmiPowerNum = $tv.Item("HDMI_POWER_NUM")
            $testTone = $toneList[$counter]
            
            "    - Path: ${scriptDirectoryPath}/HDCPPerfTest.ps1
      Option: -ArtifactPath `${RESULT_DIR} -Iterations ${Iterations} -ServerAddress ${serverAddress} -TestAddress ${testAddress}  -TestTone ${testTone} -HDMIPowerIP ${hdmiPowerIP} -HDMIPowerNum ${hdmiPowerNum}" >> $tempYml

            & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $testTone
              
            $counter = $counter + 1
        }
    }

    powershell "${scriptRootPath}\Integrate\Scripts\Invoke-TestRunner.ps1" -BuildType Develop -Platform Win32 -File $tempYml

    foreach ($tone in $toneList)
    {
        & "${scriptRootPath}/Tests/Outputs/Win32-v140/Tests/ToggleTone/Develop/ToggleTone.exe" -ipAddr $serverAddress -tone $tone -disable
    }
}
Catch
{}
Finally
{
    Remove-Item $tempYml
}