<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Enable/disable with changing active source test

    .DESCRIPTION
        Enables and disables the test SDEV and makes sure it becomes the active source accordingly
        For sleep/wake, ensures the test SDEV goes to sleep when told to through CEC
#>

param
(
    [Parameter(Mandatory=$True)]
    [Int]
    # Number of iterations
    $Iterations,
    
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the test target
    $TestAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the test target
    $TestTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the remote control
    $RemoteControlAddress,
    
    [Parameter(Mandatory=$True)]
    [Int]
    # The tone to be emitted by the remote control
    $RemoteControlTone,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The IP Address of the power source for the HDMI splitter
    $HDMIPowerIP,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The offset number of the power source for the HDMI splitter
    $HDMIPowerNum,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the microphone server
    $ServerAddress,
    
    [Parameter(Mandatory=$True)]
    [ValidateSet("SleepWake","DockUndock")]
    [String]
    # The transition to perform that will trigger CEC
    $TestType,
    
    [Parameter(Mandatory=$True)]
    [Boolean]
    # false: TV turns off
    # true: TV stays on
    $PowerOffTV
    
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

# Note: Most of the above parameters are used in Common.ps1
. "${scriptDirectoryPath}\Common.ps1"

Initialize -InitRemoteControl $True

for ($i = 1; $i -le $Iterations; ++$i)
{
    Write-Host "=== Iteration $i ==="

#################################################################
    # Disable the test machine, take control with the RC, and turn off the TV if necessary
    if ($TestType -eq "DockUndock")
    {
        Write-Host "Undocking test machine..."
        & $ControlTargetExe disable-cradle -t $TestAddress
    }

    Get-TV-Focus-With-Remote
        
    if ($PowerOffTV)
    {
        Power-Off-TV-With-Remote($true)
    }
    elseif ($TestType -eq "SleepWake")
    {
        Write-Host "Pressing power button on test machine..."
        & $ControlTargetExe press-power-button -t $TestAddress > $null
            
        Start-Sleep -s $SleepDelay
    }
    
#################################################################
    # Enable the test machine 
    if ($TestType -eq "SleepWake")
    {
        Write-Host "Waking test machine..."
        & $ControlTargetPrivateExe press-power-button-for-sc7 --ip-addr $TestAddress > $null
        
        Write-Host "Connecting..."
        & $ControlTargetExe connect -t $TestAddress > $null
    }
    elseif ($TestType -eq "DockUndock")
    {
        Write-Host "Docking test machine..."
        & $ControlTargetExe enable-cradle -t $TestAddress
    }
    
#################################################################
    # Verify the test machine is the active source
    Write-Host "Playing tone..."
    & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\PlayTone\Develop\PlayTone.nsp" --no-wait -t $TestAddress -- -frequency $TestTone -amplitude $TestAmplitude > $null
    
    Write-Host "Waiting for tone..."
    
    $attemptCounter = 1
    do
    {
        if ($attemptCounter -ge 60)
        {
            Cleanup
            Write-Error "xxx TEST MACHINE FAILED TO TURN ON TV xxx"
            
            & $CheckToneExe -activeTone $RemoteControlTone -ipAddr $ServerAddress > $null
            if ($lastexitcode -eq 0)
            {
                Write-Error "Note: TV is on, but playing the remote control's tone"
            }
            
            Write-Error "Aborting"
            exit 1
        }

        Write-Host "- Attempt #${attemptCounter} -"
        
        Start-Sleep -s 1

        & $CheckToneExe -activeTone $TestTone -silentTone $RemoteControlTone -ipAddr $ServerAddress > $null
        
        $attemptCounter = $attemptCounter + 1
    }
    while ($lastexitcode -ne 0)
}

Cleanup

Write-Host "Passed!"
