<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Underscan Test

    .DESCRIPTION
        Verifies that the underscan changes properly with a given transition
#>

param
(
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the target
    $TargetAddress
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

. "${scriptDirectoryPath}\Common.ps1"

function Change-Display([int]$resolution)
{
    Write-Host "Changing display to ${resolution}p with no underscan..."
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" -- -resolution $resolution -underscan 0 -update_display -update_underscan -update_nand -no_run
    
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = 0
}

function Get-ResUnderscan-Name([int]$resolution, [int]$underscan)
{
    if ($resolution -eq 480 -Or $underscan -eq 0)
    {
        return "${resolution}"
    }
    
    return "${resolution}_${underscan}"
}

function Test-Transition-DockUndock([int]$resolution, [int]$underscan)
{
    Write-Host "=== ${transitionCounter}: ${resolution}p underscan ${underscan}% ==="
    $name = Get-ResUnderscan-Name $resolution $underscan
    $prevName = Get-ResUnderscan-Name $prevResolution $prevUnderscan
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution $resolution -underscan $underscan -update_nand
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${prevName}_1" "${transitionCounter}p_" $LONG_WAIT
    
    Write-Host "Undocking..."
    & $ControlTargetExe disable-cradle

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "no_signal${prevResolution}" "${transitionCounter}a_" $LONG_WAIT

    Write-Host "Docking..."
    & $ControlTargetExe enable-cradle
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_1" "${transitionCounter}b_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_2" "${transitionCounter}c_" $SHORT_WAIT
    
    ${script:transitionCounter} = ${script:transitionCounter} + 1
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = $underscan
}

function Test-Transition-SleepWake([int]$resolution, [int]$underscan, [string]$targetAddress)
{
    Write-Host "=== ${transitionCounter}: ${resolution}p underscan ${underscan}% ==="
    $name = Get-ResUnderscan-Name $resolution $underscan
    $prevName = Get-ResUnderscan-Name $prevResolution $prevUnderscan
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution $resolution -underscan $underscan -update_nand
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${prevName}_1" "${transitionCounter}p_" $LONG_WAIT
    
    Write-Host "Sleeping..."
    & $ControlTargetExe press-power-button

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "no_signal${prevResolution}" "${transitionCounter}a_" $LONG_WAIT

    Start-Sleep -s 10

    Write-Host "Waking..."
    & $ControlTargetPrivateExe press-power-button-for-sc7 --ip-addr $targetAddress

    & $ControlTargetExe connect

    & $ControlTargetExe enable-cradle
    
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_1" "${transitionCounter}b_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_2" "${transitionCounter}c_" $SHORT_WAIT
    
    ${script:transitionCounter} = ${script:transitionCounter} + 1
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = $underscan
}

function Test-Transition-SystemReset([int]$resolution, [int]$underscan)
{
    Write-Host "=== ${transitionCounter}: ${resolution}p underscan ${underscan}% ==="
    $name = Get-ResUnderscan-Name $resolution $underscan
    $prevName = Get-ResUnderscan-Name $prevResolution $prevUnderscan
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution $resolution -underscan $underscan -update_nand
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${prevName}_1" "${transitionCounter}p_" $LONG_WAIT
    
    Write-Host "Restarting..."
    & $ControlTargetExe reset

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "no_signal${prevResolution}" "${transitionCounter}a_" $LONG_WAIT

    & $ControlTargetExe connect
    
    & $ControlTargetExe enable-cradle

    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_1" "${transitionCounter}b_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_2" "${transitionCounter}c_" $SHORT_WAIT
    
    ${script:transitionCounter} = ${script:transitionCounter} + 1
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = $underscan
}

function Test-Transition-HDMIHotplug([int]$resolution, [int]$underscan)
{
    Write-Host "=== ${transitionCounter}: ${resolution}p underscan ${underscan}% ==="
    $name = Get-ResUnderscan-Name $resolution $underscan
    $prevName = Get-ResUnderscan-Name $prevResolution $prevUnderscan
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution $resolution -underscan $underscan -update_nand
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${prevName}_1" "${transitionCounter}p_" $LONG_WAIT
    
    Write-Host "Disconnecting HDMI..."
    Write-HDMI-Switcher "s 2"

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "no_signal${prevResolution}" "${transitionCounter}a_" $LONG_WAIT

    # Note: As sending an HPD pulse is not reliable, the edid is reset instead to force a takedown/bringup
    Write-Host "Connecting HDMI and resetting EDID..."
    Write-HDMI-Switcher "s 1"
    Write-HDMI-Switcher "#set_edid output 1 input 1"

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_1" "${transitionCounter}b_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_2" "${transitionCounter}c_" $SHORT_WAIT
    
    ${script:transitionCounter} = ${script:transitionCounter} + 1
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = $underscan
}

function Test-Transition-ViUpdate([int]$resolution, [int]$underscan)
{
    Write-Host "=== ${transitionCounter}: ${resolution}p underscan ${underscan}% ==="
    Write-Host "Switching..."
    $name = Get-ResUnderscan-Name $resolution $underscan
    $prevName = Get-ResUnderscan-Name $prevResolution $prevUnderscan
    $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution $resolution -underscan $underscan -update_display -update_underscan
    
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_1" "${transitionCounter}b_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${name}_2" "${transitionCounter}c_" $SHORT_WAIT
    
    ${script:transitionCounter} = ${script:transitionCounter} + 1
    ${script:prevResolution} = $resolution
    ${script:prevUnderscan} = $underscan
}

function Test-Resolution([int]$resolution)
{
    Write-Host "  ==========="
    Write-Host "==== ${resolution}p ====="
    Write-Host "  ==========="
    Change-Display $resolution

    Test-Transition-DockUndock $resolution 20
    Test-Transition-DockUndock 480 20

    Test-Transition-SleepWake $resolution 19 $TargetAddress
    Test-Transition-DockUndock 480 19

    Test-Transition-SystemReset $resolution 18
    Test-Transition-DockUndock 480 18

    Test-Transition-HDMIHotplug $resolution 17
    Test-Transition-DockUndock 480 17

    Test-Transition-ViUpdate $resolution 16
    Test-Transition-SleepWake 480 16 $TargetAddress

    Test-Transition-DockUndock $resolution 15
    Test-Transition-SleepWake 480 15 $TargetAddress

    Test-Transition-SleepWake $resolution 14 $TargetAddress
    Test-Transition-SleepWake 480 14 $TargetAddress

    Test-Transition-SystemReset $resolution 13
    Test-Transition-SleepWake 480 13 $TargetAddress

    Test-Transition-HDMIHotplug $resolution 12
    Test-Transition-SystemReset 480 12

    Test-Transition-ViUpdate $resolution 11
    Test-Transition-SystemReset 480 11

    Test-Transition-DockUndock $resolution 10
    Test-Transition-SystemReset 480 10

    Test-Transition-SleepWake $resolution 9 $TargetAddress
    Test-Transition-SystemReset 480 9

    Test-Transition-SystemReset $resolution 8
    Test-Transition-HDMIHotplug 480 8

    Test-Transition-HDMIHotplug $resolution 7
    Test-Transition-HDMIHotplug 480 7

    Test-Transition-ViUpdate $resolution 6
    Test-Transition-HDMIHotplug 480 6

    Test-Transition-DockUndock $resolution 5
    Test-Transition-HDMIHotplug 480 5

    Test-Transition-SleepWake $resolution 4 $TargetAddress
    Test-Transition-ViUpdate 480 4

    Test-Transition-SystemReset $resolution 3
    Test-Transition-ViUpdate 480 3

    Test-Transition-HDMIHotplug $resolution 2
    Test-Transition-ViUpdate 480 2

    Test-Transition-ViUpdate $resolution 1
    Test-Transition-ViUpdate 480 1
}

$transitionCounter = 1
$outputFile = "${scriptDirectoryPath}\out.png"
$startRes = 1080

Init-Transition-Test $scriptDirectoryPath $outputFile $TargetAddress

###############################################################################################

Test-Resolution 720

Test-Resolution 1080

###############################################################################################

Cleanup-Transition-Test $scriptDirectoryPath

###############################################################################################
if ($failure)
{
    Write-Host "FAILURES OCCURRED"
    exit 1
}

Write-Host "Passed!"
exit 0
