<#
    Copyright (C)Nintendo All rights reserved.

    These coded instructions, statements, and computer programs contain proprietary
    information of Nintendo and/or its licensed developers and are protected by
    national and international copyright laws. They may not be disclosed to third
    parties or copied or duplicated in any form, in whole or in part, without the
    prior written consent of Nintendo.

    The content herein is highly confidential and should be handled accordingly.
#>

<#
    .SYNOPSIS
        Resolution Switching Test

    .DESCRIPTION
        Verifies that the resolution switches properly with a given transition
#>

param
(
    [Parameter(Mandatory=$False)]
    [String]
    # The artifact path
    $ArtifactPath,
    
    [Parameter(Mandatory=$True)]
    [String]
    # The ip address of the target
    $TargetAddress
)

$scriptPath          = $MyInvocation.MyCommand.Path
$scriptDirectoryPath = [System.IO.Path]::GetDirectoryName($scriptPath)

. "${scriptDirectoryPath}\Common.ps1"

function Test-Transition([int]$bank, [int]$resolution, [int]$postResolution)
{
    if ($resolution -ge 0)
    {
        $ret = & $RunOnTargetExe "${scriptRootPath}\Tests\Outputs\NX-NXFP2-a64\Tests\DisplayResolution\Develop\DisplayResolution.nsp" --no-wait -- -resolution ${resolution} -update_nand
    
        Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${prevResolution}_1" "${transitionCounter}p_" $LONG_WAIT
    }
    
    Write-HDMI-Switcher "#set_edid bank ${bank} input 1"

    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${postResolution}_1" "${transitionCounter}a_" $LONG_WAIT
    Wait-Picture "${scriptDirectoryPath}" "${outputFile}" "disp_resolution${postResolution}_2" "${transitionCounter}b_" $SHORT_WAIT
    
    $script:transitionCounter = $script:transitionCounter + 1
    $script:prevResolution = $postResolution;
}

$transitionCounter = 1
$outputFile = "${scriptDirectoryPath}\out.png"
$startRes = 1080
$prevResolution = 1080

Init-Transition-Test $scriptDirectoryPath $outputFile $TargetAddress

###############################################################################################

Write-Host "=== ${transitionCounter}: invalid (Auto) ==="
Test-Transition 1 0 480

Write-Host "=== ${transitionCounter}: 720p (Auto) ==="
Test-Transition 3 -1 720

Write-Host "=== ${transitionCounter}: none (Auto) ==="
Test-Transition 2 -1 480

Write-Host "=== ${transitionCounter}: 1080p (Auto) ==="
Test-Transition 4 -1 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (Auto) ==="
Test-Transition 6 -1 480

Write-Host "=== ${transitionCounter}: all (Auto) ==="
Test-Transition 5 -1 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (Auto) ==="
Test-Transition 7 -1 480

Write-Host "=== ${transitionCounter}: 720p (Auto) ==="
Test-Transition 3 -1 720

Write-Host "=== ${transitionCounter}: invalid (480p) ==="
Test-Transition 1 480 480

#################
Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 720 720

Write-Host "=== ${transitionCounter}: none (480p) ==="
Test-Transition 2 480 480

Write-Host "=== ${transitionCounter}: all (1080p) ==="
Test-Transition 5 1080 1080

Write-Host "=== ${transitionCounter}: all (480p) ==="
Test-Transition 5 480 480

Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 720 720

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (480p) ==="
Test-Transition 6 480 480

Write-Host "=== ${transitionCounter}: all (1080p) ==="
Test-Transition 5 1080 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (480p) ==="
Test-Transition 7 480 480

Write-Host "=== ${transitionCounter}: all (1080p) ==="
Test-Transition 5 1080 1080

Write-Host "=== ${transitionCounter}: invalid (720p) ==="
Test-Transition 1 720 480

#################
Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 -1 720

Write-Host "=== ${transitionCounter}: none (720p) ==="
Test-Transition 2 -1 480

Write-Host "=== ${transitionCounter}: 720p (720p) ==="
Test-Transition 3 -1 720

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (720p) ==="
Test-Transition 6 -1 480

Write-Host "=== ${transitionCounter}: 1080p (720p) ==="
Test-Transition 4 -1 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (720p) ==="
Test-Transition 7 -1 480

Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 -1 720

Write-Host "=== ${transitionCounter}: invalid (720p) ==="
Test-Transition 1 -1 480

Write-Host "=== ${transitionCounter}: all (720p) ==="
Test-Transition 5 -1 720

Write-Host "=== ${transitionCounter}: invalid (1080p) ==="
Test-Transition 1 1080 480

#################
Write-Host "=== ${transitionCounter}: 720p (1080p) ==="
Test-Transition 3 -1 720

Write-Host "=== ${transitionCounter}: none (1080p) ==="
Test-Transition 2 -1 480

Write-Host "=== ${transitionCounter}: 1080p (1080p) ==="
Test-Transition 4 -1 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (1st half) (1080p) ==="
Test-Transition 6 -1 480

Write-Host "=== ${transitionCounter}: all (1080p) ==="
Test-Transition 5 -1 1080

Write-Host "=== ${transitionCounter}: all but 720p/1080p (2nd half) (1080p) ==="
Test-Transition 7 -1 480

###############################################################################################

Cleanup-Transition-Test $scriptDirectoryPath

###############################################################################################
if ($failure)
{
    Write-Host "FAILURES OCCURRED"
    exit 1
}

Write-Host "Passed!"
exit 0
