﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cmath>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/audio.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Log.h>

char g_HeapBuffer[128 * 1024];

const double PI = 3.14159265358979;

//
// This sounds terrible, but it's a very clear tone
//
void GenerateSineWave(void* buffer, int sampleRate, int frequency, int sampleCount, int amplitude)
{
    NN_ASSERT_NOT_NULL(buffer);
    static int s_TotalSampleCount = 0;

    int16_t* buf = reinterpret_cast<int16_t*>(buffer);
    double increment = 2 * PI * frequency / static_cast<double>(sampleRate);

    for (int sample = 0; sample < sampleCount; sample++)
    {
        int16_t value = static_cast<int16_t>(sin(s_TotalSampleCount * increment) * amplitude);
        buf[sample] = value;
        s_TotalSampleCount++;
    }
}
extern "C" void nnMain()
{
    nn::mem::StandardAllocator allocator(g_HeapBuffer, sizeof(g_HeapBuffer));

    nn::audio::AudioOut audioOut;

    int frequency = 1000;
    int amplitude = 3500;

    int argc = nn::os::GetHostArgc();
    char** argv = nn::os::GetHostArgv();
    for(int i = 0; i < argc; ++i)
    {
        if(strcmp("-frequency", argv[i]) == 0)
        {
            NN_ASSERT(i < argc, "Expected argument after frequency!");
            frequency = atoi(argv[i + 1]);

            ++i;
            continue;
        }

        if(strcmp("-amplitude", argv[i]) == 0)
        {
            NN_ASSERT(i < argc, "Expected argument after amplitude!");
            amplitude = atoi(argv[i + 1]);

            ++i;
            continue;
        }
    }

    // Open the default audio (there's only one on the device)
    nn::os::SystemEvent systemEvent;
    nn::audio::AudioOutParameter parameter;
    nn::audio::InitializeAudioOutParameter(&parameter);
    parameter.sampleRate = 48000;
    parameter.channelCount = 2;
    if (nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsFailure())
    {
        parameter.sampleRate = 0;
        parameter.channelCount = 0;
        NN_ABORT_UNLESS(nn::audio::OpenDefaultAudioOut(&audioOut, &systemEvent, parameter).IsSuccess(), "Failed to open AudioOut.");
    }

    // Some constants
    int sampleRate = nn::audio::GetAudioOutSampleRate(&audioOut);

    const int frameRate = 20;                             // 20fps
    const int frameSampleCount = sampleRate / frameRate;  // 50msecs (in samples)
    const size_t dataSize = frameSampleCount * nn::audio::GetSampleByteSize(nn::audio::SampleFormat_PcmInt16);
    const size_t bufferSize = nn::util::align_up(dataSize, nn::audio::AudioOutBuffer::SizeGranularity);
    const int bufferCount = 4;

    nn::audio::AudioOutBuffer audioOutBuffer[bufferCount];
    void* outBuffer[bufferCount];
    for (int i = 0; i < bufferCount; ++i)
    {
        outBuffer[i] = allocator.Allocate(bufferSize, nn::audio::AudioOutBuffer::AddressAlignment);
        NN_ASSERT(outBuffer[i]);
        GenerateSineWave(outBuffer[i], sampleRate, frequency, frameSampleCount, amplitude);
        nn::audio::SetAudioOutBufferInfo(&audioOutBuffer[i], outBuffer[i], bufferSize, dataSize);
        nn::audio::AppendAudioOutBuffer(&audioOut, &audioOutBuffer[i]);
    }

    // Start playing the sound
    NN_ABORT_UNLESS(nn::audio::StartAudioOut(&audioOut).IsSuccess(), "Failed to start playback.");

    NN_LOG("Start audio playback with sample rate = %d Hz\n", sampleRate);
    nn::TimeSpan start = nn::os::GetSystemTick().ToTimeSpan();

    // Run for a maximum of 5 minutes
    while ((nn::os::GetSystemTick().ToTimeSpan() - start).GetMilliSeconds() < 5 * 1000 * 60)
    {
        systemEvent.Wait();

        nn::audio::AudioOutBuffer* pAudioOutBuffer = nullptr;

        pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        while (pAudioOutBuffer)
        {
            // Append to the audio buffer
            void* pOutBuffer = nn::audio::GetAudioOutBufferDataPointer(pAudioOutBuffer);
            size_t outSize = nn::audio::GetAudioOutBufferDataSize(pAudioOutBuffer);
            NN_ASSERT(outSize == frameSampleCount * nn::audio::GetSampleByteSize(nn::audio::SampleFormat_PcmInt16));
            GenerateSineWave(pOutBuffer, sampleRate, frequency, frameSampleCount, amplitude);
            nn::audio::AppendAudioOutBuffer(&audioOut, pAudioOutBuffer);

            pAudioOutBuffer = nn::audio::GetReleasedAudioOutBuffer(&audioOut);
        }
    }

    return;
}
