﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VideoConfigHelper.h"
#include "VideoConfigUtility.h"
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <cstdio>
#include <string>
#include <hardware/hwcomposer_defs.h>

namespace nns {

bool VideoConfigHelper::m_IsShowingDisplayInfo = true;
char VideoConfigHelper::m_VirtualEdidDirectory[PathLength];

NN_IMPLICIT VideoConfigHelper::VideoConfigHelper() NN_NOEXCEPT
:
m_IsVirtualEdidEnabled(false),
m_PreviousActiveDisplayId(-1)
{
}

VideoConfigHelper::~VideoConfigHelper() NN_NOEXCEPT
{
    if (m_IsVirtualEdidEnabled)
    {
        ResetVirtualEdid();
    }
}

bool VideoConfigHelper::IsExternalDisplayConnected() NN_NOEXCEPT
{
    return 0 < VideoConfigUtility::GetInstance().GetDisplayInfoCount();
}

void VideoConfigHelper::SetVirtualEdidDirectory(const char* path)
{
    std::strncpy(m_VirtualEdidDirectory, path, sizeof(m_VirtualEdidDirectory));
}

bool VideoConfigHelper::SetVirtualEdid(const char* name) NN_NOEXCEPT
{
    // Generate default virtual EDID path based on the location of nsp file.
    if (0 >= std::strlen(m_VirtualEdidDirectory))
    {
        char** argv = nn::os::GetHostArgv();
        std::string nspName = argv[0];
        for (int i1=0; i1<2; ++i1)
        {
            std::string::size_type pos = nspName.rfind('\\');
            if (std::string::npos != pos)
            {
                nspName = nspName.substr(0, pos);
            }
        }
        std::snprintf(m_VirtualEdidDirectory, PathLength, "%s\\Contents\\NX", nspName.c_str());
    }

    char path[PathLength];
    std::snprintf(path, PathLength, "%s\\%s", m_VirtualEdidDirectory, name);
    auto result = VideoConfigUtility::GetInstance().SetVirtualEdid(path);
    if (result)
    {
        m_IsVirtualEdidEnabled = true;
    }
    return result;
}

bool VideoConfigHelper::ResetVirtualEdid() NN_NOEXCEPT
{
    auto result = VideoConfigUtility::GetInstance().ResetVirtualEdid();
    if (result)
    {
        m_IsVirtualEdidEnabled = false;
    }
    return result;
}

void VideoConfigHelper::ShowDisplayInfo() NN_NOEXCEPT
{
    if (m_IsShowingDisplayInfo)
    {
        VideoConfigUtility::GetInstance().ShowDisplayInfo();
    }
}

void VideoConfigHelper::EnableShowingDisplayInfo(bool isEnabled) NN_NOEXCEPT
{
    m_IsShowingDisplayInfo = isEnabled;
}

int VideoConfigHelper::GetActiveDisplayId() NN_NOEXCEPT
{
    return VideoConfigUtility::GetInstance().GetActiveDisplayId();
}

int VideoConfigHelper::GetDisplayInfoCount() NN_NOEXCEPT
{
    return VideoConfigUtility::GetInstance().GetDisplayInfoCount();
}

int VideoConfigHelper::GetActiveDisplayWidth() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.w;
}

int VideoConfigHelper::GetActiveDisplayHeight() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.h;
}

float VideoConfigHelper::GetActiveDisplayFps() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.fps;
}

VideoConfigHelper::CeaAspectRatio VideoConfigHelper::GetActiveDisplayCeaAspectRatio() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    switch (info.aspectRatioFlag) {
    case HWC_DISPLAY_ASPECT_RATIO_4_3:
        NN_ASSERT(IsActiveDisplayCea());
        return AspectRatio_4x3;
    case HWC_DISPLAY_ASPECT_RATIO_16_9:
        NN_ASSERT(IsActiveDisplayCea());
        return AspectRatio_16x9;
    case HWC_DISPLAY_ASPECT_RATIO_64_27:
        NN_ASSERT(IsActiveDisplayCea());
        return AspectRatio_64x27;
    case HWC_DISPLAY_ASPECT_RATIO_256_135:
        NN_ASSERT(IsActiveDisplayCea());
        return AspectRatio_256x135;
    default:
        if (!IsActiveDisplayCea())
        {
            return AspectRatio_NotCea;
        }
        break;
    }
    return AspectRatio_Unknown;
}

bool VideoConfigHelper::IsActiveDisplayCea() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.modeFlags & HWC_DISPLAY_MODE_IS_CEA;
}

bool VideoConfigHelper::IsActiveDisplayDetailed() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.modeFlags & HWC_DISPLAY_MODE_IS_DETAILED;
}

bool VideoConfigHelper::IsActiveDisplayHdmiExtension() NN_NOEXCEPT
{
    int id = VideoConfigUtility::GetInstance().GetActiveDisplayId();
    auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(id);
    return info.modeFlags & HWC_DISPLAY_MODE_IS_HDMI_EXT;
}

bool VideoConfigHelper::SetMandatoryResolution() NN_NOEXCEPT
{
    if (0 <= m_PreviousActiveDisplayId)
    {
        return false;
    }
    int configs = VideoConfigUtility::GetInstance().GetDisplayInfoCount();
    for (int i1=0; i1<configs; ++i1)
    {
        auto info = VideoConfigUtility::GetInstance().GetDisplayInfo(i1);
        if (720 == info.w && 480 == info.h && 59.999f < info.fps && info.fps < 60.001f)
        {
            auto result = VideoConfigUtility::GetInstance().SetActiveDisplay(i1);
            if (result)
            {
                m_PreviousActiveDisplayId = VideoConfigUtility::GetInstance().GetActiveDisplayId();
            }
            return result;
        }
    }
    return false;
}

bool VideoConfigHelper::ResetPreviousResolution() NN_NOEXCEPT
{
    if (0 > m_PreviousActiveDisplayId)
    {
        return false;
    }
    auto result = VideoConfigUtility::GetInstance().SetActiveDisplay(m_PreviousActiveDisplayId);
    if (result)
    {
        m_PreviousActiveDisplayId = -1;
    }
    return result;
}

}
