﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// Note: DecklinkAPI_h.h requires this to be included :/
#include <objbase.h>
#include "DecklinkAPI_h.h"

// The base structure with the callbacks used below
struct DecklinkBase
{
    virtual void GotPreview(IDeckLinkVideoFrame*) {}
    virtual void Reset(BMDDisplayMode, BMDPixelFormat) {}
    virtual void GotVideo(IDeckLinkVideoInputFrame*, IDeckLinkAudioInputPacket*) {}

    virtual void Print(const char*, ...) {}
};

// Used to get a preview of the video frame
struct DecklinkPreviewCallback : public IDeckLinkScreenPreviewCallback
{
    explicit DecklinkPreviewCallback(DecklinkBase* _parent) : m_parent(_parent) {}

    // IUnknown interface
    virtual HRESULT STDMETHODCALLTYPE   QueryInterface(REFIID, LPVOID*) { return E_NOINTERFACE; }
    virtual ULONG   STDMETHODCALLTYPE   AddRef() { return 1; }
    virtual ULONG   STDMETHODCALLTYPE   Release() { return 0; }

    // IDeckLinkScreenPreviewCallback interface
    virtual HRESULT STDMETHODCALLTYPE   DrawFrame(IDeckLinkVideoFrame* videoFrame)
    {
        m_parent->GotPreview(videoFrame);

        return S_OK;
    }

private:

    DecklinkBase* m_parent;
};

// Used to get the video and audio data every frame
struct DecklinkInputCallback : public IDeckLinkInputCallback
{
    explicit DecklinkInputCallback(DecklinkBase* _parent) : m_parent(_parent) {}

    // IUnknown interface
    virtual HRESULT STDMETHODCALLTYPE QueryInterface(REFIID, LPVOID*) { return E_NOINTERFACE; }
    virtual ULONG   STDMETHODCALLTYPE AddRef() { return 1; }
    virtual ULONG   STDMETHODCALLTYPE Release() { return 0; }

    // IDeckLinkInputCallback interface
    virtual HRESULT STDMETHODCALLTYPE VideoInputFormatChanged(BMDVideoInputFormatChangedEvents, IDeckLinkDisplayMode *newDisplayMode, BMDDetectedVideoInputFormatFlags detectedSignalFlags)
    {
        // The only choices here are 480p, 720p, and 1080p
        // 480p must be YUV
        // This covers everything for NX, but may not cover everything in general
        BMDPixelFormat  pixelFormat = bmdFormat8BitYUV;
        if (detectedSignalFlags & bmdDetectedVideoInputRGB444)
        {
            pixelFormat = bmdFormat8BitBGRA;
        }

        m_parent->Print("Changing resolution to %ix%i\n", newDisplayMode->GetWidth(), newDisplayMode->GetHeight());
        BMDTimeValue frameDuration;
        BMDTimeScale timeScale;
        newDisplayMode->GetFrameRate(&frameDuration, &timeScale);
        m_parent->Print("Changing frame rate to %llu/%llu\n", frameDuration, timeScale);

        m_parent->Reset(newDisplayMode->GetDisplayMode(), pixelFormat);

        return S_OK;
    }

    virtual HRESULT STDMETHODCALLTYPE VideoInputFrameArrived(IDeckLinkVideoInputFrame* videoFrame, IDeckLinkAudioInputPacket* audioFrame)
    {
        m_parent->GotVideo(videoFrame, audioFrame);

        return S_OK;
    }

private:

    DecklinkBase* m_parent;
};
