﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>
#include <string>
#include <mutex>
#include <thread>
#include <memory>

#include "AudioWriter/AudioWriter.h"
#include "BitmapHelper/BitmapHelper.h"
#include "CaptureHelper.h"
#include "DecklinkInterface/DecklinkInterface.h"
#include "PNGHelper/PNGHelper.h"
#include "Time/Time.h"
#include "WorkerThreading/WorkerThreading.h"

#define ASSERT_SUCCEEDED(x)                                                         \
{                                                                                   \
    int hrFailure = 0;                                                              \
    if (!SUCCEEDED(hrFailure = (x)))                                                \
    {                                                                               \
        printf("Assertion on line %i failed (%i): %s\n", __LINE__, hrFailure, #x);  \
        exit(-1);                                                                   \
    }                                                                               \
}

namespace CaptureHelper
{

enum VideoOutputType
{
    VIDEO_OUTPUT_BMP,
    VIDEO_OUTPUT_PNG
};

static DecklinkInterface            g_deckLink(0);
static WorkerThreading              g_workerThreads;

static int                          g_finishedVideoFrames = 0;
static int                          g_currentVideoFrame = 0;
static int                          g_maxVideoFrames = 0;
static int                          g_finishedAudioFrames = 0;
static int                          g_currentAudioFrame = 0;
static int                          g_maxAudioFrames = 0;
static BMDTimeScale                 g_startFrameTime = 0;

static VideoOutputType              g_videoOutputType = VIDEO_OUTPUT_PNG;
static std::string                  g_outputFile = "out.png";

static std::string                  g_audioFileName = "out.wav";
static std::string                  g_videoFileNameBase = "out";

static std::mutex                   g_mutex;

static HANDLE                       g_audioHandle = nullptr;

// Handles any passive arguments (e.g. OutputFile, AudioChannels)
void HandlePassiveArguments(int argc, char** argv);

void HandleVideo(unsigned char* videoBytes, int videoWidth, int videoHeight, BMDPixelFormat pixelFormat,
                 unsigned char* audioBytes, int audioFrameCount,
                 bool droppedFrames);

void UpdateFilenames();

struct VideoData
{
    unsigned char* data;
    int width;
    int height;
    int size;
    int frame;
};

struct AudioData
{
    unsigned char* data;
    int size;
    int frame;
};

void Init(int argc, char** argv)
{
    HandlePassiveArguments(argc, argv);

    g_deckLink.StartCapture();

    // Need to wait a second for the capture to start getting frames
    Sleep(1000);
}

void Capture(int videoFrames, int audioFrames, const char* outputFile)
{
    __int64 startTime = GetTimeInMicroseconds();

    g_finishedVideoFrames = 0;
    g_currentVideoFrame = 0;
    g_maxVideoFrames = videoFrames;

    g_finishedAudioFrames = 0;
    g_currentAudioFrame = 0;
    g_maxAudioFrames = audioFrames;

    if (outputFile)
    {
        g_outputFile = outputFile;
        UpdateFilenames();
    }

    if (audioFrames > 0)
    {
        int audioDataSize = g_deckLink.GetAudioSampleSize() * g_deckLink.GetAudioBitrate() * audioFrames;

        ASSERT_SUCCEEDED(AudioWriter::InitWAV(g_audioFileName.c_str(), &g_audioHandle, audioDataSize, g_deckLink.GetAudioBitrate(), g_deckLink.GetAudioChannels(), g_deckLink.GetAudioSampleDepth()));
    }

    g_deckLink.SetVideoHandler(HandleVideo);

    while (g_finishedVideoFrames < g_maxVideoFrames || g_finishedAudioFrames < g_maxAudioFrames)
    {
        Sleep(1);
    }

    g_deckLink.SetVideoHandler(nullptr);

    if (audioFrames > 0)
    {
        CloseHandle(g_audioHandle);
    }

    __int64 endTime = GetTimeInMicroseconds();
    printf("Finished capture in %llu us\n", endTime - startTime);
}

// Handles any passive arguments (e.g. OutputFile, AudioChannels)
void HandlePassiveArguments(int argc, char** argv)
{
    // Check to see if we should just take a picture and quit
    for (int i = 1; i < argc; ++i)
    {
        if (!_stricmp(argv[i], "-OutputFile"))
        {
            if (i + 1 >= argc)
            {
                printf("Error: expected additional argument after -OutputFile\n");
                exit(-1);
            }

            ++i;
            g_outputFile = argv[i];
            UpdateFilenames();

            printf("Set Output File to %s\n", g_outputFile.c_str());
            continue;
        }
        if (!_stricmp(argv[i], "-AudioChannels"))
        {
            if (i + 1 >= argc)
            {
                printf("Error: expected additional argument after -AudioChannels\n");
                exit(-1);
            }

            ++i;
            g_deckLink.SetAudioChannels(atoi(argv[i]));

            printf("Set Audio Channels to %s\n", argv[i]);
            continue;
        }
        if (!_stricmp(argv[i], "-AudioSampleDepth"))
        {
            if (i + 1 >= argc)
            {
                printf("Error: expected additional argument after -AudioSampleDepth\n");
                exit(-1);
            }

            ++i;
            g_deckLink.SetAudioSampleDepth(atoi(argv[i]));

            printf("Set Audio Sample Depth to %s\n", argv[i]);
            continue;
        }
        if (!_stricmp(argv[i], "-AudioBitrate"))
        {
            if (i + 1 >= argc)
            {
                printf("Error: expected additional argument after -AudioBitrate\n");
                exit(-1);
            }

            ++i;
            g_deckLink.SetAudioBitrate(atoi(argv[i]));

            printf("Set Audio Bitrate to %s\n", argv[i]);
            continue;
        }
    }
}

const char* GetOutputFile()
{
    return g_outputFile.c_str();
}

void HandleAudio(void* ptr)
{
    __int64 startTime = GetTimeInMicroseconds();

    AudioData* pAudioData = (AudioData*)ptr;

    // Wait our turn
    while (g_finishedAudioFrames != pAudioData->frame)
    {
        Sleep(1);
    }

    ASSERT_SUCCEEDED(AudioWriter::AppendWAV(g_audioHandle, pAudioData->data, pAudioData->size));

    delete [] pAudioData->data;
    delete pAudioData;

    ++g_finishedAudioFrames;

    __int64 endTime = GetTimeInMicroseconds();
    printf("Handled audio in %llu us\n", endTime - startTime);
}

void HandleBitmap(void* ptr)
{
    __int64 startTime = GetTimeInMicroseconds();

    VideoData* pVideoData = (VideoData*)ptr;

    // 19 digits... 99 quintillion frames... should be fine
    char number[20];
    _itoa(pVideoData->frame, number, 10);

    std::string filename = g_videoFileNameBase + number + ".bmp";

    // If it's only a single picture, don't add a number
    if (g_maxVideoFrames == 1)
    {
        filename = g_videoFileNameBase + ".bmp";
    }

    ASSERT_SUCCEEDED(BitmapHelper::Write24(filename.c_str(), pVideoData->width, pVideoData->height, pVideoData->size, pVideoData->data));

    delete [] pVideoData->data;
    delete pVideoData;

    ++g_finishedVideoFrames;

    __int64 endTime = GetTimeInMicroseconds();
    printf("Handled bitmap in %llu us\n", endTime - startTime);
}

void HandlePNG(void* ptr)
{
    __int64 startTime = GetTimeInMicroseconds();

    VideoData* pVideoData = (VideoData*)ptr;

    std::vector<unsigned char> buffer;
    buffer.resize(pVideoData->size);

    // 19 digits... 99 quintillion frames... should be fine
    char number[20];
    _itoa(pVideoData->frame, number, 10);

    std::string filename = g_videoFileNameBase + number + ".png";

    // If it's only a single picture, don't add a number
    if (g_maxVideoFrames == 1)
    {
        filename = g_videoFileNameBase + ".png";
    }

    size_t compressedSize = PNGHelper::CompressPhase1ToPhase2(&buffer[0], pVideoData->size, pVideoData->data, pVideoData->size);

    if (compressedSize <= 0)
    {
        printf("HandlePNG Error: PNGHelper::CompressPhase1ToPhase2 failed\n");
        exit(-1);
    }

    ASSERT_SUCCEEDED(PNGHelper::WritePNG(filename.c_str(), pVideoData->width, pVideoData->height, (int)compressedSize, &buffer[0]));

    delete [] pVideoData->data;
    delete pVideoData;

    ++g_finishedVideoFrames;

    __int64 endTime = GetTimeInMicroseconds();
    printf("Handled PNG in %llu us\n", endTime - startTime);
}


void HandleVideo(unsigned char* videoBytes, int videoWidth, int videoHeight, BMDPixelFormat pixelFormat,
                 unsigned char* audioBytes, int audioFrameCount,
                 bool droppedFrames)
{
    __int64 curTime = GetTimeInMicroseconds();
    printf("- Got frame %i at %2i:%02i.%03i\n", g_currentVideoFrame, (int)(curTime / 60000000), (int)((curTime / 1000000) % 60), (int)((curTime / 1000) % 1000));

    int captureVideo = -1;
    int captureAudio = -1;

    // This technically shouldn't need a mutex to protect it since this callback should only happen one at a time, but just in case...
    {
        std::lock_guard<std::mutex> guard(g_mutex);

        if (videoBytes && g_currentVideoFrame < g_maxVideoFrames)
        {
            captureVideo = g_currentVideoFrame;
            ++g_currentVideoFrame;
        }

        if (audioBytes && g_currentAudioFrame < g_maxAudioFrames)
        {
            captureAudio = g_currentAudioFrame;
            ++g_currentAudioFrame;
        }
    }

    // Only care about dropped frames if it's not the first frame
    if (droppedFrames && (captureVideo > 0 || captureAudio > 0))
    {
        // For now, any dropped frames are unacceptable and should be aborted immediately
        printf("*** CAPTURE DROPPED A FRAME ***\n");
        exit(-1);
    }

    if (captureVideo >= 0)
    {
        VideoData* pVideoData = new VideoData;
        pVideoData->frame = captureVideo;
        pVideoData->width = videoWidth;
        pVideoData->height = videoHeight;

        switch (g_videoOutputType)
        {
        case VIDEO_OUTPUT_BMP:
            switch (pixelFormat)
            {
            case bmdFormat8BitYUV:
                BitmapHelper::Convert8BitYUVToBGR(videoBytes, pVideoData->width, pVideoData->height, &pVideoData->data, &pVideoData->size);
                break;
            case bmdFormat8BitBGRA:
                BitmapHelper::ConvertBGRAToBGR(videoBytes, pVideoData->width, pVideoData->height, &pVideoData->data, &pVideoData->size);
                break;
            default:
                printf("Invalid video pixel format %i!\n", pixelFormat);
                exit(-1);
                break;
            }

            g_workerThreads.RunFunction(HandleBitmap, (void*)pVideoData);

            break;
        case VIDEO_OUTPUT_PNG:
            switch (pixelFormat)
            {
            case bmdFormat8BitYUV:
                PNGHelper::Convert8BitYUVtoPhase1(videoBytes, pVideoData->width, pVideoData->height, &pVideoData->data, &pVideoData->size);
                break;
            case bmdFormat8BitBGRA:
                PNGHelper::ConvertBGRAtoPhase1(videoBytes, pVideoData->width, pVideoData->height, &pVideoData->data, &pVideoData->size);
                break;
            default:
                printf("Invalid video pixel format %i!\n", pixelFormat);
                exit(-1);
                break;
            }

            g_workerThreads.RunFunction(HandlePNG, (void*)pVideoData);

            break;
        default:
            printf("Invalid video output type %i!\n", g_videoOutputType);
            exit(-1);
            break;
        }
    }

    if (captureAudio >= 0)
    {
        AudioData* pAudioData = new AudioData;
        pAudioData->frame = captureAudio;

        pAudioData->size = g_deckLink.GetAudioSampleSize() * audioFrameCount;
        pAudioData->data = new BYTE[pAudioData->size];

        memcpy(pAudioData->data, audioBytes, pAudioData->size);

        g_workerThreads.RunFunction(HandleAudio, (void*)pAudioData);
    }

    __int64 endTime = GetTimeInMicroseconds();
    printf("Handled video/audio in %llu us\n", endTime - curTime);
}

void UpdateFilenames()
{
    g_videoFileNameBase = g_outputFile.substr(0, g_outputFile.find_last_of('.'));
    g_audioFileName = g_videoFileNameBase + ".wav";

    std::string base = g_outputFile.substr(g_outputFile.find_last_of('.'), std::string::npos);
    if (!_stricmp(base.c_str(), ".bmp"))
    {
        g_videoOutputType = VIDEO_OUTPUT_BMP;
    }
    else if (!_stricmp(base.c_str(), ".png"))
    {
        g_videoOutputType = VIDEO_OUTPUT_PNG;
    }
    else
    {
        printf("Error: Unsupported file extension set\n");
        printf("Supported extensions: bmp png\n");
        exit(-1);
    }
}

}
