﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winsock2.h>
#include <ws2tcpip.h>
#include <windows.h>
#include <cstdlib>
#include <cstdio>

#include "../AudioCaptureServer/ServerConstants.h"

#pragma comment (lib, "Ws2_32.lib")
#pragma comment (lib, "Mswsock.lib")
#pragma comment (lib, "AdvApi32.lib")

int GetCurrentTones(int** ppTones, const char* ipAddr, const char* port)
{
    WSADATA wsaData;
    SOCKET connectSocket = INVALID_SOCKET;

    addrinfo* addrInfo = NULL;
    addrinfo hints;

    int result;

    // Initialize Winsock
    result = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (result != 0)
    {
        printf("WSAStartup failed with error: %d\n", result);
        return -1;
    }

    ZeroMemory(&hints, sizeof(hints));
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = IPPROTO_TCP;

    // Resolve the server address and port
    result = getaddrinfo(ipAddr, port, &hints, &addrInfo);
    if (result != 0)
    {
        printf("getaddrinfo failed with error: %d\n", result);
        WSACleanup();
        return -1;
    }

    // Attempt to connect to an address until one succeeds
    for (addrinfo* ptr = addrInfo; ptr != NULL; ptr = ptr->ai_next)
    {
        // Create a socket for connecting to server
        connectSocket = socket(ptr->ai_family, ptr->ai_socktype, ptr->ai_protocol);
        if (connectSocket == INVALID_SOCKET)
        {
            printf("socket failed with error: %ld\n", WSAGetLastError());
            WSACleanup();
            return -1;
        }

        // Connect to server.
        result = connect(connectSocket, ptr->ai_addr, static_cast<int>(ptr->ai_addrlen));
        if (result != SOCKET_ERROR)
        {
            goto SocketConnected;
        }

        closesocket(connectSocket);
        connectSocket = INVALID_SOCKET;
    }

SocketConnected:
    freeaddrinfo(addrInfo);

    if (connectSocket == INVALID_SOCKET)
    {
        printf("Unable to connect to server!\n");
        WSACleanup();
        return -1;
    }

    // Shut down the connection since no more data will be sent
    result = shutdown(connectSocket, SD_SEND);
    if (result == SOCKET_ERROR)
    {
        printf("shutdown failed with error: %d\n", WSAGetLastError());
    }

    int dataCount;


    // Get the number of tones to be sent
    if (recv(connectSocket, reinterpret_cast<char*>(&dataCount), sizeof(dataCount), 0) <= 0)
    {
        printf("recv failed to get number of tones with error: %d\n", WSAGetLastError());
        closesocket(connectSocket);
        WSACleanup();
        return -1;
    }

    // Sanity check since the network can do anything
    if (dataCount < 0 || dataCount > 100000)
    {
        printf("Got absurd amount for dataCount (%d)\n", dataCount);
        closesocket(connectSocket);
        WSACleanup();
        return -1;
    }

    int totalSize = dataCount * sizeof(int);
    char* pTones = new char[totalSize];
    int curOffset = 0;

    // Receive until all the bytes are received
    do {
        result = recv(connectSocket, &pTones[curOffset], totalSize - curOffset, 0);
        if (result > 0)
            printf("Bytes received: %d\n", result);
        else if (result == 0)
            printf("Connection closed\n");
        else
            printf("recv failed with error: %d\n", WSAGetLastError());

        curOffset += result;

    } while (result > 0 && curOffset < totalSize);

    // Cleanup
    closesocket(connectSocket);
    WSACleanup();

    // If an error occurred during receiving, don't return the garbage
    if (curOffset != totalSize)
    {
        delete[] pTones;
        return -1;
    }

    *ppTones = reinterpret_cast<int*>(pTones);
    return dataCount;
}

int main(int argc, char **argv)
{
    const char* activeTone = NULL;
    const char* silentTone = NULL;
    const char* ipAddr = NULL;

    // Get the arguments
    for (int i = 1; i < argc; ++i)
    {

#define CHECK_ARG(name)                                     \
if (!_stricmp(argv[i], "-" #name))                          \
{                                                           \
        if (i + 1 >= argc)                                  \
        {                                                   \
            printf("Error: expected additional argument after -" #name "\n");   \
            return -1;                                      \
        }                                                   \
        ++i;                                                \
        name = argv[i];                                     \
        continue;                                           \
}

        CHECK_ARG(activeTone);
        CHECK_ARG(silentTone);
        CHECK_ARG(ipAddr);

#undef CHECK_ARG
    }

    if (!ipAddr)
    {
        printf("Usage:\n");
        printf("check_tone.exe -ipAddr ### [-activeTone ###] [-silentTone ###]\n");
        printf("        Verifies a certain tone is playing and/or a certain tone is NOT playing\n");
        printf("    activeTone: Returns false if this tone is NOT heard\n");
        printf("    silentTone: Returns false if this tone is heard\n");
        printf("    ipAddr:     The IP address of the server\n");
        return -1;
    }

    // Get the current tones
    int* pYesTones;
    int yesCount = GetCurrentTones(&pYesTones, ipAddr, s_PollYesPort);

    int* pNoTones;
    int noCount = GetCurrentTones(&pNoTones, ipAddr, s_PollNoPort);

    if (yesCount < 0 || noCount < 0)
    {
        printf("Failed to get tones\n");
        return 1;
    }

    printf("Got %d \"yes\" tones:\n", yesCount);

    for (int i = 0; i < yesCount; ++i)
    {
        printf("%d\n", pYesTones[i]);
    }

    printf("Got %d \"no\" tones:\n", noCount);

    for (int i = 0; i < noCount; ++i)
    {
        printf("%d\n", pNoTones[i]);
    }

    int failure = 0;

    // Confirm a certain tone is there
    if (activeTone)
    {
        int tone = atoi(activeTone);
        printf("Checking for active tone %d...\n", tone);
        ++failure;

        for (int i = 0; i < yesCount; ++i)
        {
            if (tone == pYesTones[i])
            {
                printf("  (matched active tone %d)\n", tone);
                --failure;
                break;
            }
        }
    }

    // Confirm a certain tone is NOT there
    if (silentTone)
    {
        int tone = atoi(silentTone);
        printf("Checking for silent tone %d...\n", tone);
        ++failure;

        for (int i = 0; i < noCount; ++i)
        {
            if (tone == pNoTones[i])
            {
                printf("  (matched silent tone %d)\n", tone);
                --failure;
                break;
            }
        }
    }

    return failure;
}
