﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>
#include <mmsystem.h>
#include <algorithm>

#include "ServerConstants.h"
#include "FrequencyList.h"
#include "Microphone.h"

FrequencyList s_PollYesList;
FrequencyList s_PollNoList;
FrequencyList s_ActiveList;

CRITICAL_SECTION s_MicrophoneCritical;

int s_MatchCounterList[s_MaxFrequency + 1] = { 0 };
int s_NoMatchCounterList[s_MaxFrequency + 1] = { 0 };

const int s_MaxBufferLen = 16000;

char s_ResultStringBuffer[s_MaxBufferLen];
char* s_ResultStringPtr;
int s_PrintHeader = 30;

// Uses mathemagic to determine if a tone of a certain frequency can be found here
bool CheckForTone(int tone, INT16* buffer, int size)
{
    double increment = 2 * PI * tone / (double)s_SampleRate;
    double radians = 0.0;
    double accumulateIntermodulation = 0.0;
    double accumulateQuadrature = 0.0;

    // In order to actually hear it, we need to amplify it a lot
    double multiplier = 4.0;

    for (int i = 0; i < size; ++i)
    {
        double sample = std::min(std::max(buffer[i] * multiplier, -32.0 * 1024), 32.0 * 1024);

        accumulateIntermodulation += sample * cos(radians);
        accumulateQuadrature += sample * sin(radians);
        radians += increment;
    }

    double result = sqrt(accumulateIntermodulation * accumulateIntermodulation + accumulateQuadrature * accumulateQuadrature) / size;

    s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "%6d ", (int)result);

    return (result >= 2185.0);
}

// Adds/removes tones from the list of currently playing tones
// A tone is added only after 10 consecutive matches
void CheckForTones(INT16* buffer, int size)
{
    EnterCriticalSection(&s_MicrophoneCritical);

    int* list;
    int count = s_ActiveList.GenerateArray(&list);

    if (count <= 0)
    {
        LeaveCriticalSection(&s_MicrophoneCritical);
        return;
    }

    // Print a header as needed
    if (s_PrintHeader <= 0)
    {
        s_PrintHeader = 30;
        s_ResultStringPtr = s_ResultStringBuffer;

        for (int i = 0; i < count; ++i)
        {
            s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "-------");
        }

        s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "\n");

        for (int i = 0; i < count - 1; ++i)
        {
            s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "%6d|", list[i]);
        }

        s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "%6d\n", list[count - 1]);

        for (int i = 0; i < count; ++i)
        {
            s_ResultStringPtr += sprintf_s(s_ResultStringPtr, s_MaxBufferLen + (s_ResultStringBuffer - s_ResultStringPtr), "-------");
        }

        LOG("%s\n", s_ResultStringBuffer);
    }

    --s_PrintHeader;
    LeaveCriticalSection(&s_MicrophoneCritical);
    s_ResultStringPtr = s_ResultStringBuffer;

    for (int i = 0; i < count; ++i)
    {
        if (CheckForTone(list[i], buffer, size))
        {
            // Get it closer to adding it to the yes list and remove it from the no list
            ++s_MatchCounterList[list[i]];
            if (s_MatchCounterList[list[i]] == 10)
            {
                s_MatchCounterList[list[i]] = 10;
                s_PollYesList.Add(list[i]);
                LOG("Detected %d\n", list[i]);
            }

            s_PollNoList.Remove(list[i]);
            s_NoMatchCounterList[list[i]] = 0;
        }
        else
        {
            // Get it closer to adding it to the no list and remove it from the yes list
            if (s_MatchCounterList[list[i]] >= 10)
            {
                LOG("%d no longer detected\n", list[i]);
            }
            s_PollYesList.Remove(list[i]);
            s_MatchCounterList[list[i]] = 0;

            ++s_NoMatchCounterList[list[i]];
            if (s_NoMatchCounterList[list[i]] == 10)
            {
                s_NoMatchCounterList[list[i]] = 10;
                s_PollNoList.Add(list[i]);
            }
        }
    }

    delete[] list;

    LOG("%s\n", s_ResultStringBuffer);
}

// Indicates that the header needs to be printed again
void PrintHeader()
{
    EnterCriticalSection(&s_MicrophoneCritical);
    s_PrintHeader = 0;
    LeaveCriticalSection(&s_MicrophoneCritical);
}

// Handles any initialization that needs to be done BEFORE the thread is started
void MicrophoneInit()
{
    InitializeCriticalSection(&s_MicrophoneCritical);
}

// Removes any tones not in the active list
void RemoveOldTones()
{
    int* list;
    int count = s_PollYesList.GenerateArray(&list);

    if (count <= 0)
    {
        return;
    }

    for (int i = 0; i < count; ++i)
    {
        if (!s_ActiveList.Contains(list[i]))
        {
            s_PollYesList.Remove(list[i]);
            s_MatchCounterList[list[i]] = 0;
            s_PollNoList.Remove(list[i]);
            s_NoMatchCounterList[list[i]] = 0;
        }
    }

    delete[] list;
}

// The function that indefinitely handles the microphone
void HandleMicrophone()
{
    MMRESULT ret;

    WAVEFORMATEX format = {};
    format.wFormatTag = WAVE_FORMAT_PCM;
    format.nChannels = 2;                      // Stereo
    format.nSamplesPerSec = s_SampleRate;      // 48000 Hz
    format.wBitsPerSample = 16;                // 16 bit
    format.nBlockAlign = format.wBitsPerSample * format.nChannels / 8;
    format.nAvgBytesPerSec = format.nBlockAlign * format.nSamplesPerSec;

    HWAVEIN wave;
    ret = waveInOpen(&wave, WAVE_MAPPER, &format, NULL, NULL, CALLBACK_NULL | WAVE_FORMAT_DIRECT);
    if (ret != MMSYSERR_NOERROR)
    {
        LOG("Microphone: waveInOpen failed (%d)\n", ret);
        exit(-1);
    }

    // 5 samples per second
    INT16 buffer[s_SampleRate / 5];
    WAVEHDR header = {0};

    header.lpData = reinterpret_cast<char*>(buffer);
    header.dwBufferLength = sizeof(buffer);

    ret = waveInPrepareHeader(wave, &header, sizeof(header));
    if (ret != MMSYSERR_NOERROR)
    {
        LOG("Microphone: waveInPrepareHeader failed (%d)\n", ret);
        exit(-1);
    }

    ret = waveInAddBuffer(wave, &header, sizeof(header));
    if (ret != MMSYSERR_NOERROR)
    {
        LOG("Microphone: waveInAddBuffer failed (%d)\n", ret);
        exit(-1);
    }

    ret = waveInStart(wave);
    if (ret != MMSYSERR_NOERROR)
    {
        LOG("Microphone: waveInStart failed (%d)\n", ret);
        exit(-1);
    }

    LOG("Microphone: Listening...\n");

    // This will always be true (this is not supposed to end)
    while (header.lpData)
    {
        if (header.dwFlags & WHDR_DONE)
        {
            RemoveOldTones();

            CheckForTones(reinterpret_cast<INT16*>(header.lpData), header.dwBufferLength / sizeof(*buffer));

            header.dwFlags = 0;
            header.dwBytesRecorded = 0;

            waveInPrepareHeader(wave, &header, sizeof(header));
            waveInAddBuffer(wave, &header, sizeof(header));
        }

        Sleep(1);
    }

    // Cleanup that never happens
    waveInStop(wave);
    waveInUnprepareHeader(wave, &header, sizeof(header));
    waveInClose(wave);
}
