﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winsock2.h>
#include <ws2tcpip.h>
#include <windows.h>
#include <cstdlib>
#include <cstdio>

#include "ServerConstants.h"
#include "FrequencyList.h"
#include "Microphone.h"

#pragma comment (lib, "Ws2_32.lib")
#pragma comment (lib,"winmm.lib")

CRITICAL_SECTION s_MainCritical;
int s_ActivityCounter = 0;

struct PollThreadData
{
    SOCKET socket;
    FrequencyList* list;
};

// Creates a listening socket on the specified port
SOCKET PrepareSocket(const char* port)
{
    int result;
    SOCKET listenSocket = INVALID_SOCKET;

    addrinfo* addrInfo = NULL;
    addrinfo hints;

    ZeroMemory(&hints, sizeof(hints));
    hints.ai_family = AF_INET;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_protocol = IPPROTO_TCP;
    hints.ai_flags = AI_PASSIVE;

    LOG("Preparing socket...\n");

    // Resolve the server address and port
    result = getaddrinfo(NULL, port, &hints, &addrInfo);
    if (result)
    {
        LOG("getaddrinfo failed with error: %d\n", result);
        return INVALID_SOCKET;
    }

    // Create a SOCKET for connecting to server
    listenSocket = socket(addrInfo->ai_family, addrInfo->ai_socktype, addrInfo->ai_protocol);
    if (listenSocket == INVALID_SOCKET)
    {
        LOG("socket failed with error: %ld\n", WSAGetLastError());
        freeaddrinfo(addrInfo);
        return INVALID_SOCKET;
    }

    // Setup the TCP listening socket
    result = bind(listenSocket, addrInfo->ai_addr, (int)addrInfo->ai_addrlen);
    if (result == SOCKET_ERROR)
    {
        LOG("bind failed with error: %d\n", WSAGetLastError());
        freeaddrinfo(addrInfo);
        closesocket(listenSocket);
        return INVALID_SOCKET;
    }

    freeaddrinfo(addrInfo);

    // Begin listening on the socket
    result = listen(listenSocket, SOMAXCONN);
    if (result == SOCKET_ERROR)
    {
        LOG("listen failed with error: %d\n", WSAGetLastError());
        closesocket(listenSocket);
        return INVALID_SOCKET;
    }

    return listenSocket;
}

// Enables checking a specific frequency
// Must be from 0 to 100000
void CommandToggleEnable(const char* data, int size)
{
    if (size != 4)
    {
        LOG("Command: Unexpected size for toggle enable (got %d, expected 4)\n", size);
        return;
    }

    int value = *reinterpret_cast<const int*>(data);

    if (value > s_MaxFrequency || value < 0)
    {
        LOG("Command: Rejected request to add invalid frequency %d (max %d)\n", value, s_MaxFrequency);
        return;
    }

    s_ActiveList.Add(value);
    PrintHeader();
    LOG("Command: Enabled frequency %d\n", value);
}

// Disables checking a specific frequency
void CommandToggleDisable(const char* data, int size)
{
    if (size != 4)
    {
        LOG("Command: Unexpected size for toggle disable (got %d, expected 4)\n", size);
        return;
    }

    int value = *reinterpret_cast<const int*>(data);

    s_ActiveList.Remove(value);
    PrintHeader();
    LOG("Command: Disabled frequency %d\n", value);
}

// Resets the activity counter
void ResetActivityCounter()
{
    // Reset the activity counter
    EnterCriticalSection(&s_MainCritical);
    s_ActivityCounter = 0;
    LeaveCriticalSection(&s_MainCritical);
}

// The worker thread for polling operations from clients
DWORD WINAPI HandlePollThread(LPVOID ptr)
{
    PollThreadData* pThreadData = reinterpret_cast<PollThreadData*>(ptr);
    SOCKET clientSocket = pThreadData->socket;
    FrequencyList* pList = pThreadData->list;

    delete pThreadData;

    int result;
    int* bytes;
    int dataCount = pList->GenerateArray(&bytes);

    // Send the number of bytes (excluding this byte)
    result = send(clientSocket, reinterpret_cast<char*>(&dataCount), sizeof(dataCount), 0);
    if (result == SOCKET_ERROR)
    {
        LOG("Poll: send number of bytes failed with error: %d\n", WSAGetLastError());
    }

    // Send the microphone data
    result = send(clientSocket, reinterpret_cast<char*>(bytes), dataCount * sizeof(int), 0);
    if (result == SOCKET_ERROR)
    {
        LOG("Poll: send bytes failed with error: %d\n", WSAGetLastError());
    }

    // Free the bytes
    if (dataCount)
    {
        delete[] bytes;
    }

    // Shut down the connection
    result = shutdown(clientSocket, SD_SEND);
    if (result == SOCKET_ERROR)
    {
        LOG("Poll: shutdown failed with error: %d\n", WSAGetLastError());
    }

    closesocket(clientSocket);
    return 0;
}

// The worker thread for commands from clients
DWORD WINAPI HandleCommandThread(LPVOID ptr)
{
    SOCKET clientSocket = reinterpret_cast<SOCKET>(ptr);

    int result;
    int byteCount = 0;

    // Get the number of bytes to be sent
    if (recv(clientSocket, reinterpret_cast<char*>(&byteCount), sizeof(byteCount), 0) > 0)
    {
        // Sanity check since the network can do anything
        if (byteCount > 0 && byteCount <= 100000)
        {
            char* data = new char[byteCount];
            int curOffset = 0;

            // Receive until all the bytes are received
            do {
                result = recv(clientSocket, &data[curOffset], byteCount - curOffset, 0);
                if (result > 0)
                    LOG("Command: Bytes received: %d\n", result);
                else if (result == 0)
                    LOG("Command: Connection closed\n");
                else
                    LOG("Command: recv failed with error: %d\n", WSAGetLastError());

                curOffset += result;

            } while (result > 0 && curOffset < byteCount);

            // If an error occurred during receiving, don't use the garbage
            if (curOffset == byteCount)
            {
                char command = data[0];
                switch (command)
                {
                case COMMAND_TOGGLE_ENABLE:
                    CommandToggleEnable(&data[1], byteCount - 1);
                    break;
                case COMMAND_TOGGLE_DISABLE:
                    CommandToggleDisable(&data[1], byteCount - 1);
                    break;
                default:
                    LOG("Command: Got unknown command %d\n", command);
                }
            }
            else
            {
                LOG("Command: curOffset (%d) != byteCount (%d)\n", curOffset, byteCount);
            }

            delete[] data;
        }
        else
        {
            LOG("Command: Got absurd amount for byteCount (%d)\n", byteCount);
        }
    }
    else
    {
        LOG("Command: recv failed to get data with error: %d\n", WSAGetLastError());
    }

    closesocket(clientSocket);
    return 0;
}

// The main threads that handle all polling operations from clients
DWORD WINAPI HandlePolls(LPVOID ptr)
{
    PollThreadData* pThreadData = reinterpret_cast<PollThreadData*>(ptr);
    SOCKET listenSocket = pThreadData->socket;

    // This will always be true (this is not supposed to end)
    while (listenSocket != INVALID_SOCKET)
    {
        LOG("Poll: Waiting for accept...\n");

        // Accept a client socket
        pThreadData->socket = accept(listenSocket, NULL, NULL);
        if (pThreadData->socket == INVALID_SOCKET)
        {
            LOG("Poll: accept failed with error: %d\n", WSAGetLastError());
            Sleep(1000);
            continue;
        }

        LOG("Poll: Got connection!\n");

        ResetActivityCounter();

        PollThreadData* pTemp = new PollThreadData;
        *pTemp = *pThreadData;

        CreateThread(NULL, 0, HandlePollThread, pTemp, 0, NULL);
    }

    return 0;
}

// The main thread that handles any commands from clients (e.g. toggling checking a tone)
DWORD WINAPI HandleCommands(LPVOID)
{
    SOCKET listenSocket = PrepareSocket(s_CommandPort);
    SOCKET clientSocket = INVALID_SOCKET;

    if (listenSocket == INVALID_SOCKET)
    {
        LOG("Command: Aborting\n");
        exit(-1);
    }

    // This will always be true (this is not supposed to end)
    while (listenSocket != INVALID_SOCKET)
    {
        LOG("Command: Waiting for accept...\n");

        // Accept a client socket
        clientSocket = accept(listenSocket, NULL, NULL);
        if (clientSocket == INVALID_SOCKET)
        {
            LOG("Command: accept failed with error: %d\n", WSAGetLastError());
            Sleep(1000);
            continue;
        }

        LOG("Command: Got connection!\n");

        ResetActivityCounter();

        CreateThread(NULL, 0, HandleCommandThread, reinterpret_cast<LPVOID>(clientSocket), 0, NULL);
    }

    return 0;
}

// The thread that handles the microphone
DWORD WINAPI HandleMicrophone(LPVOID)
{
    s_PollYesList.Clear();
    s_PollNoList.Clear();

    HandleMicrophone();

    return 0;
}

int main()
{
    WSADATA wsaData;
    int result;

    // Initialize Winsock
    result = WSAStartup(MAKEWORD(2, 2), &wsaData);
    if (result)
    {
        LOG("WSAStartup failed with error: %d\n", result);
        LOG("Aborting\n");
        return 1;
    }

    MicrophoneInit();
    InitializeCriticalSection(&s_MainCritical);

    PollThreadData yesPoll, noPoll;
    yesPoll.list = &s_PollYesList;
    yesPoll.socket = PrepareSocket(s_PollYesPort);
    noPoll.list = &s_PollNoList;
    noPoll.socket = PrepareSocket(s_PollNoPort);

    if (yesPoll.socket == INVALID_SOCKET || noPoll.socket == INVALID_SOCKET)
    {
        LOG("Aborting\n");
        return 1;
    }

    CreateThread(NULL, 0, HandlePolls, &yesPoll, 0, NULL);
    CreateThread(NULL, 0, HandlePolls, &noPoll, 0, NULL);
    CreateThread(NULL, 0, HandleCommands, NULL, 0, NULL);
    CreateThread(NULL, 0, HandleMicrophone, NULL, 0, NULL);

    // The main thread just waits indefinitely
    while (!result)
    {
        Sleep(60000);

        EnterCriticalSection(&s_MainCritical);
        ++s_ActivityCounter;

        // If there's been no activity for 30 minutes, clear the list of frequencies to check for
        if (s_ActivityCounter > 30)
        {
            s_ActiveList.Clear();
            s_ActivityCounter = 0;
        }
        LeaveCriticalSection(&s_MainCritical);
    }

    // Cleanup that's never actually done
    WSACleanup();

    return 0;
}
