﻿using System;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text.RegularExpressions;
using System.Threading;

namespace TargetUtility
{
    public class Program
    {
        private static void RunTargetApplication(Target target, string fileName)
        {
            const int tryCountMax = 10;

            for (var tryCount = 0; tryCount < tryCountMax; tryCount++)
            {
                var sw = new Stopwatch();
                sw.Start();
                var result = Runner.RunApplication(
                    EnvironmentInfo.RunOnTarget, $"-v -t {target.Name} {fileName}", TimeSpan.FromSeconds(180));
                sw.Stop();
                if (result.ExitCode == 0)
                {
                    var actualCount = Regex.Matches(result.StandardOutputString, "hello, world").Count;
                    const int expectedCount = 100000;
                    var resultString =
                        actualCount == expectedCount ? "Pass" : "Fail";
                    Console.WriteLine($"{target.Name,-20} [{resultString}] {actualCount}/{expectedCount} ({sw.ElapsedMilliseconds} msecs)");
                    return;
                }
            }

            throw new Exception(
                $"Failed to run target application");
        }

        private static void TestSimultaneousInitialization()
        {
            var addressPattern = "169.254.*.*";

            try
            {
                Console.WriteLine("### Reboot Target Manager");
                TargetManager.Reboot(ForInitialize: true);
                Console.WriteLine("  Done");

                Console.WriteLine("### Enumerate Targets");
                var targetList = Target.Enumerate(addressPattern);
                Console.WriteLine("  Done");

                Console.WriteLine("### Power Off All Targets.");
                targetList.Invoke(target =>
                {
                    target.PowerOff();
                    Console.WriteLine($"  Done: {target.Name}");
                });

                //Console.WriteLine("### Write Boot Config of All Targets.");
                //targetList.Invoke(target =>
                //{
                //    target.WriteProperBootConfig();
                //    Console.WriteLine($"  Done: {target.Name}");
                //});

                //Console.WriteLine("### Build firmware.");
                //Builder.BuildFirmware();

                Console.WriteLine("### Initliaze All Targets.");
                targetList.Invoke(target =>
                {
                    target.Initialize();
                    Console.WriteLine($"  Done: {target.Name}");
                });
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
            }
        }

        private static void TestSimultaneousConnection()
        {
            var addressPattern = "169.254.*.*";
            var interval = TimeSpan.FromSeconds(2);

            try
            {
                Console.WriteLine("### Reboot Target Manager");
                TargetManager.Reboot();
                Console.WriteLine("  Done");

                Console.WriteLine("### Enumerate Targets");
                var targetList = Target.Enumerate(addressPattern).Take(15); // 16 台以上で TMS 死亡。
                Console.WriteLine("  Done");

                Console.WriteLine("### Power Off All Targets.");
                targetList.Invoke(target =>
                {
                    target.PowerOff();
                    Console.WriteLine($"  Done: {target.Name}");
                });

                foreach (var i in Enumerable.Range(1, 10))
                {
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));
                    Console.WriteLine($" {i}");
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));

                    Console.WriteLine("### Connect All Targets.");
                    targetList.Invoke(target =>
                    {
                        target.Connect();
                        Console.WriteLine($"  Done: {target.Name}");
                    });

                    Thread.Sleep(interval);

                    Console.WriteLine("### Disconnect All Targets.");
                    targetList.Invoke(target =>
                    {
                        target.Disconnect();
                        Console.WriteLine($"  Done: {target.Name}");
                    });

                    Thread.Sleep(interval);
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
            }
        }

        private static void TestSimultaneousPowerCycle()
        {
            var addressPattern = "169.254.*.*";
            var interval = TimeSpan.FromSeconds(2);

            try
            {
                Console.WriteLine("### Reboot Target Manager");
                TargetManager.Reboot();
                Console.WriteLine("  Done");

                Console.WriteLine("### Enumerate Targets");
                var targetList = Target.Enumerate(addressPattern);
                Console.WriteLine("  Done");

                foreach (var i in Enumerable.Range(0, 10))
                {
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));
                    Console.WriteLine($" {i}");
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));

                    Console.WriteLine("### Reset All Targets.");
                    targetList.Invoke(target =>
                    {
                        target.Reset();
                        Console.WriteLine($"  Done: {target.Name}");
                    });

                    Console.WriteLine("### Connect All Targets.");
                    targetList.Invoke(target =>
                    {
                        target.Connect();
                        Console.WriteLine($"  Done: {target.Name}");
                    });

                    Thread.Sleep(interval);
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
            }
        }

        private static void TestSimultaneousExecution()
        {
            var addressPattern = "169.254.*.*";
            var targetApplication =
                Path.Combine(EnvironmentInfo.NintendoSdkRoot, @"Tests\Outputs\NX-NXFP2-a64\Tests\testDiag_LogOutput\Develop\testDiag_LogOutput.nsp");

            Console.WriteLine("### Reboot Target Manager");
            TargetManager.Reboot(disableLogging: true);
            Console.WriteLine("  Done");

            Console.WriteLine("### Enumerate Targets");
            var targetList = Target.Enumerate(addressPattern);
            Console.WriteLine("  Done");

            Console.WriteLine("### Start Uart Logging on All Targets.");
            targetList.Invoke(target =>
            {
                target.StartUartLogging();
                Console.WriteLine($"  Done: {target.Name}");
            });

            try
            {
                Console.WriteLine("### Power Off All Targets.");
                targetList.Invoke(target =>
                {
                    target.PowerOff();
                    Console.WriteLine($"  Done: {target.Name}");
                });

                Console.WriteLine("### Connect All Targets.");
                targetList.Invoke(target =>
                {
                    target.Connect();
                    Console.WriteLine($"  Done: {target.Name}");
                });

                Console.WriteLine("### Run Application.");
                foreach (var i in Enumerable.Range(1, 1000))
                {
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));
                    Console.WriteLine($" {i}");
                    Console.WriteLine(new string('-', Console.WindowWidth - 1));

                    targetList.Invoke(target =>
                    {
                        RunTargetApplication(target, targetApplication);
                    });
                }
            }
            catch (Exception e)
            {
                Console.WriteLine(e.ToString());
            }
            finally
            {
                targetList.Invoke(target => target.StopUartLogging());
            }

        }

        public static void Main(string[] args)
        {
            Console.WriteLine(new string('=', Console.WindowWidth - 1));
            TestSimultaneousConnection();
            Console.WriteLine(new string('=', Console.WindowWidth - 1));
            TestSimultaneousExecution();
            Console.WriteLine(new string('=', Console.WindowWidth - 1));
            TestSimultaneousPowerCycle();
            Console.WriteLine(new string('=', Console.WindowWidth - 1));
            //TestSimultaneousInitialization();
            //Console.WriteLine(new string('=', Console.WindowWidth - 1));
        }
    }
}
