﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#define NN_LOG_USE_DEFAULT_LOCALE_CHARSET

#include <cstring>
#include <nn/nn_Log.h>
#include <nnt/nntest.h>
#include "../../../../../Programs/Eris/Sources/Libraries/lm/impl/lm_LogBuffer.h"

namespace {

const char* g_ExpectedStringList[] = {
    // 入れてすぐ取り出すパターン
    "1234",
    // NULL 終端されていないパターン
    "@@@@",
    // 分割して入れるパターン
    "ABCD",
    // バッファに収まらないパターン
    "abcd",
    "efgh"
};
int g_ExpectedStringIndex;
int g_FlushFunctionCallCount;

bool LogBufferFlushFunction(const uint8_t* message, size_t messageSize)
{
    auto expectedMessage = g_ExpectedStringList[g_ExpectedStringIndex];
    auto expectedSize = std::strlen(expectedMessage);

    char string[16];
    memcpy(string, message, messageSize);
    string[messageSize] = '\0';

    NN_LOG("%s(\"%s\", %d) called when message=%s, size=%d are expected.\n",
        NN_CURRENT_FUNCTION_NAME, string, messageSize, expectedMessage, expectedSize);

    EXPECT_EQ(expectedSize, messageSize);
    EXPECT_STREQ(expectedMessage, string);
    g_FlushFunctionCallCount++;
    g_ExpectedStringIndex++;

    return true;
}

} // anonymous

// 意図した実装通りに動作するかのテスト
TEST(LogBufferTest, WhiteBoxTest)
{
    static uint8_t s_LogBufferStorage[(4 + 1) * 2];
    nn::lm::impl::LogBuffer logBuffer(s_LogBufferStorage, sizeof(s_LogBufferStorage), LogBufferFlushFunction);

    g_ExpectedStringIndex = 0;

    // ヌルポインタの入力を許容する
    EXPECT_TRUE(logBuffer.TryPush(nullptr, 0));

    // 長さ 0 の文字列の入力を許容する
    EXPECT_TRUE(logBuffer.TryPush("", 0));

    // 空のバッファからの取り出しは失敗する
    g_FlushFunctionCallCount = 0;
    EXPECT_FALSE(logBuffer.TryFlush());
    EXPECT_EQ(0, g_FlushFunctionCallCount);

    // 入れてすぐ取り出すパターン
    g_FlushFunctionCallCount = 0;
    EXPECT_TRUE(logBuffer.TryPush("1234", 4));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);

    // NULL 終端されていないパターン
    g_FlushFunctionCallCount = 0;
    EXPECT_TRUE(logBuffer.TryPush("@@@@****", 4));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);

    // 分割して入れるパターン
    g_FlushFunctionCallCount = 0;
    EXPECT_TRUE(logBuffer.TryPush("AB", 2));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryPush("CD", 2));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryPush("EF", 2));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);

    // バッファに収まらないパターン
    g_FlushFunctionCallCount = 0;
    EXPECT_TRUE(logBuffer.TryPush("abcd", 4));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryPush("efgh", 4));
    EXPECT_EQ(0, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryFlush());
    EXPECT_EQ(1, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryPush("efgh", 4));
    EXPECT_EQ(1, g_FlushFunctionCallCount);
    EXPECT_TRUE(logBuffer.TryFlush());
    EXPECT_EQ(2, g_FlushFunctionCallCount);
    EXPECT_FALSE(logBuffer.TryFlush());
    EXPECT_EQ(2, g_FlushFunctionCallCount);
}
