﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_ScopeExit.h>
#include <nnt/nntest.h>

namespace {

#if defined(NN_SDK_BUILD_DEBUG)
    const char* BuildTypeString = "Debug";
#elif defined(NN_SDK_BUILD_DEVELOP)
    const char* BuildTypeString = "Develop";
#elif defined(NN_SDK_BUILD_RELEASE)
    const char* BuildTypeString = "Release";
#else
    #error ”未サポートのビルドタイプです。”
#endif

#if defined(NN_BUILD_CONFIG_ADDRESS_32)
    const size_t AddressSize = 32;
#elif defined(NN_BUILD_CONFIG_ADDRESS_64)
    const size_t AddressSize = 64;
#else
    #error ”未サポートのアドレスサイズです。”
#endif

class LogPerformanceTest
{
private:
    using PrintFunction = void (*)(const char* string, int stringSize);

private:
    struct TestListItem
    {
        int     threadCount;
        int     stringSize;
        int64_t microSecondsPerPrint;
        double  bytesPerSeconds;
    };

private:
    static const int TestThreadCountMax = 4;
    static const int TestStringCount = 11; // 1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024
    static const int TestCount = TestThreadCountMax * TestStringCount;
    static const int TestPrintCount = 1000;
    static const int TestStringSizeMax = 1 << (TestStringCount - 1);

private:
    static const size_t ThreadStackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK char m_ThreadStacks[TestThreadCountMax][ThreadStackSize];
    nn::os::ThreadType m_Threads[TestThreadCountMax];

private:
    TestListItem m_TestList[TestCount];

private:
    char m_TestString[TestStringSizeMax];
    int m_TestStringSize;
    PrintFunction m_TestFunction;

private:
    static void ThreadFunction(void* argument) NN_NOEXCEPT
    {
        auto& test = *static_cast<LogPerformanceTest*>(argument);
        for (int j = 0; j < TestPrintCount; j++)
        {
            test.m_TestFunction(test.m_TestString, test.m_TestStringSize);
        }
    }

public:
    void Initialize() NN_NOEXCEPT
    {
        int testIndex = 0;
        for (int threadCount = 1; threadCount <= TestThreadCountMax; threadCount++)
        {
            for (int stringSize = 1; stringSize <= TestStringSizeMax; stringSize <<= 1)
            {
                m_TestList[testIndex].threadCount = threadCount;
                m_TestList[testIndex].stringSize  = stringSize;
                testIndex++;
                NN_ASSERT_LESS_EQUAL(testIndex, TestCount);
            }
        }
        NN_ASSERT_EQUAL(testIndex, TestCount);

        const int AsciiPrintableBegin = ' ';
        const int AsciiPrintableEnd = '~' + 1;
        const int AsciiPrintableCount = (AsciiPrintableEnd - AsciiPrintableBegin) + 1;

        for (int i = 0; i < sizeof(m_TestString); i++)
        {
            m_TestString[i] = AsciiPrintableBegin + (i % AsciiPrintableCount);
        }
    }

    void Run(PrintFunction function) NN_NOEXCEPT
    {
        m_TestFunction = function;

        for (auto& item : m_TestList)
        {
            const auto tmp = m_TestString[item.stringSize - 1];
            NN_UTIL_SCOPE_EXIT { m_TestString[item.stringSize - 1] = tmp; };
            m_TestString[item.stringSize - 1] = '\n'; // SIGLO-20852 対策

            m_TestStringSize = item.stringSize;

            for (auto threadIndex = 0; threadIndex < item.threadCount; threadIndex++)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Threads[threadIndex], ThreadFunction, this, m_ThreadStacks[threadIndex], sizeof(m_ThreadStacks[threadIndex]), nn::os::DefaultThreadPriority));
            }

            const auto startTick = nn::os::GetSystemTick();
            for (auto threadIndex = 0; threadIndex < item.threadCount; threadIndex++)
            {
                nn::os::StartThread(&m_Threads[threadIndex]);
            }
            for (auto threadIndex = 0; threadIndex < item.threadCount; threadIndex++)
            {
                nn::os::WaitThread(&m_Threads[threadIndex]);
            }
            const auto endTick = nn::os::GetSystemTick();
            const auto elapsed = (endTick - startTick).ToTimeSpan();

            for (auto threadIndex = 0; threadIndex < item.threadCount; threadIndex++)
            {
                nn::os::DestroyThread(&m_Threads[threadIndex]);
            }

            item.microSecondsPerPrint = elapsed.GetMicroSeconds() / TestPrintCount;
            item.bytesPerSeconds = item.stringSize / (item.microSecondsPerPrint / 1000000.0);

            nn::os::SleepThread(elapsed);
        }
    }

    void PrintResult(const char* apiName) NN_NOEXCEPT
    {
        NN_LOG("\n*%s*\n\n", apiName);
        NN_LOG(
            "| Thread count | Size (byte) | Time (usec) | Throughput (bytes/sec) |\n"
            "|:------------:|:-----------:|:-----------:|:----------------------:|\n");
        for (auto& item : m_TestList)
        {
            NN_LOG("| %12d | %11d | %11lld | %22.2lf |\n",
                item.threadCount, item.stringSize, item.microSecondsPerPrint, item.bytesPerSeconds);
        }
    }

    void PrintResultForTeamCity(const char* apiName) NN_NOEXCEPT
    {
        NN_LOG("\n");
        for (auto& item : m_TestList)
        {
            if (item.threadCount == 1)
            {
                NN_LOG("##teamcity[buildStatisticValue key='%s Execution Time (usec) @ %d Bytes, %zu bit, %s' value='%lld']\n",
                    apiName, item.stringSize, AddressSize, BuildTypeString, item.microSecondsPerPrint);
            }
        }
        NN_LOG("\n");
    }
} g_TestForNnLog, g_TestForNnPut;

const int LogPerformanceTest::TestCount;

} // anonymous

// 1~4 スレッド、1~1024 byte(s) のログを出力して、性能を測定します。
TEST(LogPerformanceTest, MeasureThroughput)
{
    g_TestForNnLog.Initialize();
    g_TestForNnLog.Run([](const char* string, int stringSize) -> void { NN_LOG("%.*s", stringSize, string); });
    g_TestForNnPut.Initialize();
    g_TestForNnPut.Run([](const char* string, int stringSize) -> void { NN_PUT(string, stringSize); });

    g_TestForNnLog.PrintResult("NN_LOG");
    g_TestForNnLog.PrintResultForTeamCity("NN_LOG");
    g_TestForNnPut.PrintResult("NN_PUT");
    g_TestForNnPut.PrintResultForTeamCity("NN_PUT");
}
