﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nnt/nntest.h>
#include <nn/nn_Log.h>
#include <nn/nn_SdkLog.h>
#include <nn/diag.h>

// 現在ログは標準出力に出力されるため、
// 出力されること、出力が正しいことは自動テストできません。

namespace
{

int g_LogObserverResults[4];
int g_LogObserverValue;

nn::diag::LogMetaData g_LogMetaData;

const size_t LogMessageBufferLength = 512;
char g_LogMessageBuffer[LogMessageBufferLength];

void LogObserver0(const nn::diag::LogMetaData&, const nn::diag::LogBody&, void* argument)
{
    *reinterpret_cast<int*>(argument) = g_LogObserverValue;
    g_LogObserverResults[0] = g_LogObserverValue;
}
void LogObserver1(const nn::diag::LogMetaData&, const nn::diag::LogBody&, void* argument)
{
    *reinterpret_cast<int*>(argument) = g_LogObserverValue;
    g_LogObserverResults[1] = g_LogObserverValue;
}
void LogObserver2(const nn::diag::LogMetaData&, const nn::diag::LogBody&, void* argument)
{
    *reinterpret_cast<int*>(argument) = g_LogObserverValue;
    g_LogObserverResults[2] = g_LogObserverValue;
}
void LogObserver3(const nn::diag::LogMetaData&, const nn::diag::LogBody&, void* argument)
{
    *reinterpret_cast<int*>(argument) = g_LogObserverValue;
    g_LogObserverResults[3] = g_LogObserverValue;
}

// ０と１は static な場所にホルダを置く
nn::diag::LogObserverHolder g_logObserverHolder0;
nn::diag::LogObserverHolder g_logObserverHolder1;

int g_LogObserverArgument0;
int g_LogObserverArgument1;

// ローカル変数をクリアするためにマクロになっています
#define CLEAR_ARGUMENTS()           \
do {                                \
    g_LogObserverArgument0 = 0;     \
    g_LogObserverArgument1 = 0;     \
    logObserverArgument2   = 0;     \
    logObserverArgument3   = 0;     \
} while (NN_STATIC_CONDITION(false))

void ClearResults()
{
    for(int index = 0; index < 4; ++index)
    {
        g_LogObserverResults[index] = 0;
    }
}

}

TEST(LogTest, Interface)
{
    NN_LOG("no argument\n");
    NN_LOG("decimal: %d\n", 1);
    NN_PUT("1234567\n", 8);

    NN_SDK_LOG("no argument\n");
    NN_SDK_LOG("decimal: %d\n", 1);
    NN_SDK_PUT("1234567\n", 8);
}

namespace {
    const char* g_CallVlogName = "";
    int g_ExpectLineNumberByVlog = 0;

    void CallVlog(const char* format, ...)
    {
        g_CallVlogName = NN_CURRENT_FUNCTION_NAME;
        va_list args;
        va_start(args, format);
        NN_VLOG(format, args); g_ExpectLineNumberByVlog = __LINE__;
        va_end(args);
    };

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    const char* g_CallSdkVlogName = "";
    int g_ExpectLineNumberBySdkVlog = 0;

    void CallSdkVlog(const char* format, ...)
    {
        g_CallSdkVlogName = NN_CURRENT_FUNCTION_NAME;
        va_list args;
        va_start(args, format);
        NN_SDK_VLOG(format, args); g_ExpectLineNumberBySdkVlog = __LINE__;
        va_end(args);
    };
#endif // defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
}

TEST(LogTest, RegisterObserver)
{
    // ２と３はスタックにホルダを置く
    nn::diag::LogObserverHolder logObserverHolder2;
    nn::diag::LogObserverHolder logObserverHolder3;

    int logObserverArgument2 = 2;
    int logObserverArgument3 = 3;

    nn::diag::InitializeLogObserverHolder(&g_logObserverHolder0, LogObserver0, &g_LogObserverArgument0);
    nn::diag::InitializeLogObserverHolder(&g_logObserverHolder1, LogObserver1, &g_LogObserverArgument1);
    nn::diag::InitializeLogObserverHolder(&logObserverHolder2, LogObserver2, &logObserverArgument2);
    nn::diag::InitializeLogObserverHolder(&logObserverHolder3, LogObserver3, &logObserverArgument3);


    g_LogObserverValue = 0;

    // すべて追加する
    nn::diag::RegisterLogObserver(&g_logObserverHolder0);
    nn::diag::RegisterLogObserver(&g_logObserverHolder1);
    nn::diag::RegisterLogObserver(&logObserverHolder2);
    nn::diag::RegisterLogObserver(&logObserverHolder3);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[0]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[1]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[2]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[3]);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument0);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument1);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument2);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument3);

    // 途中のノードを取り外す
    nn::diag::UnregisterLogObserver(&g_logObserverHolder1);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[0]);
    EXPECT_EQ(0                 , g_LogObserverResults[1]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[2]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[3]);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument0);
    EXPECT_EQ(0                 , g_LogObserverArgument1);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument2);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument3);

    // 末尾のノードを取り外す
    nn::diag::UnregisterLogObserver(&logObserverHolder3);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[0]);
    EXPECT_EQ(0                 , g_LogObserverResults[1]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[2]);
    EXPECT_EQ(0                 , g_LogObserverResults[3]);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument0);
    EXPECT_EQ(0                 , g_LogObserverArgument1);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument2);
    EXPECT_EQ(0                 , logObserverArgument3);

    // 先頭のノードを取り外す
    nn::diag::UnregisterLogObserver(&g_logObserverHolder0);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(0                 , g_LogObserverResults[0]);
    EXPECT_EQ(0                 , g_LogObserverResults[1]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[2]);
    EXPECT_EQ(0                 , g_LogObserverResults[3]);

    EXPECT_EQ(0                 , g_LogObserverArgument0);
    EXPECT_EQ(0                 , g_LogObserverArgument1);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument2);
    EXPECT_EQ(0                 , logObserverArgument3);


    // 唯一のノードを取り外す
    nn::diag::UnregisterLogObserver(&logObserverHolder2);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(0                 , g_LogObserverResults[0]);
    EXPECT_EQ(0                 , g_LogObserverResults[1]);
    EXPECT_EQ(0                 , g_LogObserverResults[2]);
    EXPECT_EQ(0                 , g_LogObserverResults[3]);

    EXPECT_EQ(0                 , g_LogObserverArgument0);
    EXPECT_EQ(0                 , g_LogObserverArgument1);
    EXPECT_EQ(0                 , logObserverArgument2);
    EXPECT_EQ(0                 , logObserverArgument3);

    // 再度すべて追加する
    nn::diag::RegisterLogObserver(&g_logObserverHolder0);
    nn::diag::RegisterLogObserver(&g_logObserverHolder1);
    nn::diag::RegisterLogObserver(&logObserverHolder2);
    nn::diag::RegisterLogObserver(&logObserverHolder3);

    ClearResults();
    CLEAR_ARGUMENTS();
    g_LogObserverValue++;
    NN_LOG("set value %d\n", g_LogObserverValue);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[0]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[1]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[2]);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverResults[3]);

    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument0);
    EXPECT_EQ(g_LogObserverValue, g_LogObserverArgument1);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument2);
    EXPECT_EQ(g_LogObserverValue, logObserverArgument3);

    // テストで追加したログオブザーバをすべて削除する
    nn::diag::UnregisterLogObserver(&g_logObserverHolder0);
    nn::diag::UnregisterLogObserver(&g_logObserverHolder1);
    nn::diag::UnregisterLogObserver(&logObserverHolder2);
    nn::diag::UnregisterLogObserver(&logObserverHolder3);
}

TEST(LogTest, HandleCallback)
{
    // テスト用オブザーバーの登録
    nn::diag::LogObserverHolder logObserverHolder;
    nn::diag::InitializeLogObserverHolder(
        &logObserverHolder,
        []( const nn::diag::LogMetaData& metaData,
            const nn::diag::LogBody& body,
            void* /*argument*/)
        {
            g_LogMetaData = metaData;
            const size_t writeBytes = std::min(LogMessageBufferLength, body.messageBytes);
            std::memcpy(
                g_LogMessageBuffer, body.message, writeBytes);
            g_LogMessageBuffer[writeBytes] = '\0';
        },
        NULL);
    nn::diag::RegisterLogObserver(&logObserverHolder);

    // -------------------------------------------------------------------------
    // NN_LOG() のテスト
    // -------------------------------------------------------------------------
    NN_LOG("no argument by NN_LOG\n");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMessageBuffer, "no argument by NN_LOG\n");

    NN_LOG("decimal: %d, string: %s by NN_LOG\n", 1, "test");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMessageBuffer, "decimal: 1, string: test by NN_LOG\n");

    // -------------------------------------------------------------------------
    // NN_VLOG() のテスト
    // -------------------------------------------------------------------------
    CallVlog("no argument by NN_VLOG\n");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, g_ExpectLineNumberByVlog);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, g_CallVlogName);
    EXPECT_STREQ(g_LogMessageBuffer, "no argument by NN_VLOG\n");

    CallVlog("decimal: %d, string: %s by NN_VLOG\n", 1, "test");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, g_ExpectLineNumberByVlog);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, g_CallVlogName);
    EXPECT_STREQ(g_LogMessageBuffer, "decimal: 1, string: test by NN_VLOG\n");

    // -------------------------------------------------------------------------
    // NN_PUT() のテスト
    // -------------------------------------------------------------------------
    NN_PUT("1234567\n89", 8);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMessageBuffer, "1234567\n");

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    // -------------------------------------------------------------------------
    // NN_SDK_LOG() のテスト
    // -------------------------------------------------------------------------
    NN_SDK_LOG("no argument by NN_SDK_LOG\n");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, NN_CURRENT_FUNCTION_NAME);
    EXPECT_STREQ(g_LogMessageBuffer, "no argument by NN_SDK_LOG\n");

    NN_SDK_LOG("decimal: %d, string: %s by NN_SDK_LOG\n", 1, "test");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, NN_CURRENT_FUNCTION_NAME);
    EXPECT_STREQ(g_LogMessageBuffer, "decimal: 1, string: test by NN_SDK_LOG\n");

    // -------------------------------------------------------------------------
    // NN_SDK_VLOG() のテスト
    // -------------------------------------------------------------------------
    CallSdkVlog("no argument by NN_SDK_VLOG\n");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, g_ExpectLineNumberBySdkVlog);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, g_CallSdkVlogName);
    EXPECT_STREQ(g_LogMessageBuffer, "no argument by NN_SDK_VLOG\n");

    CallSdkVlog("decimal: %d, string: %s by NN_SDK_VLOG\n", 1, "test");
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, g_ExpectLineNumberBySdkVlog);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.functionName, g_CallSdkVlogName);
    EXPECT_STREQ(g_LogMessageBuffer, "decimal: 1, string: test by NN_SDK_VLOG\n");

    // -------------------------------------------------------------------------
    // NN_SDK_PUT() のテスト
    // -------------------------------------------------------------------------
    NN_SDK_PUT("1234567\n89", 8);
    EXPECT_EQ(g_LogMetaData.sourceInfo.lineNumber, __LINE__ - 1);
    EXPECT_STREQ(g_LogMetaData.sourceInfo.fileName, __FILE__);
    EXPECT_STREQ(g_LogMessageBuffer, "1234567\n");

#endif // #if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)

    // テスト用オブザーバーの登録解除
    nn::diag::UnregisterLogObserver(&logObserverHolder);
}

int g_SequenceNumber;
int g_LastSequenceNumber;
bool g_IsHeadResults[5];
bool g_IsTailResults[5];
size_t g_MessageBytesResults[5];

TEST(LogTest, LogFragmentation)
{
    static char s_SourceString[512];
    for (int i = 0; i < sizeof(s_SourceString) - 1; i++)
    {
        s_SourceString[i] = '0' + (i % 10);
    }
    s_SourceString[sizeof(s_SourceString) - 1] = '\0';

    // テスト用オブザーバーの登録
    nn::diag::LogObserverHolder logObserverHolder;
    nn::diag::InitializeLogObserverHolder(
        &logObserverHolder,
        []( const nn::diag::LogMetaData& /*metaData*/,
            const nn::diag::LogBody& body,
            void* /*argument*/)
        {
            g_LastSequenceNumber = g_SequenceNumber;
            g_IsHeadResults[g_SequenceNumber] = body.isHead;
            g_IsTailResults[g_SequenceNumber] = body.isTail;
            g_MessageBytesResults[g_SequenceNumber] = body.messageBytes;
            g_SequenceNumber++;
        },
        NULL);
    nn::diag::RegisterLogObserver(&logObserverHolder);

    char c;

    // ログがギリギリ分割されないパターン（127 Bytes）
    g_SequenceNumber = 0;
    c = s_SourceString[127];
    s_SourceString[127] = '\0';
    g_LastSequenceNumber = -1;
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = false;
    g_MessageBytesResults[0] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(0, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_TRUE(g_IsTailResults[0]);
    EXPECT_EQ(g_MessageBytesResults[0], 127);
    s_SourceString[127] = c;

    // ログがギリギリ分割されるパターン（128 Bytes）
    g_SequenceNumber = 0;
    c = s_SourceString[128];
    s_SourceString[128] = '\0';
    g_LastSequenceNumber = -1;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = false;
    g_MessageBytesResults[0] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(0, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_TRUE(g_IsTailResults[0]);
    EXPECT_EQ(g_MessageBytesResults[0], 128);
#else
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = true;
    g_IsHeadResults[1] = true;
    g_IsTailResults[1] = false;
    g_MessageBytesResults[0] = 0;
    g_MessageBytesResults[1] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(1, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_FALSE(g_IsTailResults[0]);
    EXPECT_FALSE(g_IsHeadResults[1]);
    EXPECT_TRUE(g_IsTailResults[1]);
    EXPECT_EQ(g_MessageBytesResults[0], 127);
    EXPECT_EQ(g_MessageBytesResults[1], 1);
#endif
    s_SourceString[128] = c;

    // ログの分割がギリギリ 2 つで済むパターン（254 Bytes）
    g_SequenceNumber = 0;
    c = s_SourceString[254];
    s_SourceString[254] = '\0';
    g_LastSequenceNumber = -1;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = false;
    g_MessageBytesResults[0] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(0, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_TRUE(g_IsTailResults[0]);
    EXPECT_EQ(g_MessageBytesResults[0], 254);
#else
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = true;
    g_IsHeadResults[1] = true;
    g_IsTailResults[1] = false;
    g_MessageBytesResults[0] = 0;
    g_MessageBytesResults[1] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(1, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_FALSE(g_IsTailResults[0]);
    EXPECT_FALSE(g_IsHeadResults[1]);
    EXPECT_TRUE(g_IsTailResults[1]);
    EXPECT_EQ(g_MessageBytesResults[0], 127);
    EXPECT_EQ(g_MessageBytesResults[1], 127);
#endif
    s_SourceString[254] = c;

    // ログの分割がギリギリ 3 になるパターン（255 Bytes）
    g_SequenceNumber = 0;
    c = s_SourceString[255];
    s_SourceString[255] = '\0';
    g_LastSequenceNumber = -1;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = false;
    g_MessageBytesResults[0] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(0, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_TRUE(g_IsTailResults[0]);
    EXPECT_EQ(g_MessageBytesResults[0], 255);
#else
    g_IsHeadResults[0] = false;
    g_IsTailResults[0] = true;
    g_IsHeadResults[1] = true;
    g_IsTailResults[1] = true;
    g_IsHeadResults[1] = true;
    g_IsTailResults[1] = false;
    g_MessageBytesResults[0] = 0;
    g_MessageBytesResults[1] = 0;
    g_MessageBytesResults[2] = 0;
    NN_LOG("%s", s_SourceString);
    EXPECT_EQ(2, g_LastSequenceNumber);
    EXPECT_TRUE(g_IsHeadResults[0]);
    EXPECT_FALSE(g_IsTailResults[0]);
    EXPECT_FALSE(g_IsHeadResults[1]);
    EXPECT_FALSE(g_IsTailResults[1]);
    EXPECT_FALSE(g_IsHeadResults[2]);
    EXPECT_TRUE(g_IsTailResults[2]);
    EXPECT_EQ(g_MessageBytesResults[0], 127);
    EXPECT_EQ(g_MessageBytesResults[1], 127);
    EXPECT_EQ(g_MessageBytesResults[2], 1);
#endif
    s_SourceString[255] = c;

    // テスト用オブザーバーの登録解除
    nn::diag::UnregisterLogObserver(&logObserverHolder);
} //NOLINT(impl/function_size)

TEST(NnLogTest, SuperLong)
{
    static char string[512 * 1024];
    const int length = sizeof(string) - 1;

    for (int i = 0; i < length - 1; i++)
    {
        string[i] = '0' + (i % 10);
    }
    string[length - 1] = '\n';
    string[length] = '\0';

    NN_LOG("Start to NN_LOG\n");
    NN_LOG("%s", string);
    NN_LOG("Finished to NN_LOG\n\n");
}

TEST(NnPutTest, SuperLong)
{
    static char string[512 * 1024];
    const int length = sizeof(string) - 1;

    for (int i = 0; i < length - 1; i++)
    {
        string[i] = '0' + (i % 10);
    }
    string[length - 1] = '\n';
    string[length] = '\0';

    NN_LOG("Start to NN_PUT\n");
    NN_PUT(string, length);
    NN_LOG("Finished to NN_PUT\n\n");
}
