﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nnt/nntest.h>
#include <nn/nn_Abort.h>

// ExpectFailureAssertReferenceNotNull に渡すためのヌルポインタ。
// 最適化で ヌルポインタであると判定されにくくするため、non-const にしてグローバルに置いておく。
int* g_NullPtrToInt = NULL;

// インターフェースが揃っていることをテスト
TEST(AbortTest, Interface)
{
    NN_ABORT_UNLESS(true);
    NN_ABORT_UNLESS(true, "failed");
}

// 条件が成立しないときに終了することをテスト
TEST(AbortDeathTest, ConditionFailure)
{
    EXPECT_DEATH(
        NN_ABORT_UNLESS(false),
        "");
}

// ヌルポインタをデリファレンスした参照を受け取り、それを NN_ABORT_UNLESS_NOT_NULL に渡して abort させる。
//
// ヌルポインタをデリファレンスする操作は未定義動作になるため、
// コンパイラによっては参照がヌルポインタではないことを仮定して最適化を行うものがある（例: ARMCC）。
// その場合は abort しない（テストが失敗する）。
// そのようなコンパイラが採用された場合、何らかの対処を行うべきである。
NN_NOINLINE void ExpectFailureAssertReferenceNotNull(int &n)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_NOT_NULL(&n), "");
}

// 意味づけされた Assert マクロのテスト
TEST(AbortDeathTest, AssertNotNull)
{
    {
        const int n = 0;
        NN_ABORT_UNLESS_NOT_NULL(&n);
    }

    EXPECT_DEATH(NN_ABORT_UNLESS_NOT_NULL(static_cast<void*>(NULL)), "");

    ExpectFailureAssertReferenceNotNull(*g_NullPtrToInt);
}

TEST(AbortDeathTest, AssertEqual)
{
    NN_ABORT_UNLESS_EQUAL(1, 1);
    NN_ABORT_UNLESS_EQUAL(-1, -1);
    EXPECT_DEATH(NN_ABORT_UNLESS_EQUAL(1, -1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_EQUAL(0, -1), "");
}

TEST(AbortDeathTest, AssertNotEqual)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_NOT_EQUAL(1, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_NOT_EQUAL(-1, -1), "");
    NN_ABORT_UNLESS_NOT_EQUAL(1, -1);
    NN_ABORT_UNLESS_NOT_EQUAL(0, -1);
}

TEST(AbortDeathTest, AssertLess)
{
    NN_ABORT_UNLESS_LESS(0, 1);
    NN_ABORT_UNLESS_LESS(-1, 1);
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS(1, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS(-1, -1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS(1, -1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS(0, -1), "");
}

TEST(AbortDeathTest, AssertLessEqual)
{
    NN_ABORT_UNLESS_LESS_EQUAL(0, 1);
    NN_ABORT_UNLESS_LESS_EQUAL(-1, 1);
    NN_ABORT_UNLESS_LESS_EQUAL(1, 1);
    NN_ABORT_UNLESS_LESS_EQUAL(-1, -1);
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS_EQUAL(1, -1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_LESS_EQUAL(0, -1), "");
}

TEST(AbortDeathTest, AssertGreater)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER(0, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER(-1, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER(1, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER(-1, -1), "");
    NN_ABORT_UNLESS_GREATER(1, -1);
    NN_ABORT_UNLESS_GREATER(0, -1);
}

TEST(AbortDeathTest, AssertGreaterEqual)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER_EQUAL(0, 1), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_GREATER_EQUAL(-1, 1), "");
    NN_ABORT_UNLESS_GREATER_EQUAL(1, 1);
    NN_ABORT_UNLESS_GREATER_EQUAL(-1, -1);
    NN_ABORT_UNLESS_GREATER_EQUAL(1, -1);
    NN_ABORT_UNLESS_GREATER_EQUAL(0, -1);
}

TEST(AbortDeathTest, AssertStringEqual)
{
    NN_ABORT_UNLESS_STRING_EQUAL("a", "a");
    NN_ABORT_UNLESS_STRING_EQUAL("abc", "abc");
    NN_ABORT_UNLESS_STRING_EQUAL("abc\n", "abc\n");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL(nullptr, nullptr), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL(nullptr, "a"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL("a", nullptr), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL("a", "b"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL("abc", "xyz\n"), "");
}

TEST(AbortDeathTest, AssertStringNotEqual)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_NOT_EQUAL("a", "a"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_NOT_EQUAL("abc", "abc"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_NOT_EQUAL("abc\n", "abc\n"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL(nullptr, nullptr), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL(nullptr, "a"), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_STRING_EQUAL("a", nullptr), "");
    NN_ABORT_UNLESS_STRING_NOT_EQUAL("a", "b");
    NN_ABORT_UNLESS_STRING_NOT_EQUAL("abc", "xyz\n");
}

TEST(AbortDeathTest, AssertAligned)
{
    NN_ABORT_UNLESS_ALIGNED(1, 1);
    NN_ABORT_UNLESS_ALIGNED(2, 1);
    NN_ABORT_UNLESS_ALIGNED(12, 4);
    EXPECT_DEATH(NN_ABORT_UNLESS_ALIGNED(12, 8), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_ALIGNED(32, 64), "");
}

TEST(AbortDeathTest, AssertRange)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_RANGE(0, 1, 16), "");
    NN_ABORT_UNLESS_RANGE(1, 1, 16);
    NN_ABORT_UNLESS_RANGE(2, 1, 16);
    NN_ABORT_UNLESS_RANGE(15, 1, 16);
    EXPECT_DEATH(NN_ABORT_UNLESS_RANGE(16, 1, 16), "");
    EXPECT_DEATH(NN_ABORT_UNLESS_RANGE(17, 1, 16), "");
}

TEST(AbortDeathTest, AssertMinMax)
{
    EXPECT_DEATH(NN_ABORT_UNLESS_MINMAX(0, 1, 16), "");
    NN_ABORT_UNLESS_MINMAX(1, 1, 16);
    NN_ABORT_UNLESS_MINMAX(2, 1, 16);
    NN_ABORT_UNLESS_MINMAX(15, 1, 16);
    NN_ABORT_UNLESS_MINMAX(16, 1, 16);
    EXPECT_DEATH(NN_ABORT_UNLESS_MINMAX(17, 1, 16), "");
}

// 条件が成立しないときにメッセージを出力して終了することをテスト
TEST(AbortDeathTest, ConditionFailureMessage)
{
    EXPECT_DEATH(
        NN_ABORT_UNLESS(false, ""),
        "");
}

// 条件が成立するときに正常終了（終了コード 0）する
void ConditionSuccessAndExit0()
{
    NN_ABORT_UNLESS(true);
    NN_ABORT_UNLESS(true, "");
    std::exit(0);
}

// 条件が成立する場合に異常終了しないことをテスト
TEST(AbortDeathTest, ConditionSuccess)
{
    EXPECT_EXIT(
        ConditionSuccessAndExit0(),
        [](int exitCode) { return exitCode == 0; },
        "");
}

// NN_ABORT マクロが終了することをテスト
TEST(AbortDeathTest, Abort)
{
    EXPECT_DEATH(
        NN_ABORT("abort test"),
        "");
}
