﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

//-----------------------------------------------------------------------------
//  デバイスツリーの API のテスト
//-----------------------------------------------------------------------------

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

#include "testDt_Common.h"

namespace nnt { namespace dt {

TEST(PropertyApi, IsPropertyExist)
{
    nn::Result result;
    nn::dt::Node node;
    bool exist;

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::IsPropertyExist(&exist, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_TRUE(exist);

    ASSERT_TRUE(nn::dt::IsPropertyExist(&node, "property_1byte"));

    result = nn::dt::IsPropertyExist(&exist, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_FALSE(exist);

    ASSERT_FALSE(nn::dt::IsPropertyExist(&node, "not_exist"));

    // 異常系
    result = nn::dt::IsPropertyExist(nullptr, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::IsPropertyExist(&exist, nullptr, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    ASSERT_FALSE(nn::dt::IsPropertyExist(nullptr, "property_1byte"));

    result = nn::dt::IsPropertyExist(&exist, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    ASSERT_FALSE(nn::dt::IsPropertyExist(&node, nullptr));
}

TEST(PropertyApi, GetPropertySize)
{
    nn::Result result;
    nn::dt::Node node;
    size_t size;

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertySize(&size, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, size);

    result = nn::dt::GetPropertySize(&size, &node, "property_2byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, size);

    // 異常系
    result = nn::dt::GetPropertySize(&size, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetPropertySize(nullptr, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertySize(&size, nullptr, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertySize(&size, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetPropertyCount)
{
    nn::Result result;
    nn::dt::Node node;
    int count;

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertyCount<int8_t>(&count, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(8, count);

    result = nn::dt::GetPropertyCount<int16_t>(&count, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(4, count);

    result = nn::dt::GetPropertyCount<int32_t>(&count, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, count);

    result = nn::dt::GetPropertyCount<int64_t>(&count, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, count);

    // 異常系
    result = nn::dt::GetPropertyCount<int8_t>(&count, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetPropertyCount<int16_t>(&count, &node, "property_3byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetPropertyCount<int16_t>(nullptr, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyCount<int32_t>(&count, nullptr, "property_8byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyCount<int64_t>(&count, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetProperty)
{
    nn::Result result;
    nn::dt::Node node;

    uint8_t val8;
    uint16_t val16;
    uint32_t val32;
    uint64_t val64;

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系 (without index)
    result = nn::dt::GetProperty(&val8, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x00, val8);

    result = nn::dt::GetProperty(&val16, &node, "property_2byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0001, val16);

    result = nn::dt::GetProperty(&val32, &node, "property_4byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x00010203, val32);

    result = nn::dt::GetProperty(&val64, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0001020304050607, val64);

    // 正常系 (with index)
    result = nn::dt::GetProperty(&val8, &node, "property_5byte", 3);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x03, val8);

    result = nn::dt::GetProperty(&val16, &node, "property_6byte", 1);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0203, val16);

    result = nn::dt::GetProperty(&val32, &node, "property_8byte", 0);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x00010203, val32);

    result = nn::dt::GetProperty(&val64, &node, "property_8byte", 0);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0001020304050607, val64);

    // 異常系 (without index)

    // ResultPropertyTypeError (実際のプロパティサイズが、取得するプロパティのサイズより小さい)
    result = nn::dt::GetProperty(&val64, &node, "property_4byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyTypeError (実際のプロパティのサイズが、取得するプロパティのサイズの N 倍ではない)
    result = nn::dt::GetProperty(&val16, &node, "property_7byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyNotExist
    result = nn::dt::GetProperty(&val64, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    // ResultInvalidArgument
    result = nn::dt::GetProperty(&val8, nullptr, "property_4byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetProperty(&val16, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    // 異常系 (with index)

    // ResultPropertyTypeError (実際のプロパティサイズが、取得するプロパティのサイズより小さい)
    result = nn::dt::GetProperty(&val64, &node, "property_4byte", 0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyTypeError (実際のプロパティのサイズが、取得するプロパティのサイズの N 倍ではない)
    result = nn::dt::GetProperty(&val16, &node, "property_7byte", 1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyNotExist
    result = nn::dt::GetProperty(&val8, &node, "property_4byte", 6);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetProperty(&val32, &node, "not_exist", 0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    // ResultInvalidArgument
    result = nn::dt::GetProperty(&val8, nullptr, "property_4byte", 1);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetProperty(&val16, &node, nullptr, 0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetPropertyArray)
{
    nn::Result result;
    nn::dt::Node node;

    uint8_t array8[8];
    uint16_t array16[4];
    uint32_t array32[2];
    uint64_t array64[1];

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertyArray(array8, 5, &node, "property_5byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x00, array8[0]);
    ASSERT_EQ(0x01, array8[1]);
    ASSERT_EQ(0x02, array8[2]);
    ASSERT_EQ(0x03, array8[3]);
    ASSERT_EQ(0x04, array8[4]);

    result = nn::dt::GetPropertyArray(array16, 3, &node, "property_6byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0001, array16[0]);
    ASSERT_EQ(0x0203, array16[1]);
    ASSERT_EQ(0x0405, array16[2]);

    result = nn::dt::GetPropertyArray(array32, 2, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x00010203, array32[0]);
    ASSERT_EQ(0x04050607, array32[1]);

    result = nn::dt::GetPropertyArray(array64, 1, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(0x0001020304050607, array64[0]);

    // 異常系

    // ResultPropertyNotExist
    result = nn::dt::GetPropertyArray(array8, 8, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    // ResultPropertyTypeError (実際のプロパティサイズが、取得するプロパティのサイズより小さい)
    result = nn::dt::GetPropertyArray(array64, 1, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyTypeError (実際のプロパティのサイズが、取得するプロパティのサイズより大きい)
    result = nn::dt::GetPropertyArray(array32, 1, &node, "property_5byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultSizeIncorrect
    result = nn::dt::GetPropertyArray(array8, 8, &node, "property_4byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);

    // ResultInvalidArgument
    result = nn::dt::GetPropertyArray(array8, 8, nullptr, "property_8byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyArray(array8, 8, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetPropertyList)
{
    nn::Result result;
    nn::dt::Node node;

    int count;
    uint8_t list8[8];
    uint16_t list16[4];
    uint32_t list32[2];
    uint64_t list64[1];

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertyList(list8, &count, 8, &node, "property_5byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(5, count);
    ASSERT_EQ(0x00, list8[0]);
    ASSERT_EQ(0x01, list8[1]);
    ASSERT_EQ(0x02, list8[2]);
    ASSERT_EQ(0x03, list8[3]);
    ASSERT_EQ(0x04, list8[4]);

    result = nn::dt::GetPropertyList(list16, &count, 4, &node, "property_6byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(3, count);
    ASSERT_EQ(0x0001, list16[0]);
    ASSERT_EQ(0x0203, list16[1]);
    ASSERT_EQ(0x0405, list16[2]);

    result = nn::dt::GetPropertyList(list32, &count, 2, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, count);
    ASSERT_EQ(0x00010203, list32[0]);
    ASSERT_EQ(0x04050607, list32[1]);

    result = nn::dt::GetPropertyList(list64, &count, 1, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, count);
    ASSERT_EQ(0x0001020304050607, list64[0]);

    // バッファがプロパティのサイズより小さい場合
    result = nn::dt::GetPropertyList(list8, &count, 3, &node, "property_4byte");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(3, count);
    ASSERT_EQ(0x00, list8[0]);
    ASSERT_EQ(0x01, list8[1]);
    ASSERT_EQ(0x02, list8[2]);

    // 異常系

    // ResultPropertyNotExist
    result = nn::dt::GetPropertyList(list8, &count, 8, &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    // ResultPropertyTypeError (実際のプロパティサイズが、取得するプロパティのサイズより小さい)
    result = nn::dt::GetPropertyList(list64, &count, 1, &node, "property_1byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultPropertyTypeError (実際のプロパティのサイズが、取得するプロパティのサイズの N 倍ではない)
    result = nn::dt::GetPropertyList(list32, &count, 2, &node, "property_5byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    // ResultInvalidArgument
    result = nn::dt::GetPropertyList(list8, nullptr, 8, &node, "property_8byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyList(list8, &count, 8, nullptr, "property_8byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyList(list8, &count, 8, &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetPropertyString)
{
    nn::Result result;
    nn::dt::Node node;
    size_t propertySize;

    char buffer[64];
    const char emptyString[] = "";
    const char sampleString[] = "string";

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "empty_string");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(sizeof(emptyString), propertySize);
    ASSERT_STREQ(emptyString, buffer);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "sample_string");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(sizeof(sampleString), propertySize);
    ASSERT_STREQ(sampleString, buffer);

    // 異常系

    result = nn::dt::GetPropertyString(&propertySize, buffer, 0, &node, "empty_string");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, 1, &node, "sample_string");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "property_4byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetPropertyString(&propertySize, nullptr, sizeof(buffer), &node, "sample_string");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), nullptr, "sample_string");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

TEST(PropertyApi, GetPropertyStringList)
{
    nn::Result result;
    nn::dt::Node node;
    size_t propertySize;

    char buffer[64];
    const char emptyString[] = "";
    const char sampleStringList[] = "string0\0string1\0string2";

    // 準備
    result = nn::dt::FindNodeByPath(&node, "/sample_property");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "empty_string");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(sizeof(emptyString), propertySize);
    ASSERT_EQ(0, std::memcmp(emptyString, buffer, sizeof(emptyString)));

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "sample_string_list");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(sizeof(sampleStringList), propertySize);
    ASSERT_EQ(0, std::memcmp(sampleStringList, buffer, sizeof(sampleStringList)));

    // 異常系
    result = nn::dt::GetPropertyString(&propertySize, buffer, 2, &node, "sample_string_list");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(sampleStringList) - 1, &node, "sample_string_list");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultBufferTooSmall(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "not_exist");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, "property_7byte");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyTypeError(), result);

    result = nn::dt::GetPropertyString(&propertySize, nullptr, sizeof(buffer), &node, "sample_string_list");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), nullptr, "sample_string_list");
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetPropertyString(&propertySize, buffer, sizeof(buffer), &node, nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

}}
