﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/dt.h>
#include <nn/dt/detail/dt_InternalTypes.h>

#include <nnt/nntest.h>

#include "testDt_Common.h"

namespace nnt { namespace dt {

TEST(InterruptApi, GetInterruptCount)
{
    int count;
    nn::Result result;
    nn::dt::Node cpu0, uart;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetInterruptCount(&count, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(3, count);

    // 異常系
    result = nn::dt::GetInterruptCount(&count, &cpu0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);
}

TEST(InterruptApi, GetInterrupt)
{
    nn::Result result;
    nn::dt::Node cpu0, uart, i2c;

    nn::dt::InterruptInfo info;

    // 準備
    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetInterrupt(&info, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    ASSERT_EQ(4, static_cast<int>(info.name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info.trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info.polarity);

    // 異常系
    result = nn::dt::GetInterrupt(&info, &cpu0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetInterrupt(&info, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);
}

TEST(InterruptApi, GetInterruptArray)
{
    nn::Result result;
    nn::dt::Node cpu0, uart, i2c;

    nn::dt::InterruptInfo info[3];

    // 準備
    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetInterruptArray(info, sizeof(info) / sizeof(info[0]), &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    ASSERT_EQ(1, static_cast<int>(info[0].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info[0].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[0].polarity);

    ASSERT_EQ(2, static_cast<int>(info[1].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Edge, info[1].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[1].polarity);

    ASSERT_EQ(3, static_cast<int>(info[2].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Level, info[2].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_ActiveHigh, info[2].polarity);

    result = nn::dt::GetInterruptArray(info, 1, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    ASSERT_EQ(4, static_cast<int>(info[0].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info[0].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[0].polarity);

    // 異常系
    result = nn::dt::GetInterruptArray(info, sizeof(info) / sizeof(info[0]), &cpu0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetInterruptArray(nullptr, sizeof(info) / sizeof(info[0]), &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetInterruptArray(info, sizeof(info) / sizeof(info[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    // 異常系 (API に与える配列長 < 実際の割り込みの数)
    result = nn::dt::GetInterruptArray(info, 2, &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);

    // 異常系 (API に与える配列長 > 実際の割り込みの数)
    result = nn::dt::GetInterruptArray(info, 2, &i2c);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultSizeIncorrect(), result);
}

TEST(InterruptApi, GetInterruptList)
{
    int count;
    nn::Result result;
    nn::dt::Node cpu0, uart, i2c;

    nn::dt::InterruptInfo info[3];

    // 準備
    result = nn::dt::FindNodeByPath(&cpu0, "/cpus/cpu@0");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&uart, "/bus/uart");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    result = nn::dt::FindNodeByPath(&i2c, "/bus/i2c");
    NNT_DT_ASSERT_RESULT_SUCCESS(result);

    // 正常系
    result = nn::dt::GetInterruptList(info, &count, sizeof(info) / sizeof(info[0]), &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(3, count);

    ASSERT_EQ(1, static_cast<int>(info[0].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info[0].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[0].polarity);

    ASSERT_EQ(2, static_cast<int>(info[1].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Edge, info[1].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[1].polarity);

    ASSERT_EQ(3, static_cast<int>(info[2].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Level, info[2].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_ActiveHigh, info[2].polarity);

    // 正常系 (API に与える配列長 < 実際の割り込みの数)
    result = nn::dt::GetInterruptList(info, &count, 2, &uart);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(2, count);

    ASSERT_EQ(1, static_cast<int>(info[0].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info[0].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[0].polarity);

    ASSERT_EQ(2, static_cast<int>(info[1].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Edge, info[1].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[1].polarity);

    // 正常系 (API に与える配列長 > 実際の割り込みの数)
    result = nn::dt::GetInterruptList(info, &count, 2, &i2c);
    NNT_DT_ASSERT_RESULT_SUCCESS(result);
    ASSERT_EQ(1, count);

    ASSERT_EQ(4, static_cast<int>(info[0].name));
    ASSERT_EQ(nn::dt::InterruptTriggerType_Unspecified, info[0].trigger);
    ASSERT_EQ(nn::dt::InterruptPolarityType_Unspecified, info[0].polarity);

    // 異常系
    result = nn::dt::GetInterruptList(info, &count, sizeof(info) / sizeof(info[0]), &cpu0);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultPropertyNotExist(), result);

    result = nn::dt::GetInterruptList(nullptr, &count, sizeof(info) / sizeof(info[0]), &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetInterruptList(info, nullptr, sizeof(info) / sizeof(info[0]), &uart);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);

    result = nn::dt::GetInterruptList(info, &count, sizeof(info) / sizeof(info[0]), nullptr);
    NNT_DT_ASSERT_RESULT_INCLUDED(nn::dt::ResultInvalidArgument(), result);
}

}}
