﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/ddsf/ddsf_Result.h>
#include <nn/ddsf/detail/ddsf_ForEach.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

TEST(ForEach, Result)
{
    std::array<int, 5> list{ 1, 2, 3, 4, 5 };
    nn::os::SdkMutex mutex;

    int invokedCount;
    auto funcThatSucceeds = [&](int* pVal) -> nn::Result
    {
        ++invokedCount;
        NN_ASSERT_EQUAL(*pVal, invokedCount);
        NN_RESULT_SUCCESS;
    };
    auto funcThatFails = [&](const int* pVal) -> nn::Result
    {
        ++invokedCount;
        if ( *pVal == 2 )
        {
            return nn::ddsf::ResultAccessModeDenied();
        }
        if ( *pVal == 3 )
        {
            return nn::ddsf::ResultDeviceCodeNotFound();
        }
        if ( *pVal == 4 )
        {
            return nn::ddsf::ResultInternalError();
        }
        NN_RESULT_SUCCESS;
    };

    invokedCount = 0;
    NNT_EXPECT_RESULT_SUCCESS(nn::ddsf::detail::ForEach(&list, funcThatSucceeds, &mutex, true));
    EXPECT_EQ(5, invokedCount);

    invokedCount = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultAccessModeDenied, nn::ddsf::detail::ForEach(&list, funcThatFails, &mutex, true));
    EXPECT_EQ(2, invokedCount);

    invokedCount = 0;
    NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultAccessModeDenied, nn::ddsf::detail::ForEach(&list, funcThatFails, &mutex, false));
    EXPECT_EQ(5, invokedCount);
}

TEST(ForEach, Bool)
{
    std::array<int, 5> list{ 1, 2, 3, 4, 5 };
    nn::os::SdkMutex mutex;

    int invokedCount;
    auto func = [&](const int* pVal) -> bool
    {
        ++invokedCount;
        if ( *pVal == 3 )
        {
            return false;
        }
        return true;
    };

    invokedCount = 0;
    EXPECT_EQ(2, nn::ddsf::detail::ForEach(&list, func, &mutex));
    EXPECT_EQ(3, invokedCount);
}
