﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <array>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Abort.h>

#include <nn/ddsf/ddsf_DeviceCodeEntry.h>
#include <nn/ddsf/ddsf_DeviceCodeEntryManager.h>
#include <nn/ddsf/ddsf_IDevice.h>
#include <nn/ddsf/ddsf_IDriver.h>
#include <nn/ddsf/ddsf_Result.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

namespace {

    class TestDriver : public nn::ddsf::IDriver
    {
    };
    TestDriver g_DummyDriver;

    class TestDevice : public nn::ddsf::IDevice
    {
    };

}

TEST(DeviceCodeEntry, Build)
{
    nn::ddsf::DeviceCodeEntryHolder entryHolder;
    TestDevice device;
    nn::DeviceCode deviceCode = 0xffff;

    EXPECT_FALSE(entryHolder.IsBuilt());
    EXPECT_DEATH_IF_SUPPORTED(entryHolder.Get(), "");
    EXPECT_DEATH_IF_SUPPORTED(entryHolder.Destroy(), "");

    auto& entry = entryHolder.Build(deviceCode, &device);
    EXPECT_EQ(deviceCode, entry.GetDeviceCode());
    EXPECT_EQ(&device, static_cast<TestDevice*>(&entry.GetDevice()));
    EXPECT_EQ(&entry, &entryHolder.Get());

    EXPECT_TRUE(entryHolder.IsBuilt());
    EXPECT_DEATH_IF_SUPPORTED(entryHolder.Build(deviceCode, &device), "");

    entryHolder.Destroy();
    EXPECT_FALSE(entryHolder.IsBuilt());
    EXPECT_DEATH_IF_SUPPORTED(entryHolder.Get(), "");
    EXPECT_DEATH_IF_SUPPORTED(entryHolder.Destroy(), "");
}

TEST(DeviceCodeEntryManager, ContainerType)
{
    // http://en.cppreference.com/w/cpp/language/range-for
    // 1. If range_expression is an expression of array type, then begin_expr is __range and end_expr is (__range + __bound),
    //    where __bound is the number of elements in the array (if the array has unknown size or is of an incomplete type, the program is ill-formed)
    // 2. If range_expression is an expression of a class type C that has a member named begin and/or a member named end
    //    (regardless of the type or accessibility of such member), then begin_expr is __range.begin() and end_expr is __range.end();
    // 3. Otherwise, begin_expr is begin(__range) and end_expr is end(__range), which are found via argument - dependent lookup (non - ADL lookup is not performed).
    {
        nn::ddsf::DeviceCodeEntryHolder entries[128];
        nn::ddsf::DeviceCodeEntryManager manager(&entries);
        EXPECT_EQ(NN_ARRAY_SIZE(entries), manager.GetCapacity());
    }
    {
        std::array<nn::ddsf::DeviceCodeEntryHolder, 64> entries;
        nn::ddsf::DeviceCodeEntryManager manager(&entries);
        EXPECT_EQ(entries.max_size(), manager.GetCapacity());
    }
}

TEST(DeviceCodeEntryManager, AddRemove)
{
    const int EntryCount = 128;
    nn::ddsf::DeviceCodeEntryHolder entries[EntryCount];
    nn::ddsf::DeviceCodeEntryManager manager(&entries);

    TestDevice device;
    EXPECT_DEATH_IF_SUPPORTED(manager.Add(nn::DeviceCode(0x1000 + 0), &device), ""); // device が何らかのドライバに RegisterDevice されている事前条件の確認

    g_DummyDriver.RegisterDevice(&device);
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);

        NNT_EXPECT_RESULT_SUCCESS(manager.Add(deviceCode, &device));
    }
    NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultOutOfResource, manager.Add(nn::DeviceCode(0x1000 + EntryCount), &device));

    nn::ddsf::DeviceCodeEntry* pEntry = nullptr;
    NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultDeviceCodeNotFound, manager.FindDeviceCodeEntry(&pEntry, nn::DeviceCode(0x1000 + EntryCount)));
    EXPECT_EQ(nullptr, pEntry);

    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);

        NNT_EXPECT_RESULT_SUCCESS(manager.FindDeviceCodeEntry(&pEntry, deviceCode));
        EXPECT_NE(nullptr, pEntry);
        EXPECT_EQ(&device, static_cast<TestDevice*>(&pEntry->GetDevice()));

        const nn::ddsf::DeviceCodeEntry* pEntryConst = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(manager.FindDeviceCodeEntry(&pEntryConst, deviceCode));
        EXPECT_NE(nullptr, pEntryConst);
        EXPECT_EQ(&device, static_cast<const TestDevice*>(&pEntryConst->GetDevice()));

        nn::ddsf::IDevice* pDevice = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(manager.FindDevice(&pDevice, deviceCode));
        EXPECT_EQ(&device, static_cast<TestDevice*>(pDevice));

        const nn::ddsf::IDevice* pDeviceConst = nullptr;
        NNT_EXPECT_RESULT_SUCCESS(manager.FindDevice(&pDeviceConst, deviceCode));
        EXPECT_EQ(&device, static_cast<const TestDevice*>(pDeviceConst));

        EXPECT_TRUE(manager.Remove(deviceCode));
        EXPECT_FALSE(manager.Remove(deviceCode));
        NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultDeviceCodeNotFound, manager.FindDeviceCodeEntry(&pEntry, deviceCode));
    }
}

TEST(DeviceCodeEntryManager, Reset)
{
    const int EntryCount = 128;
    nn::ddsf::DeviceCodeEntryHolder entries[EntryCount];
    nn::ddsf::DeviceCodeEntryManager manager(&entries);

    TestDevice device;
    g_DummyDriver.RegisterDevice(&device);
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);
        NNT_EXPECT_RESULT_SUCCESS(manager.Add(deviceCode, &device));
    }

    manager.Reset();

    // すべてのエントリが登録済リストから削除されたことの確認
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);
        nn::ddsf::DeviceCodeEntry* pEntry = nullptr;
        NNT_EXPECT_RESULT_FAILURE(nn::ddsf::ResultDeviceCodeNotFound, manager.FindDeviceCodeEntry(&pEntry, deviceCode));
    }

    // リソースがすべて解放されており、再登録可能なことの確認
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);
        NNT_EXPECT_RESULT_SUCCESS(manager.Add(deviceCode, &device));
    }
}

TEST(DeviceCodeEntryManager, Duplicate)
{
    const int EntryCount = 128;
    nn::ddsf::DeviceCodeEntryHolder entries[EntryCount];
    nn::ddsf::DeviceCodeEntryManager manager(&entries);

    TestDevice device;
    g_DummyDriver.RegisterDevice(&device);
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);

        NNT_EXPECT_RESULT_SUCCESS(manager.Add(deviceCode, &device));
    }

    EXPECT_DEATH_IF_SUPPORTED(manager.Add(nn::DeviceCode(0x1000 + 0), &device), "");
    EXPECT_DEATH_IF_SUPPORTED(manager.Add(nn::DeviceCode(0x1000 + EntryCount - 1), &device), "");
}

TEST(DeviceCodeEntryManager, ForEach)
{
    const int EntryCount = 128;
    nn::ddsf::DeviceCodeEntryHolder entries[EntryCount];
    nn::ddsf::DeviceCodeEntryManager manager(&entries);

    TestDevice device;
    g_DummyDriver.RegisterDevice(&device);
    for ( int i = 0; i < EntryCount; ++i )
    {
        nn::DeviceCode deviceCode = (0x1000 + i);
        NNT_ASSERT_RESULT_SUCCESS(manager.Add(deviceCode, &device));
    }

    EXPECT_EQ(EntryCount,
        manager.ForEachEntry(
            [](nn::ddsf::DeviceCodeEntry*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
    // const DeviceCodeEntry* を取るラムダならば const オブジェクトで呼べることをテスト
    EXPECT_EQ(EntryCount,
        static_cast<const nn::ddsf::DeviceCodeEntryManager&>(manager).ForEachEntry(
            [](const nn::ddsf::DeviceCodeEntry*) NN_NOEXCEPT -> bool
            {
                return true;
            }
        )
    );
}
