﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../Common/test_Pragma.h"
#include <nn/TargetConfigs/build_Base.h>

#include <stdint.h>
#include <stdbool.h>

#include <nnc/nn_Log.h>
#include <nnc/dd.h>

#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include "test_LinkageC-os.horizon.h"
#endif

//-----------------------------------------------------------------------------

#define NNT_OS_RETURN_FALSE_UNLESS(condition)                           \
                                do                                      \
                                {                                       \
                                    if (!(condition)) { return false; } \
                                } while (false)

//----------------------------------------------------------------------------
// 非 I/O な物理アドレスを返すテスト補助関数。
// Win32 環境では 0 を返し、Horizon 環境では明らかに DDR メモリな
// 物理アドレスを返す。
//
nnddPhysicalAddress GetNonIoPhysAddress()
{
#if defined(NN_BUILD_CONFIG_OS_WIN32)
    return 0;
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    nnddPhysicalAddress padrs;
    nnResult result = nnddQuerySinglePhysicalAddress(&padrs, &padrs, 1);
    if (!nnResultIsSuccess(result))
    {
        return 0;
    }
    return padrs;
#endif
}

//-----------------------------------------------------------------------------

void nntosTestTypeInfoDumpForC(void)
{
    NNC_LOG("\n");
    NNC_LOG("C  : PhysicalMemoryInfo: size=%d align=%d\n", sizeof(nnddPhysicalMemoryInfo), NNC_ALIGNOF(nnddPhysicalMemoryInfo));
    NNC_LOG("\n");
}

//-----------------------------------------------------------------------------

char buffer[4 * 1024 * 1024];


bool nntddCacheTestForC(void)
{
    NNC_LOG("キャッシュ操作関連のテスト開始\n");

    nnddInvalidateDataCache(buffer, sizeof(buffer));
    nnddStoreDataCache(buffer, sizeof(buffer));
    nnddFlushDataCache(buffer, sizeof(buffer));

    NNC_LOG("キャッシュ操作関連のテスト終了\n");
    return true;
}

//----------------------------------------------------------------------------
//
bool nntddPhysicalAddressTestForC(void)
{
    NNC_LOG("物理アドレス変換関連のテスト開始\n");

    nnResult result;
    nnddPhysicalMemoryInfo pmi;
    nnddInitializePhysicalMemoryInfo(&pmi, buffer, sizeof(buffer));
    while (nnResultIsSuccess(result = nnddQueryNextPhysicalMemoryInfo(&pmi)))
    {
        nnddGetPhysicalAddress(&pmi);
        nnddGetVirtualAddress(&pmi);
        nnddGetSize(&pmi);
    }

    nnddPhysicalAddress address;
    nnddQuerySinglePhysicalAddress(&address, buffer, 1);

    NNC_LOG("物理アドレス変換関連のテスト終了\n");
    return true;
}

//----------------------------------------------------------------------------
//
bool PrintJudgement( bool result )
{
    if (result)
    {
        NNC_LOG(" ... OK\n");
        return true;
    }
    else
    {
        NNC_LOG(" ... NG\n");
        return false;
    }
}


bool nntddIoMappingAddressTestForC(void)
{
    NNC_LOG("I/O アドレス変換関連のテスト開始\n");

    NNC_LOG("I/O ではない物理アドレスを渡すと 0 を返すか\n");
    {
        nnddPhysicalAddress nonIoPhysAddress = GetNonIoPhysAddress();
        if (nonIoPhysAddress == 0)
        {
            return false;
        }
        uintptr_t vadrs = nnddQueryIoMappingAddress(nonIoPhysAddress, 1);
        NNC_LOG("PhysicalAddress = 0x%016llx\n", nonIoPhysAddress);
        NNC_LOG("VirtualAddress  = 0x%p", vadrs);
        if (!PrintJudgement( vadrs == 0 ))
        {
            return false;
        }
        NNC_LOG("\n");
    }

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
    NNC_LOG("I/O 物理アドレスに対する返値の確認\n");
#if 0   // ここの #if 0 はカーネルでの固定マップが削除されたら
        // テストとしてパスするようになるはず。
    {
        NNC_LOG("I/O 物理アドレス -1 への Query に失敗するか\n");
        const nnddPhysicalAddress   padrs = TargetIoPhysicalAddress - 1;
        const size_t                size  = TargetIoSize;
        uintptr_t vadrs = nnddQueryIoMappingAddress(padrs, size);
        ("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NNC_LOG("VirtualAddress  = 0x%p", vadrs);
        if (!PrintJudgement( vadrs == 0 ))
        {
            return false;
        }
        NNC_LOG("\n");
    }
#endif
    {
        NNC_LOG("I/O 物理アドレスへの Query に成功するか\n");
        const nnddPhysicalAddress   padrs = TargetIoPhysicalAddress;
        const size_t                size  = TargetIoSize;
        uintptr_t vadrs = nnddQueryIoMappingAddress(padrs, size);
        NNC_LOG("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NNC_LOG("VirtualAddress  = 0x%p", vadrs);
        if (!PrintJudgement( vadrs != 0 ))
        {
            return false;
        }
        NNC_LOG("\n");
    }
#if 0   // ここの #if 0 はカーネルでの固定マップが削除されたら
        // テストとしてパスするようになるはず。
    {
        NNC_LOG("I/O 物理アドレス +1 への Query に失敗するか\n");
        const nnddPhysicalAddress   padrs = TargetIoPhysicalAddress + 1;
        const size_t                size  = TargetIoSize;
        uintptr_t vadrs = nnddQueryIoMappingAddress(padrs, size);
        NNC_LOG("PhysicalAddress = 0x%016llx  size=0x%08x\n", padrs, size);
        NNC_LOG("VirtualAddress  = 0x%p", vadrs);
        if (!PrintJudgement( vadrs == 0 ))
        {
            return false;
        }
        NNC_LOG("\n");
    }
#endif
#endif

    NNC_LOG("I/O アドレス変換関連のテスト終了\n");
    return true;
}

//----------------------------------------------------------------------------
//
bool nntddMemoryBarrierTestForC(void)
{
    NNC_LOG("メモリバリア関連のテスト開始\n");

    nnddEnsureMemoryAccess();

    NNC_LOG("メモリバリア関連のテスト終了\n");
    return true;
}

//-----------------------------------------------------------------------------
