﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>
#include <string>
#include <mutex>

#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringUtil.h>
#include <nn/util/util_TFormatString.h>

#include <nn/oe.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiClientManagement.h>

#include <nn/ssl.h>
#include <nn/socket.h>

#include <nn/dauth/dauth_Api.h>

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>

//!----------------------------------------------------------------------------
//! [CAUTION]
//!     - 本テストを実機で実行するにはサービスディスカバリを初期化しておく必要があります。
//!----------------------------------------------------------------------------

using namespace nn;

namespace {

//!----------------------------------------------------------------------------
class StatefulInitializer
{
public:
    StatefulInitializer() NN_NOEXCEPT : m_IsInitialized(false) {}

    template<typename Predicate>
    void Initialize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        NN_LOG("==== Initializing %s\n", pLogPrefix);
        executor();
        m_IsInitialized = true;
    }

    template<typename Predicate>
    void Finalize(const char* pLogPrefix, Predicate executor) NN_NOEXCEPT
    {
        NN_LOG("==== Cleaning up %s\n", pLogPrefix);
        if (m_IsInitialized)
        {
            m_IsInitialized = false;
            executor();
        }
    }

private:
    bool    m_IsInitialized;
};

//!----------------------------------------------------------------------------
class Environment
{
private:
    socket::ConfigDefaultWithMemory m_SocketConfig;

    StatefulInitializer m_InitialSocket;
    StatefulInitializer m_InitialNifm;
    StatefulInitializer m_InitialSsl;

public:
    void Initialize() NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
        oe::Initialize();
#endif

        m_InitialNifm.Initialize("NIFM", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(nifm::Initialize());
            nifm::SubmitNetworkRequestAndWait();
        });
        m_InitialSsl.Initialize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Initialize());
        });
        m_InitialSocket.Initialize("socket", [this]() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Initialize(m_SocketConfig));
        });
    }

    void Finalize() NN_NOEXCEPT
    {
        m_InitialSocket.Finalize("socket", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(socket::Finalize());
        });
        m_InitialSsl.Finalize("SSL", []() -> void
        {
            NNT_ASSERT_RESULT_SUCCESS(ssl::Finalize());
        });
        m_InitialNifm.Finalize("NIFM", []() -> void
        {
            nifm::CancelNetworkRequest();
        });
    }

};

//!----------------------------------------------------------------------------
class TestForAcquire : public ::testing::Test
{
private:
    static Environment& GetEnvironment() NN_NOEXCEPT
    {
        NN_FUNCTION_LOCAL_STATIC(Environment, s_Instance);
        return s_Instance;
    }

protected:

    virtual void SetUp()
    {
    }

    virtual void TearDown()
    {
    }

    static void SetUpTestCase()
    {
        GetEnvironment().Initialize();
    }

    static void TearDownTestCase()
    {
        GetEnvironment().Finalize();
    }
};

} // ::~unnamed

//!----------------------------------------------------------------------------
extern "C" void nninitStartup()
{
    NN_FUNCTION_LOCAL_STATIC(NN_ALIGNAS(os::MemoryPageSize) uint8_t, s_MallocBuffer, [1 * 1024 * 1024]);

    nn::init::InitializeAllocator(s_MallocBuffer, sizeof(s_MallocBuffer));
}

//!----------------------------------------------------------------------------
namespace {
void VerifySingleDeviceAuthenticationToken(uint64_t const clientId) NN_NOEXCEPT
{
    TimeSpan expiration;
    int availableTokenLength;
    NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) char pResponse[dauth::RequiredBufferSizeForDeviceAuthenticationToken] = {'\0'};

    NNT_ASSERT_RESULT_SUCCESS(dauth::AcquireDeviceAuthenticationToken(
        &expiration,
        &availableTokenLength,
        pResponse,
        sizeof(pResponse),
        clientId,
        true,
        nullptr
    ));

    const auto remainExpireSeconds = (expiration - os::GetSystemTick().ToTimeSpan()).GetSeconds();
    const auto expireSeconds = expiration.GetSeconds();
    ASSERT_TRUE(expireSeconds > 0);
    ASSERT_TRUE(availableTokenLength > 0);
    ASSERT_EQ(static_cast<size_t>(availableTokenLength), util::Strnlen(pResponse, sizeof(pResponse)));

    NN_LOG("==========\nCLIENT [%llx] Token information {\n    availableTokenLength : %d\n    expiration : %lld sec\n    remain : %lld sec\n}\n", clientId, availableTokenLength, expireSeconds, remainExpireSeconds);
    NN_LOG("==========\nDeviceAuthenticationToken:\n%s\n\n", pResponse);
}
void VerifySingleEdgeToken(uint64_t const clientId) NN_NOEXCEPT
{
    TimeSpan expiration;
    int availableTokenLength;
    NN_ALIGNAS(std::alignment_of<std::max_align_t>::value) char pResponse[dauth::RequiredBufferSizeForEdgeToken] = {'\0'};

    NNT_ASSERT_RESULT_SUCCESS(dauth::AcquireEdgeToken(
        &expiration,
        &availableTokenLength,
        pResponse,
        sizeof(pResponse),
        clientId,
        nullptr
    ));

    const auto remainExpireSeconds = (expiration - os::GetSystemTick().ToTimeSpan()).GetSeconds();
    const auto expireSeconds = expiration.GetSeconds();
    ASSERT_TRUE(expireSeconds > 0);
    ASSERT_TRUE(availableTokenLength > 0);
    ASSERT_EQ(static_cast<size_t>(availableTokenLength), util::Strnlen(pResponse, sizeof(pResponse)));

    NN_LOG("==========\nCLIENT [%llx] Token information {\n    availableTokenLength : %d\n    expiration : %lld sec\n    remain : %lld sec\n}\n", clientId, availableTokenLength, expireSeconds, remainExpireSeconds);
    NN_LOG("==========\nEdgeToken:\n%s\n\n", pResponse);
}
} // ~namespace <anonymous>

//!----------------------------------------------------------------------------
TEST_F(TestForAcquire, Basic)
{
    VerifySingleDeviceAuthenticationToken(0x8f849b5d34778d8eull);
    VerifySingleDeviceAuthenticationToken(0x16e96f76850156d1ull);
    VerifySingleEdgeToken(0x146c8ac7b8a0db52);
}

