﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include <nnt/nntest.h>

#include "testCrypto_AesCipherTypes.h"

extern int  g_RepeatCount;
extern bool g_IsPerfCheckEnabled;
extern int  g_PerformanceMargin;

/*
  計測ケースの追加手順は testCrypto_AesCipherTypes.h で定義されている
  AesCipherTypes で定義される型に実装を加えてください。
 */

/*
  @brief テストフィクスチャクラスです。
 */
template <typename T>
class AesPerformanceTest : public ::testing::Test {
public:
    // テストで使うバッファサイズ
    static const int64_t TestBufferSize = 192 * 1024 * 1024; // 192MB

protected:
    // テストケースごとの初期化としてバッファを確保します
    virtual void SetUp()
    {
        m_InBuffer = static_cast<uint8_t*>(std::malloc(TestBufferSize));
        m_OutBuffer = static_cast<uint8_t*>(std::malloc(TestBufferSize));
        ASSERT_TRUE(m_InBuffer != nullptr);
        ASSERT_TRUE(m_OutBuffer != nullptr);

        m_pAesCipher = new T;
        ASSERT_TRUE(m_pAesCipher != nullptr);
    }

    // テストケースごとの終了処理としてバッファを開放します
    virtual void TearDown()
    {
        delete m_pAesCipher;

        std::free(m_InBuffer);
        std::free(m_OutBuffer);
    }

protected:
    uint8_t* m_InBuffer;
    uint8_t* m_OutBuffer;
    T*       m_pAesCipher;
};

/**
  @brief  AesPerformance で計測するテストケースを登録します。

  @detail
  テストケースを追加する際は AesCipherTypes で定義される型に実装を加えてください。

 */
TYPED_TEST_CASE(AesPerformanceTest, AesCipherTypes);

/**
  @brief  入出力バッファが同一の場合の計測プログラムです。

  @details
  Initialize() は単純に処理時間を計測して表示します。
  Update() は TestBufferSize に対して行なった処理結果を元にスループットを計算して表示します。

 */
TYPED_TEST(AesPerformanceTest, InPlace)
{
    NN_LOG("Mode: %s (In-Place)\n", this->m_pAesCipher->GetName());

    uint64_t performanceInTotal = 0;

    for (auto i = 0;  i < g_RepeatCount + 1; ++i)
    {
        nn::os::Tick startTick;
        nn::os::Tick duration;

        // 初期化を実行して時間を計測する
        startTick = nn::os::GetSystemTick();
        this->m_pAesCipher->Initialize();
        duration = nn::os::GetSystemTick() - startTick;
        //NN_LOG("Initialize took %lld usec\n", nn::os::ConvertToTimeSpan(duration).GetMicroSeconds());

        // メインの処理の実行時間を計測する
        startTick = nn::os::GetSystemTick();
        this->m_pAesCipher->Update(this->m_OutBuffer, this->TestBufferSize, this->m_OutBuffer, this->TestBufferSize);
        duration = nn::os::GetSystemTick() - startTick;

        // スループットを計算して表示する（B/us = MB/s で計算できる）。
        int64_t performance = (this->TestBufferSize * 100) / nn::os::ConvertToTimeSpan(duration).GetMicroSeconds();
        NN_LOG("Operation took %lld usec for %lld bytes (%lld.%lld MB/s)\n",
               nn::os::ConvertToTimeSpan(duration).GetMicroSeconds(),
               this->TestBufferSize,
               performance / 100, performance % 100);

        // 初回分だけ計測誤差が多い環境があるので初回分だけ切り捨てる
        if (i > 0)
        {
            performanceInTotal += performance;
        }
    }

    int64_t       performanceAverage = performanceInTotal / g_RepeatCount;
    const int64_t TargetPerformanceLowerBoudary = this->m_pAesCipher->TargetPerformanceInPlace * (100 - g_PerformanceMargin) / 100;
    const int64_t TargetPerformanceHigherBoudary = this->m_pAesCipher->TargetPerformanceInPlace * (100 + g_PerformanceMargin) / 100;

    // TeamCity 用の表示(In-Place 分は省略)
    //NN_LOG("##teamcity[buildStatisticValue key='AES Performance (%s:In-Place)' value='%lld.%lld']\n", this->m_pAesCipher->GetName(), performanceAverage / 100, performanceAverage % 100);

    NN_LOG("\n-------------------------------------------------------------------------------\n");
    NN_LOG("INFO: Performance change (%s:In-Place) %lld -> %lld (%.2f%%)\n",
           this->m_pAesCipher->GetName(), this->m_pAesCipher->TargetPerformanceInPlace,
           performanceAverage, performanceAverage / static_cast<double>(this->m_pAesCipher->TargetPerformanceInPlace) * 100);
    NN_LOG("-------------------------------------------------------------------------------\n\n");

    // オプションが指定されたらパフォーマンスがマージン込みで所定の値と同等かチェックする
    if (g_IsPerfCheckEnabled)
    {
        EXPECT_GE(performanceAverage, TargetPerformanceLowerBoudary);
        EXPECT_LE(performanceAverage, TargetPerformanceHigherBoudary);
    }
}

/**
  @brief  入出力バッファが異なる場合の計測プログラムです。

  @details
  Initialize() は単純に処理時間を計測して表示します。
  Update() は TestBufferSize に対して行なった処理結果を元にスループットを計算して表示します。

 */
TYPED_TEST(AesPerformanceTest, OutPlace)
{
    NN_LOG("Mode: %s (Out-Place)\n", this->m_pAesCipher->GetName());

    uint64_t performanceInTotal = 0;

    for (auto i = 0;  i < g_RepeatCount + 1; ++i)
    {
        nn::os::Tick startTick;
        nn::os::Tick duration;

        // 初期化を実行して時間を計測する
        startTick = nn::os::GetSystemTick();
        this->m_pAesCipher->Initialize();
        duration = nn::os::GetSystemTick() - startTick;
        //NN_LOG("Initialize took %lld usec\n", nn::os::ConvertToTimeSpan(duration).GetMicroSeconds());

        // メインの処理の実行時間を計測する
        startTick = nn::os::GetSystemTick();
        this->m_pAesCipher->Update(this->m_OutBuffer, this->TestBufferSize, this->m_InBuffer, this->TestBufferSize);
        duration = nn::os::GetSystemTick() - startTick;

        // スループットを計算して表示する（B/us = MB/s で計算できる）。
        int64_t performance = (this->TestBufferSize * 100) / nn::os::ConvertToTimeSpan(duration).GetMicroSeconds();
        NN_LOG("Operation took %lld usec for %lld bytes (%lld.%lld MB/s)\n",
               nn::os::ConvertToTimeSpan(duration).GetMicroSeconds(),
               this->TestBufferSize,
               performance / 100, performance % 100);

        // 初回分だけ計測誤差が多い環境があるので初回分だけ切り捨てる
        if (i > 0)
        {
            performanceInTotal += performance;
        }
    }

    int64_t       performanceAverage = performanceInTotal / g_RepeatCount;
    const int64_t TargetPerformanceLowerBoudary = this->m_pAesCipher->TargetPerformance * (100 - g_PerformanceMargin) / 100;
    const int64_t TargetPerformanceHigherBoudary = this->m_pAesCipher->TargetPerformance * (100 + g_PerformanceMargin) / 100;

    // TeamCity 用の表示
    NN_LOG("##teamcity[buildStatisticValue key='Throughput (%s)' value='%lld.%lld']\n", this->m_pAesCipher->GetName(), performanceAverage / 100, performanceAverage % 100);

    NN_LOG("\n-------------------------------------------------------------------------------\n");
    NN_LOG("INFO: Performance change (%s) %lld -> %lld (%.2f%%)\n",
           this->m_pAesCipher->GetName(), this->m_pAesCipher->TargetPerformance,
           performanceAverage, performanceAverage / static_cast<double>(this->m_pAesCipher->TargetPerformance) * 100);
    NN_LOG("-------------------------------------------------------------------------------\n\n");

    // オプションが指定されたらパフォーマンスがマージン込みで所定の値と同等かチェックする
    if (g_IsPerfCheckEnabled)
    {
        EXPECT_GE(performanceAverage, TargetPerformanceLowerBoudary);
        EXPECT_LE(performanceAverage, TargetPerformanceHigherBoudary);
    }
}
