﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_RsaCalculator.h>
#include <cstring>
#include <cstdlib>

#include <nnt/nntest.h>
#include "CavsParser.h"
#include "ProfileCheck.h"
#include "OctetString.h"

class RsaDpParser : public CavsParser
{
public:
    enum
    {
        Count,
        N,
        E,
        D,
        C,
        K,
        Mod,
        Result
    };

    RsaDpParser()
    {
        m_AllowedTokens[Count]  = AllowedToken("COUNT = ", false);
        m_AllowedTokens[N]      = AllowedToken("n = ", true);
        m_AllowedTokens[E]      = AllowedToken("e = ", true);
        m_AllowedTokens[D]      = AllowedToken("d = ", true);
        m_AllowedTokens[C]      = AllowedToken("c =  ", true);
        m_AllowedTokens[K]      = AllowedToken("k = ", true);
        m_AllowedTokens[Mod]    = AllowedToken("[mod = ", false);
        m_AllowedTokens[Result] = AllowedToken("Result = ", false);
        m_TokenForUnitComplete  = Result;
    }

    virtual void TestUnit() override
    {
        ASSERT_EQ(m_UnitTokens.count(N), 1);
        ASSERT_EQ(m_UnitTokens.count(E), 1);
        ASSERT_EQ(m_UnitTokens.count(C), 1);
        ASSERT_EQ(m_UnitTokens.count(Result), 1);
        ASSERT_EQ(m_UnitTokens.count(Mod), 1);

        const std::string& nStr   = m_UnitTokens[N];
        const std::string& eStr   = m_UnitTokens[E];
        const std::string& dStr   = m_UnitTokens[D];
        const std::string& cStr   = m_UnitTokens[C];
        const std::string& kStr   = m_UnitTokens[K];

        if (m_UnitTokens[Mod] == "1024]")
        {
            if (m_UnitTokens[Result] == "Pass")
            {
                nn::crypto::RsaCalculator<128, 128> rsa;
                std::string outStr(128, char(0));

                // 1. Encryption: k^e == c ?
                EXPECT_TRUE( rsa.Initialize(nStr.c_str(), nStr.size(), eStr.c_str(), eStr.size()) );
                EXPECT_TRUE( rsa.ModExp(const_cast<char*>(outStr.c_str()), outStr.size(), kStr.c_str(), kStr.size()) );
                EXPECT_TRUE( outStr == cStr );

                // 2. Decryption: c^d == k ?
                EXPECT_TRUE( rsa.Initialize(nStr.c_str(), nStr.size(), dStr.c_str(), dStr.size()) );
                EXPECT_TRUE( rsa.ModExp(const_cast<char*>(outStr.c_str()), outStr.size(), cStr.c_str(), cStr.size()) );
                EXPECT_TRUE( outStr == kStr );
            }
        }
        else if (m_UnitTokens[Mod] == "2048]")
        {
            if (m_UnitTokens[Result] == "Pass")
            {
                nn::crypto::RsaCalculator<256, 256> rsa;
                std::string outStr(256, char(0));

                // 1. Encryption: k^e == c ?
                EXPECT_TRUE( rsa.Initialize(nStr.c_str(), nStr.size(), eStr.c_str(), eStr.size()) );
                EXPECT_TRUE( rsa.ModExp(const_cast<char*>(outStr.c_str()), outStr.size(), kStr.c_str(), kStr.size()) );
                EXPECT_TRUE( outStr == cStr );

                // 2. Decryption: c^d == k ?
                EXPECT_TRUE( rsa.Initialize(nStr.c_str(), nStr.size(), dStr.c_str(), dStr.size()) );
                EXPECT_TRUE( rsa.ModExp(const_cast<char*>(outStr.c_str()), outStr.size(), cStr.c_str(), cStr.size()) );
                EXPECT_TRUE( outStr == kStr );
            }
        }
        else
        {
            // SHOULD NOT COME HERE
            FAIL() << "unknown modulus size: " << m_UnitTokens[Mod].c_str();
        }

        m_UnitTokens.erase( Result );
    }
};

TEST(RsaTest, DpFile)
{
    RsaDpParser tester;
    tester.TestFile("RSADPtestvectors/RSADPComponent800_56B.txt", 60);
}

TEST(RsaTest, DpFile2)
{
    RsaDpParser tester;
    tester.TestFile("GeneratedRsaDpTestVectors.txt", 256);
}
