﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nnt/nntest.h>
#include <cstdlib>
#include "CavsParser.h"

class RsaParser : public CavsParser
{
public:
    RsaParser()
    {
        m_AllowedTokens[N] = AllowedToken("n = ", true);
        m_AllowedTokens[E] = AllowedToken("e = ", true);
        m_AllowedTokens[D] = AllowedToken("d = ", true);
        m_AllowedTokens[SHAAlg] = AllowedToken("SHAAlg = ", false);
        m_AllowedTokens[Mod] = AllowedToken("[mod = ", false);
        m_AllowedTokens[S] = AllowedToken("S = ", true);
        m_AllowedTokens[Msg] = AllowedToken("Msg = ", true);
        m_AllowedTokens[Result] = AllowedToken("Result = ", false);
        m_AllowedTokens[SaltVal] = AllowedToken("SaltVal = ", true);
        m_TokenForUnitComplete = Msg;
    }

protected:
    enum
    {
        Mod,
        N,
        E,
        D,
        K,
        SHAAlg,
        Msg,
        SaltVal,
        S,
        Result
    };

    inline void TrimLeadingZeros(std::string& str)
    {
        size_t i = 0;
        for (; i < str.size(); ++i)
        {
            if (str[i] != 0)
            {
                break;
            }
        }
        str = str.substr(i);
    }

    virtual void TestUnit() override
    {
        ASSERT_EQ(1u,m_UnitTokens.count(Msg));
        ASSERT_EQ(1u,m_UnitTokens.count(S));
        ASSERT_EQ(1u,m_UnitTokens.count(SHAAlg));
        ASSERT_EQ(1u,m_UnitTokens.count(N));
        ASSERT_EQ(1u,m_UnitTokens.count(E));
        ASSERT_EQ(1u,m_UnitTokens.count(D));

        // Trim leading 0s from e
        TrimLeadingZeros(m_UnitTokens[E]);

        const std::string& nStr = m_UnitTokens[N];
        const std::string& eStr = m_UnitTokens[E];
        const std::string& dStr = m_UnitTokens[D];

        TestWithKey( nStr, eStr, dStr);

        m_UnitTokens.erase(Msg);
        m_UnitTokens.erase(S);
        m_UnitTokens.erase(SHAAlg);
        m_UnitTokens.erase(SaltVal);
    }

    virtual void TestWithKey( const std::string& nStr, const std::string& eStr,
                              const std::string& dStr ) = 0;
};

class RsaSigVerParser : public RsaParser
{
public:
    virtual void TestWithKey( const std::string& nStr, const std::string& eStr,
                              const std::string& ) override
    {
        ASSERT_GE((int)m_UnitTokens[Result].size(), 0);
        bool expected = m_UnitTokens[Result][0] == 'P';

        const std::string& msgStr = m_UnitTokens[Msg];
        const std::string& correctSign = m_UnitTokens[S];
        const std::string& shaAlgStr = m_UnitTokens[SHAAlg];

        TestVerify( shaAlgStr, nStr.c_str(), nStr.size(), eStr.c_str(), eStr.size(),
                    msgStr.c_str(), msgStr.size(), correctSign.c_str(),
                    correctSign.size(), expected);
    }

    virtual void TestVerify( const std::string& sha,
                             const void* n, size_t nSize,
                             const void* e, size_t eSize,
                             const void* pMessage, size_t messageSize,
                             const void* pSignature, size_t signatureSize, bool expected) = 0;
};

class RsaSigGenParser : public RsaParser
{
public:
    virtual void TestWithKey( const std::string& nStr, const std::string&,
                              const std::string& dStr ) override
    {
        const std::string& msgStr = m_UnitTokens[Msg];
        const std::string& modStr = m_UnitTokens[Mod];
        const std::string& correctSign = m_UnitTokens[S];
        const std::string& shaAlgStr = m_UnitTokens[SHAAlg];
        int keySize = std::atoi(modStr.c_str());
        std::string signBuf(keySize / 8, '\x00');

        if ( nStr.size() && dStr.size() )
        {
            size_t signSize = signBuf.size();
            TestSign( shaAlgStr, nStr.c_str(), nStr.size(), dStr.c_str(), dStr.size(),
                      msgStr.c_str(), msgStr.size(),
                      const_cast<char*>(signBuf.c_str()), signSize, correctSign );
        }
    }

    virtual void TestSign(const std::string& sha,
                          const void* n, size_t nSize,
                          const void* d, size_t dSize,
                          const void* pMessage, size_t messageSize,
                          void* pSignature, size_t signatureSize, const std::string& expected) = 0;
};
