﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_EcdsaP256Sha256Signer.h>
#include <nn/crypto/crypto_EcdsaP256Sha256Verifier.h>
#include <nnt/nntest.h>

/*
Generate and dump keys with :

openssl ecparam -name prime256v1 -param_enc explicit -genkey -out key256.pem
openssl ec -text -in key256.pem | sed 1,/priv/d | sed '/^[^ ]/,$d' | tr -d -c [:xdigit:]             | xxd -p -r | xxd -i -c 16
openssl ec -text -in key256.pem | sed 1,/pub/d  | sed '/^[^ ]/,$d' | tr -d -c [:xdigit:] | cut -c 3- | xxd -p -r | xxd -i -c 16
*/
static const unsigned char gKeyPrivate[] =
{
    0x3a, 0x7c, 0x84, 0x93, 0xab, 0x4d, 0x51, 0x32, 0x07, 0x71, 0x49, 0xa5, 0xd1, 0xde, 0x93, 0xc3,
    0xf7, 0x9d, 0x34, 0xae, 0x80, 0xea, 0x4e, 0xb3, 0xd5, 0xdf, 0xb3, 0xe0, 0x9d, 0x55, 0xa8, 0x4f
};

static const unsigned char gKeyPublic[] =
{
    0x16, 0xab, 0x73, 0xed, 0x12, 0x81, 0x28, 0x74, 0xa2, 0x58, 0x71, 0xde, 0x85, 0x19, 0x27, 0x92,
    0x88, 0xee, 0x78, 0xc9, 0xb4, 0x6c, 0x41, 0xd8, 0xa2, 0x30, 0x60, 0x8e, 0x8a, 0x51, 0x5a, 0x26,
    0xcb, 0x3c, 0xf6, 0x13, 0xb1, 0x75, 0xe1, 0x52, 0x24, 0xef, 0x37, 0x50, 0x37, 0x03, 0xdd, 0x3f,
    0x10, 0x5d, 0xe8, 0x35, 0x41, 0x0f, 0x58, 0x8f, 0xc2, 0x11, 0xeb, 0xae, 0x88, 0xfa, 0x1a, 0xb7
};

/*
Create a message and sign a SHA256 appendix with ECDSA :

echo -ne "I owe you 1 beer\0000" >msg.txt
openssl dgst -sha256 -sign key256.pem msg.txt >msg.ecdsa
openssl asn1parse -inform der -in msg.ecdsa | awk '/INTEGER/ { printf "%64s", substr($7,2) }' | tr ' ' 0 | xxd -p -r | xxd -i -c 16
*/
static const unsigned char gMessage[] = {
    "I owe you 1 beer"
};

static const unsigned char gMessageSignature[] = {
    0xae, 0x0b, 0x6c, 0xd6, 0x70, 0xe0, 0x2c, 0x62, 0x3a, 0xa8, 0x78, 0x98, 0x3c, 0xad, 0x87, 0xed,
    0x3c, 0xd2, 0x97, 0x8d, 0x6f, 0xb4, 0xb7, 0xf0, 0x03, 0x91, 0xb0, 0x69, 0x5d, 0xd3, 0x46, 0x03,
    0x49, 0x78, 0x12, 0x83, 0x64, 0xdb, 0x18, 0x70, 0x08, 0xfa, 0x88, 0x74, 0x72, 0xb1, 0x5a, 0xc8,
    0x85, 0xb0, 0xb8, 0x5a, 0x04, 0xf2, 0x7a, 0x81, 0xf6, 0x94, 0x4f, 0x4e, 0xbd, 0xfc, 0x7a, 0xde
};

TEST(EccP256Test, VerifyHardcoded)
{
    nn::crypto::EcdsaP256Sha256Verifier ecdsa;
    EXPECT_TRUE( ecdsa.Verify( gMessageSignature, sizeof(gMessageSignature),
                               gKeyPublic, sizeof(gKeyPublic),
                               gMessage,sizeof(gMessage) ) );
}

TEST(EccP256Test, SignAndVerify)
{
    nn::crypto::EcdsaP256Sha256Signer gen;
    char signature[gen.SignatureSize];
    const char random[gen.RandomSize] = {
        "Fill this with real random !"
    };
    gen.Sign( signature, sizeof(signature),
              gKeyPrivate, sizeof(gKeyPrivate),
              gMessage, sizeof(gMessage), random, sizeof(random) );
    /*
    Watch signature in a Memory window, and copy-paste the 4 lines into, say, file mem.txt
    cat mem.txt | cut -d ' ' -f 2-18 | tr -d -c [:xdigit:] | cut -c -128 | awk '{printf "asn1=SEQUENCE:sig\n[sig]\nc=INT:0x" substr($1,1,64) "\nd=INT:0x" substr($1,65) "\n"}' >asn1.txt
    openssl asn1parse -genconf asn1.txt -out msg2.ecdsa
    openssl asn1parse -inform der -in msg2.ecdsa
    openssl ec -in key256.pem -pubout -out key256-public.pem
    openssl.exe dgst -sha256 -verify key256-public.pem -signature msg2.ecdsa msg.txt
    */
    nn::crypto::EcdsaP256Sha256Verifier ver;

    EXPECT_TRUE( ver.Verify( signature, sizeof(signature),
                             gKeyPublic, sizeof(gKeyPublic),
                             gMessage, sizeof(gMessage) ) );

    EXPECT_FALSE( ver.Verify( signature, sizeof(signature),
                              gKeyPublic, sizeof(gKeyPublic),
                              "another message", 15 ) );

    signature[sizeof(signature) / 3] ^= 0x10; // flip a bit
    EXPECT_FALSE( ver.Verify( signature, sizeof(signature),
                              gKeyPublic, sizeof(gKeyPublic),
                              gMessage, sizeof(gMessage) ) );
}
