﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_RsaOaepSha256Decoder.h>
#include <nn/crypto/crypto_RsaOaepSha256Decryptor.h>
#include <nn/crypto/crypto_RsaOaepSha256Encryptor.h>

#include "testCrypto_Util.h"

/*
Generate and dump keys with :

openssl genrsa -out key2048.pem 2048
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 2p | xxd -p -r | xxd -i -c 16
openssl asn1parse -in key2048.pem | awk '/INTEGER/ { printf "%512s\n", substr($7,2) }' | tr ' ' 0 | sed -n 4p | xxd -p -r | xxd -i -c 16
*/
static const unsigned char gKeyPublicN[] =
{
    0xdf, 0xa6, 0x14, 0x44, 0x94, 0x4a, 0xc2, 0xa6, 0xa8, 0x61, 0x7b, 0x09, 0x1e, 0x36, 0x35, 0xfd,
    0x09, 0x23, 0x6b, 0x3d, 0xf1, 0xd0, 0x4d, 0xcf, 0xf7, 0x60, 0x63, 0xa9, 0xdb, 0xa6, 0xa2, 0x74,
    0x28, 0x9b, 0x2a, 0xd7, 0xb3, 0xf8, 0x9d, 0x74, 0xea, 0x3f, 0x85, 0xba, 0x20, 0x94, 0xf6, 0xa5,
    0x24, 0xc2, 0xff, 0x0d, 0x32, 0xaa, 0xa6, 0xe1, 0xf9, 0x91, 0x32, 0xe3, 0x6e, 0xa8, 0x76, 0x0b,
    0xda, 0x9c, 0xef, 0xe4, 0x52, 0x94, 0x1c, 0x72, 0xc1, 0x5d, 0x9e, 0x10, 0x3c, 0x3e, 0x0c, 0x6b,
    0x3c, 0xe4, 0x12, 0x88, 0x5f, 0xb6, 0xce, 0xad, 0x3d, 0xe1, 0xc2, 0xa4, 0xe1, 0x28, 0xd3, 0x3d,
    0x05, 0x42, 0xdb, 0x64, 0x7d, 0x6e, 0x82, 0x43, 0xc1, 0x6e, 0x93, 0x31, 0x0b, 0xa5, 0x99, 0x13,
    0x5f, 0x72, 0xe3, 0x9c, 0x02, 0x05, 0x59, 0x77, 0xf1, 0x79, 0x54, 0x24, 0x65, 0x73, 0xd1, 0xf5,
    0x37, 0x2b, 0xdc, 0xd1, 0x78, 0x77, 0x04, 0xb8, 0xd5, 0x83, 0x0b, 0xe0, 0x1b, 0x4c, 0xa5, 0x1b,
    0x27, 0x82, 0x38, 0x94, 0x9c, 0x3b, 0xcf, 0x06, 0x7a, 0x33, 0xfb, 0x1b, 0xc7, 0x86, 0xcf, 0x44,
    0x3c, 0x78, 0x84, 0x10, 0x41, 0x22, 0x11, 0x1f, 0x7b, 0x37, 0xad, 0x66, 0xfd, 0x56, 0xd7, 0x61,
    0xa5, 0x32, 0xe5, 0x87, 0xd2, 0x16, 0x64, 0x12, 0xde, 0xc7, 0xbf, 0x6d, 0xb5, 0x1f, 0x6f, 0xcb,
    0xc4, 0xa1, 0x9a, 0xd3, 0xcf, 0x21, 0xa9, 0xe9, 0x02, 0x50, 0x6b, 0x13, 0x34, 0x20, 0xf0, 0x26,
    0x4a, 0x2e, 0x3e, 0x38, 0xcb, 0x4e, 0x77, 0x3b, 0x02, 0xaa, 0xfa, 0x0c, 0x82, 0xc2, 0x6c, 0x97,
    0x7e, 0x7c, 0x6f, 0x55, 0x59, 0xc9, 0xb4, 0x2b, 0x1c, 0xf7, 0xcd, 0x99, 0x15, 0xd7, 0xa8, 0x5a,
    0xe1, 0xeb, 0x67, 0x61, 0x33, 0x8e, 0xc7, 0x75, 0x28, 0xc5, 0xa9, 0x20, 0x45, 0x62, 0x7b, 0x19
};
static const unsigned char gKeyPublicE[] =
{
    0x01,0x00,0x01
};

static const unsigned char gKeyPrivateD[] =
{
    0xc9, 0x41, 0x6b, 0x95, 0xb7, 0x18, 0x7b, 0x87, 0x7b, 0x98, 0x95, 0x1e, 0x44, 0x68, 0xa6, 0x36,
    0xb8, 0xe5, 0x7d, 0x98, 0xf5, 0xa4, 0xec, 0x90, 0xcb, 0xdc, 0x9a, 0x72, 0xfc, 0xed, 0x3b, 0xdd,
    0x69, 0xb9, 0x10, 0x59, 0xa1, 0x63, 0x85, 0x55, 0xf3, 0x82, 0x0c, 0x35, 0xcd, 0x22, 0x1f, 0xbd,
    0xc6, 0x92, 0x0e, 0xf0, 0x72, 0x7a, 0xac, 0xc8, 0x47, 0x4b, 0xc9, 0x4e, 0x6e, 0x82, 0xfc, 0x0f,
    0x78, 0xbf, 0xa7, 0x04, 0x80, 0xe7, 0x35, 0x9e, 0x72, 0xb1, 0x6b, 0x0e, 0xd3, 0x17, 0x47, 0x17,
    0xad, 0x68, 0xd3, 0xe0, 0x92, 0xb9, 0x2f, 0xcd, 0x28, 0x15, 0xa3, 0x91, 0xb9, 0xe8, 0x82, 0x06,
    0xb4, 0x58, 0x50, 0x93, 0x62, 0x68, 0x72, 0xa2, 0xc4, 0x4b, 0x5b, 0xfb, 0x53, 0xae, 0xc6, 0x4c,
    0x6f, 0xa5, 0x34, 0x7b, 0x6c, 0x3a, 0x22, 0x23, 0xd4, 0x51, 0xc9, 0x4d, 0x23, 0x5a, 0x91, 0x40,
    0xf9, 0x89, 0x82, 0x91, 0x49, 0x89, 0x36, 0xdb, 0x2a, 0x45, 0xfb, 0x09, 0xa9, 0xf9, 0x00, 0xa6,
    0x4b, 0x90, 0xbf, 0xb0, 0xff, 0xa0, 0xce, 0x6b, 0xfd, 0xaa, 0x8c, 0x34, 0x5e, 0x2d, 0x6c, 0xc0,
    0xe9, 0xb5, 0x32, 0x6b, 0xc5, 0xa9, 0x77, 0xe7, 0x8f, 0xaa, 0x08, 0x0c, 0x03, 0xf1, 0x1a, 0xb1,
    0x98, 0x19, 0xc2, 0xf6, 0x39, 0x64, 0xc0, 0xfa, 0x39, 0x3e, 0xfd, 0x0a, 0x95, 0x4d, 0x7f, 0x04,
    0x80, 0xb6, 0x10, 0x49, 0xa9, 0x0d, 0x75, 0x8b, 0x46, 0xd9, 0x2d, 0xf0, 0x88, 0x1e, 0x15, 0x39,
    0xf0, 0xf1, 0x8a, 0x49, 0x49, 0xcf, 0xb1, 0x5e, 0xce, 0xb9, 0x0c, 0x93, 0x17, 0x36, 0x29, 0x14,
    0xf8, 0x49, 0xc4, 0xee, 0x93, 0xc1, 0x9d, 0xda, 0xf2, 0xfa, 0x5a, 0x73, 0xa6, 0xe4, 0xa6, 0x86,
    0x04, 0x0d, 0xe1, 0xbd, 0x48, 0x78, 0x4e, 0x71, 0x2d, 0x2b, 0x9a, 0x3a, 0x65, 0x39, 0xb5, 0x31
};

/*
Create a message :

echo -ne "I owe you 1 beer\0000" >msg.txt

Encrypt it with recovery using RSA-OAEP :
openssl rsa -in key2048.pem -pubout -out key2048-public.pem
openssl pkeyutl -encrypt -pubin -inkey key2048-public.pem -pkeyopt rsa_padding_mode:oaep -pkeyopt rsa_oaep_md:sha256 -in msg.txt -out msg.oaep
xxd -i -c 16 msg.oaep

Decrypt it with recovery using RSA-OAEP :
openssl pkeyutl -decrypt -inkey key2048.pem -pkeyopt rsa_padding_mode:oaep -pkeyopt rsa_oaep_md:sha256 -in msg.oaep
*/
static const unsigned char gMessage[] = {
    "I owe you 1 beer"
};

static const unsigned char gSeed[] = {
    0x61, 0x51, 0xcd, 0x69, 0xaa, 0x69, 0x0e, 0x92, 0x03, 0x0c, 0x95, 0xd8, 0x7e, 0xa8, 0x90, 0x54,
    0x0a, 0x6a, 0x4c, 0xc6, 0x7d, 0xf0, 0x83, 0xf5, 0x83, 0x2c, 0x4a, 0x7e, 0x81, 0x9c, 0x08, 0x38
};

static const unsigned char gCipher[] = {
    0x3b, 0xc4, 0x58, 0xcd, 0x02, 0x64, 0x9f, 0xa0, 0xdf, 0x10, 0xd3, 0xb4, 0x68, 0x55, 0x7e, 0x7b,
    0x4b, 0x6e, 0x64, 0x21, 0x99, 0xf4, 0x26, 0x70, 0x07, 0x92, 0x18, 0xea, 0xcf, 0x89, 0x13, 0xa7,
    0xa3, 0x28, 0xf3, 0x80, 0xea, 0x52, 0x9b, 0xf3, 0x83, 0x70, 0x14, 0xe8, 0x88, 0xe7, 0x87, 0x82,
    0x16, 0x2a, 0xf3, 0x51, 0x6d, 0x21, 0xb8, 0xfd, 0x27, 0x5d, 0x9e, 0x02, 0x58, 0x2a, 0xec, 0x56,
    0xd6, 0x99, 0x5a, 0x50, 0x69, 0x7e, 0xec, 0x9e, 0x6f, 0x45, 0x7d, 0xf5, 0xee, 0x55, 0x9a, 0xd4,
    0xa6, 0x8d, 0xa2, 0x2c, 0x37, 0x80, 0x89, 0x27, 0x47, 0x08, 0xb8, 0x1e, 0x95, 0x75, 0xa3, 0x53,
    0x3d, 0x8d, 0xb3, 0xc6, 0xe0, 0x09, 0x14, 0xf2, 0xc3, 0x79, 0x4b, 0x94, 0xaf, 0x14, 0x61, 0x28,
    0xa0, 0x2b, 0x84, 0x1f, 0x67, 0x8e, 0x31, 0x0a, 0xee, 0x3c, 0xd4, 0xd1, 0x0a, 0xe0, 0xb4, 0xaa,
    0x3a, 0xb0, 0x62, 0x6d, 0x0e, 0xff, 0xa9, 0x8c, 0x05, 0xe7, 0x13, 0xa8, 0x34, 0x33, 0x68, 0xf3,
    0xad, 0xc2, 0xa4, 0x43, 0x62, 0x1c, 0x16, 0x0b, 0x60, 0x44, 0x16, 0x0d, 0xa1, 0xec, 0xb7, 0xa6,
    0xd0, 0x5a, 0xb8, 0x19, 0xf5, 0xc6, 0x3e, 0x28, 0x39, 0x35, 0x32, 0x52, 0x38, 0xed, 0xbf, 0x03,
    0xa4, 0x80, 0x60, 0xf9, 0xa3, 0xce, 0xb2, 0x2f, 0xc4, 0x42, 0x61, 0xe3, 0x0f, 0xcb, 0xbb, 0xe5,
    0xe4, 0xe8, 0x44, 0xfa, 0x8a, 0x39, 0x76, 0x8c, 0x30, 0xa8, 0xab, 0x14, 0x04, 0xab, 0xbc, 0x20,
    0x4a, 0x5f, 0xfa, 0x72, 0xde, 0x01, 0x18, 0xe0, 0xf9, 0xbc, 0xf8, 0x41, 0xaa, 0x9e, 0xf9, 0x19,
    0x37, 0x81, 0x1f, 0xf7, 0x0c, 0x24, 0x74, 0xbc, 0x4a, 0x95, 0x73, 0xe7, 0x6c, 0x35, 0x41, 0x28,
    0xa3, 0x12, 0xb0, 0xf4, 0x4d, 0x2c, 0x9b, 0x4b, 0x0d, 0x43, 0xa2, 0xd4, 0x86, 0x30, 0x30, 0x09
};

/* Label digest for null message */
static const unsigned char gLabelDigest[] = {
    0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14, 0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
    0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c, 0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55
};

/*
Cipher text when specifying the non-null label string.
Keys, message, seed is the same as above.

This result was calculated by OpenSSL API (not command)
*/
static const unsigned char gLabel2[] = {
    "optional label"
};

static const unsigned char gLabel2Digest[] = {
    0xe6, 0x79, 0x4a, 0x4b, 0x75, 0xdd, 0xb1, 0x6a, 0xa1, 0x41, 0x3f, 0x44, 0x0f, 0xd8, 0xf1, 0x13,
    0x5d, 0x3a, 0xa5, 0x9e, 0xf2, 0x9f, 0x00, 0xe6, 0xcd, 0x7c, 0xcb, 0x81, 0x50, 0x52, 0x7e, 0x60
};

static const unsigned char gCipher2[] = {
    0xab, 0xfb, 0x08, 0x67, 0x21, 0xbd, 0xa7, 0xb8, 0x67, 0x47, 0x2b, 0x4d, 0x73, 0x37, 0x97, 0x30,
    0xa2, 0xf4, 0xca, 0x01, 0x3e, 0xf5, 0x6b, 0xf6, 0xe4, 0xb7, 0x2b, 0xf9, 0xeb, 0x3f, 0x27, 0x31,
    0x2d, 0xf5, 0xc8, 0x78, 0xf2, 0x0a, 0x56, 0x79, 0x97, 0xe5, 0x69, 0x0a, 0x0b, 0x98, 0xdb, 0x22,
    0x31, 0x90, 0x75, 0xb5, 0x2d, 0xba, 0x6a, 0x94, 0xfe, 0x19, 0xda, 0xe5, 0xe2, 0xb7, 0x7f, 0xea,
    0xa3, 0x1e, 0xb3, 0x5a, 0x01, 0x45, 0xa5, 0xe6, 0x12, 0x30, 0xa4, 0x48, 0x85, 0x92, 0x1d, 0x72,
    0xbf, 0xcc, 0x8e, 0x28, 0x35, 0x9c, 0xa2, 0x08, 0x97, 0x8c, 0x74, 0x99, 0xc7, 0x9b, 0x89, 0x90,
    0x3d, 0x8a, 0x6e, 0xea, 0x0c, 0xc1, 0x88, 0xec, 0x86, 0x10, 0x14, 0xc9, 0xe1, 0x40, 0xac, 0x7f,
    0x98, 0xb1, 0x16, 0xe6, 0xca, 0xe2, 0xaf, 0x93, 0x93, 0x2d, 0x1c, 0x2a, 0x82, 0xf1, 0xb8, 0x66,
    0xc5, 0xcf, 0xa0, 0x61, 0x34, 0xd6, 0x04, 0x5e, 0xcf, 0x4b, 0x99, 0x1e, 0xaf, 0xef, 0x7b, 0x1d,
    0x1d, 0xca, 0x68, 0xa1, 0x88, 0x2f, 0xe4, 0x59, 0x3f, 0x43, 0xfd, 0x4e, 0x8a, 0x0d, 0x88, 0x66,
    0xd1, 0xd2, 0x78, 0xec, 0x5e, 0xf7, 0xb7, 0x2d, 0x7b, 0xeb, 0x59, 0x3e, 0xb8, 0x3c, 0xfa, 0x66,
    0x2c, 0x13, 0xc2, 0xc1, 0x31, 0xfc, 0x24, 0xaa, 0x81, 0xfe, 0xe1, 0x7b, 0xe9, 0xe2, 0x12, 0x34,
    0x2c, 0x4b, 0x5c, 0xa7, 0x04, 0x97, 0xa0, 0xe3, 0x4b, 0x19, 0xe5, 0x7c, 0x41, 0x12, 0x0b, 0x72,
    0x65, 0x64, 0x9d, 0x6b, 0x11, 0xb0, 0x05, 0x2c, 0xf1, 0x63, 0x5c, 0x60, 0xdd, 0xf3, 0x4c, 0x94,
    0xae, 0x34, 0x0f, 0x12, 0x7c, 0x06, 0x5f, 0x89, 0xfa, 0x1f, 0x4b, 0x6e, 0xae, 0x25, 0x91, 0x91,
    0x34, 0xe8, 0x24, 0xca, 0xc0, 0x6f, 0x88, 0x51, 0x2f, 0x75, 0xe4, 0xca, 0x8f, 0xb6, 0x38, 0xb1
};

TEST(RsaTest, OaepEncrypt)
{
    // 1. Class interface
    nn::crypto::Rsa2048OaepSha256Encryptor oaep;
    oaep.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));
    char cipher[sizeof(gCipher)];

    EXPECT_LE(sizeof(cipher), nn::crypto::Rsa2048OaepSha256Encryptor::BlockSize);

    std::memset(cipher, 0xBD, sizeof(cipher));
    oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed));
    EXPECT_ARRAY_EQ(cipher, gCipher, sizeof(cipher));

    // 2. Function interface without work buffer
    std::memset(cipher, 0xBD, sizeof(cipher));
    nn::crypto::EncryptRsa2048OaepSha256(
        cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
        gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), nullptr, 0 );
    EXPECT_ARRAY_EQ(cipher, gCipher, sizeof(cipher));

    // 3. Function interface with work buffer
    uint8_t workBuffer[nn::crypto::Rsa2048OaepSha256Encryptor::RequiredWorkBufferSize];

    std::memset(cipher, 0xBD, sizeof(cipher));
    nn::crypto::EncryptRsa2048OaepSha256(
        cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
        gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), nullptr, 0, workBuffer, sizeof(workBuffer) );
    EXPECT_ARRAY_EQ(cipher, gCipher, sizeof(cipher));

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Encrypt() with invalid Seed size
    EXPECT_DEATH_IF_SUPPORTED( oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage),
                                            gSeed, sizeof(gSeed) * 160 / 256), "" );
#endif
}

TEST(RsaTest, OaepDecrypt)
{
    // 1. Class interface
    nn::crypto::Rsa2048OaepSha256Decryptor oaep;
    oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );
    char plain[sizeof(gMessage)];

    EXPECT_LE(sizeof(plain), nn::crypto::Rsa2048OaepSha256Decryptor::BlockSize);

    std::memset(plain, 0xBD, sizeof(plain));
    size_t sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher, sizeof(gCipher) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 2. Function interface without work buffer
    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = nn::crypto::DecryptRsa2048OaepSha256(
        plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
        gCipher, sizeof(gCipher), nullptr, 0 );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 3. Function interface with work buffer
    uint8_t workBuffer[nn::crypto::Rsa2048OaepSha256Decryptor::RequiredWorkBufferSize];

    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = nn::crypto::DecryptRsa2048OaepSha256(
        plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
        gCipher, sizeof(gCipher), nullptr, 0, workBuffer, sizeof(workBuffer) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));
}

TEST(RsaTest, OaepEncryptWithLabel)
{
    // 1. Class interface
    nn::crypto::Rsa2048OaepSha256Encryptor oaep;
    oaep.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));
    char cipher[sizeof(gCipher)];

    EXPECT_LE(sizeof(cipher), nn::crypto::Rsa2048OaepSha256Encryptor::BlockSize);

    oaep.UpdateLabel(gLabel2, sizeof(gLabel2));

    std::memset(cipher, 0xBD, sizeof(cipher));
    oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed));
    EXPECT_ARRAY_EQ(cipher, gCipher2, sizeof(cipher));

    // 2. Class interface with label digest
    oaep.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));

    oaep.SetLabelDigest(gLabel2Digest, sizeof(gLabel2Digest));

    std::memset(cipher, 0xBD, sizeof(cipher));
    oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed));
    EXPECT_ARRAY_EQ(cipher, gCipher2, sizeof(cipher));

    // 3. Function interface without work buffer
    std::memset(cipher, 0xBD, sizeof(cipher));
    nn::crypto::EncryptRsa2048OaepSha256(
        cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
        gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), gLabel2, sizeof(gLabel2) );
    EXPECT_ARRAY_EQ(cipher, gCipher2, sizeof(cipher));

    // 4. Function interface with work buffer
    uint8_t workBuffer[nn::crypto::Rsa2048OaepSha256Encryptor::RequiredWorkBufferSize];

    std::memset(cipher, 0xBD, sizeof(cipher));
    nn::crypto::EncryptRsa2048OaepSha256(
        cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
        gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), gLabel2, sizeof(gLabel2), workBuffer, sizeof(workBuffer) );
    EXPECT_ARRAY_EQ(cipher, gCipher2, sizeof(cipher));
}

TEST(RsaTest, OaepDecryptWithLabel)
{
    // 1. Class interface
    nn::crypto::Rsa2048OaepSha256Decryptor oaep;
    oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );
    char plain[sizeof(gMessage)];

    EXPECT_LE(sizeof(plain), nn::crypto::Rsa2048OaepSha256Decryptor::BlockSize);

    oaep.UpdateLabel(gLabel2, sizeof(gLabel2));

    std::memset(plain, 0xBD, sizeof(plain));
    size_t sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher2, sizeof(gCipher2) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 2. Class interface with label digest
    oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );

    oaep.SetLabelDigest(gLabel2Digest, sizeof(gLabel2Digest));

    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher2, sizeof(gCipher2) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 3. Function interface without work buffer
    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = nn::crypto::DecryptRsa2048OaepSha256(
        plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
        gCipher2, sizeof(gCipher2), gLabel2, sizeof(gLabel2) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 4. Function interface with work buffer
    uint8_t workBuffer[nn::crypto::Rsa2048OaepSha256Decryptor::RequiredWorkBufferSize];

    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = nn::crypto::DecryptRsa2048OaepSha256(
        plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
        gCipher2, sizeof(gCipher2), gLabel2, sizeof(gLabel2), workBuffer, sizeof(workBuffer) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

    // 5. Invalid but non-null label will cause decryption failure
    const char* label = "invalid label";

    oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );

    oaep.UpdateLabel(label, sizeof(label));

    std::memset(plain, 0xBD, sizeof(plain));
    sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher2, sizeof(gCipher2) );
    EXPECT_EQ(0, sizeOut);
}

TEST(RsaTest, OaepDecode)
{
    nn::crypto::detail::RsaOaepPadding<nn::crypto::Sha256Generator> oaep;
    char cipher[sizeof(gCipher)];
    char plain[sizeof(gMessage)];

    // Encode message
    oaep.BuildPad(cipher, sizeof(cipher), gLabelDigest, sizeof(gLabelDigest),
                  gMessage, sizeof(gMessage), gSeed, sizeof(gSeed));

    // Decode message
    size_t sizeOut = nn::crypto::DecodeRsa2048OaepSha256(plain, sizeof(plain),
                                                         gLabelDigest, sizeof(gLabelDigest),
                                                         cipher, sizeof(cipher));
    EXPECT_EQ(sizeOut, sizeof(gMessage));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));
}

TEST(RsaTest, OaepLabelDigest)
{
    // Encryption
    {
        nn::crypto::Rsa2048OaepSha256Encryptor oaep;
        oaep.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));
        char cipher[sizeof(gCipher)];

        EXPECT_LE(sizeof(cipher), nn::crypto::Rsa2048OaepSha256Encryptor::BlockSize);

        oaep.SetLabelDigest(gLabelDigest, sizeof(gLabelDigest));

        std::memset(cipher, 0xBD, sizeof(cipher));
        oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed));
        EXPECT_ARRAY_EQ(cipher, gCipher, sizeof(cipher));
    }

    // Decryption
    {
        nn::crypto::Rsa2048OaepSha256Decryptor oaep;
        oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );
        char plain[sizeof(gMessage)];

        EXPECT_LE(sizeof(plain), nn::crypto::Rsa2048OaepSha256Decryptor::BlockSize);

        oaep.SetLabelDigest(gLabelDigest, sizeof(gLabelDigest));

        std::memset(plain, 0xBD, sizeof(plain));
        size_t sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher, sizeof(gCipher) );
        EXPECT_EQ(sizeOut, sizeof(plain));
        EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));
    }
}

TEST(RsaTest, OaepEncryptStates)
{
    nn::crypto::Rsa2048OaepSha256Encryptor oaep;
    char cipher[sizeof(gCipher)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling UpdateLabel() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.UpdateLabel( gMessage, sizeof(gMessage) ), "");

    // Calling Encrypt() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed)), "");
#endif

    oaep.Initialize(gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE));

    EXPECT_LE(sizeof(cipher), nn::crypto::Rsa2048OaepSha256Encryptor::BlockSize);

    std::memset(cipher, 0xBD, sizeof(cipher));
    oaep.Encrypt(cipher, sizeof(cipher), gMessage,sizeof(gMessage), gSeed,sizeof(gSeed));
    EXPECT_ARRAY_EQ(cipher, gCipher, sizeof(cipher));

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling UpdateLabel() after Encrypt() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.UpdateLabel( gMessage, sizeof(gMessage) ), "");
#endif
}

TEST(RsaTest, OaepDecryptStates)
{
    nn::crypto::Rsa2048OaepSha256Decryptor oaep;
    char plain[sizeof(gMessage)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling UpdateLabel() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.UpdateLabel( gMessage, sizeof(gMessage) ), "");

    // Calling Decrypt() without Initialize() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.Decrypt( plain, sizeof(plain), gCipher, sizeof(gCipher) ), "");
#endif

    oaep.Initialize( gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD) );

    EXPECT_LE(sizeof(plain), nn::crypto::Rsa2048OaepSha256Decryptor::BlockSize);

    std::memset(plain, 0xBD, sizeof(plain));
    size_t sizeOut = oaep.Decrypt( plain, sizeof(plain), gCipher, sizeof(gCipher) );
    EXPECT_EQ(sizeOut, sizeof(plain));
    EXPECT_ARRAY_EQ(plain, gMessage, sizeof(plain));

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling UpdateLabel() after Decrypt() fails
    EXPECT_DEATH_IF_SUPPORTED(oaep.UpdateLabel( gMessage, sizeof(gMessage) ), "");
#endif
}

static const unsigned char gZero[2048 / 8] =
{
    0
};

TEST(RsaTest, OaepZeroes)
{
    nn::crypto::Rsa2048OaepSha256Decryptor dec;
    // Calling Initialize() with invalid key does not assert
    EXPECT_FALSE(dec.Initialize(gZero, sizeof(gZero), gZero, sizeof(gZero)));

    char plain[sizeof(gMessage)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Decrypt() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(dec.Decrypt(plain, sizeof(plain), gCipher, sizeof(gCipher)), "");
#endif

    std::memset(plain, 0xBD, sizeof(plain));
    size_t sizeOut = nn::crypto::Rsa2048OaepSha256Decryptor::Decrypt(
        plain, sizeof(plain), gZero, sizeof(gZero), gZero, sizeof(gZero),
        gCipher, sizeof(gCipher), nullptr, 0);
    EXPECT_EQ(sizeOut, 0);

    // Calling Initialize() with invalid key does not assert
    nn::crypto::Rsa2048OaepSha256Encryptor enc;
    EXPECT_FALSE(enc.Initialize(gZero, sizeof(gZero), gZero, sizeof(gZero)));

    char cipher[sizeof(gCipher)];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // Calling Decrypt() fails if Initialize fails
    EXPECT_DEATH_IF_SUPPORTED(enc.Encrypt(cipher, sizeof(cipher), gMessage, sizeof(gMessage), gSeed,
                                          sizeof(gSeed)), "");
#endif

    std::memset(cipher, 0xBD, sizeof(cipher));
    bool enc0Ok = nn::crypto::Rsa2048OaepSha256Encryptor::Encrypt(
        cipher, sizeof(cipher), gZero, sizeof(gZero), gZero, sizeof(gZero),
        gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), nullptr, 0);
    EXPECT_FALSE(enc0Ok);
}

TEST(RsaTest, OaepMemoryShortage)
{
    uint8_t workBuffer[nn::crypto::Rsa2048OaepSha256Encryptor::RequiredWorkBufferSize];

    // Encrypt
    {
        char cipher[sizeof(gCipher)];

        // 0 バイトから 1KB 刻みくらいで複数パターンチェックしておく
        for (size_t bufferSize = 0; bufferSize < sizeof(workBuffer); bufferSize += 1024)
        {
            // ワークバッファが足りないと Develop ビルドでは事前条件チェックで ASSERT され、
            // Release ビルドでは false が返される
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            EXPECT_DEATH_IF_SUPPORTED(
                nn::crypto::EncryptRsa2048OaepSha256(
                    cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
                    gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), nullptr, 0, workBuffer, bufferSize ), ""
            );
#else
            EXPECT_FALSE(
                nn::crypto::EncryptRsa2048OaepSha256(
                    cipher, sizeof(cipher), gKeyPublicN, sizeof(gKeyPublicN), gKeyPublicE, sizeof(gKeyPublicE),
                    gMessage, sizeof(gMessage), gSeed, sizeof(gSeed), nullptr, 0, workBuffer, bufferSize )
            );
#endif
        }
    }

    // Decrypt
    {
        char plain[sizeof(gMessage)];

        // 0 バイトから 1KB 刻みくらいで複数パターンチェックしておく
        for (size_t bufferSize = 0; bufferSize < sizeof(workBuffer); bufferSize += 1024)
        {
            // ワークバッファが足りないと Develop ビルドでは事前条件チェックで ASSERT され、
            // Release ビルドでは false が返される
#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
            EXPECT_DEATH_IF_SUPPORTED(
                nn::crypto::DecryptRsa2048OaepSha256(
                    plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
                    gCipher, sizeof(gCipher), nullptr, 0, workBuffer, bufferSize ), ""
            );
#else
            EXPECT_EQ(
                0,
                nn::crypto::DecryptRsa2048OaepSha256(
                    plain, sizeof(plain), gKeyPublicN, sizeof(gKeyPublicN), gKeyPrivateD, sizeof(gKeyPrivateD),
                    gCipher, sizeof(gCipher), nullptr, 0, workBuffer, bufferSize )
            );
#endif
        }
    }
}
