﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_Md5Generator.h>
#include <nn/nn_Log.h>

#include "testCrypto_Util.h"

struct Md5TestVector
{
    const char* data;
    size_t      repeat;
    const char* expectedHash;
};

/* RFC1321 の A.5 Test suite を参照 */
const Md5TestVector md5TestVectors[] =
{
    //  0 byte (null message) (入力メッセージがない)
    {
        "",
        1,
        "\xD4\x1D\x8C\xD9\x8F\x00\xB2\x04\xE9\x80\x09\x98\xEC\xF8\x42\x7E",
    },
    {
        "a",
        1,
        "\x0C\xC1\x75\xB9\xC0\xF1\xB6\xA8\x31\xC3\x99\xE2\x69\x77\x26\x61",
    },
    {
        "abc",
        1,
        "\x90\x01\x50\x98\x3C\xD2\x4F\xB0\xD6\x96\x3F\x7D\x28\xE1\x7F\x72",
    },
    {
        "message digest",
        1,
        "\xF9\x6B\x69\x7D\x7C\xB7\x93\x8D\x52\x5A\x2F\x31\xAA\xF1\x61\xD0",
    },
    {
        "abcdefghijklmnopqrstuvwxyz",
        1,
        "\xC3\xFC\xD3\xD7\x61\x92\xE4\x00\x7D\xFB\x49\x6C\xCA\x67\xE1\x3B",
    },
    {
        "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
        1,
        "\xD1\x74\xAB\x98\xD2\x77\xD9\xF5\xA5\x61\x1C\x2C\x9F\x41\x9D\x9F",
    },
    {
        "1234567890",
        8,
        "\x57\xED\xF4\xA2\x2B\xE3\xC9\x55\xAC\x49\xDA\x2E\x21\x07\xB6\x7A",
    },
    {
        "FHXXKCVGGOFRILWGDLFQLZBLQSNYXWUCTXCBBVHUDWXLSFNIFTBNCGOMLXPQ"
        "QFFJEOBXFBZNEDWMSKPPEWXONABBPRBHVDNQXPRHQGMLUHJVUPOXYUHDEXYM"
        "ZBJCFNYZSKMPNLPYADWGXEKPYPKGNHQZMPBMSCYYYUPVPQZSYPPPIGKBQZUW"
        "OUCMVLFKORUIZWVWFVBEPZBXASLDAVBEHMJGYPYBQZAKHNLOFCPQVRRGKOVV"
        "RNERIFFJZIYAECMNHTRCUWGAKRZXPORLXLRXVRVVJIRTNDDIEFNGWFXBFDPP"
        "FOMPFWRGXQUBJYTNCCXKTUGRQYYGRFMDPKTWWFREVAUNHSAWUQQJOPIGRWHJ"
        "DDOVHZHHGXDKVSDYRGXGCWMKQSENCIMJQOGNGAZFTDXYEVNRDWDKCKPJPIXZ"
        "KCGJPEGFIJANNPJYHIRLPUBQAJSFMYLVGCPQZRUNKRRWEBLZYYOQUHGEMKGC"
        "XPKBWRXRANLRRHLGZSMONYDRIFPHOYSVYKZJRUSOVVUCULTSBOSGKYUVYCVH"
        "MIASIBFKGPRCKLVXZAGKCQUKMIHOIQJRBCJBKOYRNACLNHPSDBKUFBXAODNV"
        "HVTVVHHZEBAVRVMKBBEOYJSABXUDBJQEFLLYVBPWMSQQVLTGCASWRSHPIDOZ"
        "RPZUCHYREWFCDZJOMMUGPHPTKWEIJIIGEHNFXKLWIHPKYORNBXESRYZMUNTJ"
        "DGNQFGAFFRAWCKYRBZGNBZKOWAGWXZCOOEERHWPYIVXHDSZAVYFUBXNRJCVD"
        "DUJWWMYYXMCTXANZOPVULYAGPXZUQRBOAWEJKTUKIIDZVBVQRKINSAMSYKZV"
        "CINUUWVBVVUGGXLKMVBTFEESRQTNPOEYZRZVJBUEEXIQOXEWLMUUJQUWNOQT"
        "QQNJFBDNTCXMNJMJBIOCQZUCKCZFECNAXBXHLTCVURSEBLVAUIJKVGWZETIV"
        "GHWOFGRFPLIWQAUMTZOELCXBMVLHBRFZUIEIYWWUY",
        1,
        "\x72\xcc\x6a\x07\x35\x47\x48\x5d\xe7\x53\xcd\xa5\x90\xcb\x90\xbd",
    },
    {
        "YEDCGPVKFGWXYSDLTQOYAFHSJYABKIMQHJGDCKUSLRDQFJDGPVWBSBQ",
        2,
        "\x5c\x07\x79\xcb\x86\xb1\x89\x2b\xe5\x3b\xdc\x72\x7b\x3b\x9b\x41",
    },
    {
        "YEDCGPVKFGWXYSDLTQOYAFHSJYABKIMQHJGDCKUSLRDQFJDGPVWBSBQZ",
        2,
        "\xc8\x69\xd4\x26\xf3\x4f\x78\x1b\xed\xaf\xdc\x9d\x29\xaf\x75\x7f",
    },
    {
        "YEDCGPVKFGWXYSDLTQOYAFHSJYABKIMQHJGDCKUSLRDQFJDGPVWBSBQZZ",
        2,
        "\xa3\x1b\xd7\x8a\x72\x15\xa7\xc8\x59\x41\x51\xd5\x4b\xc8\x7d\x30"
    },
    // 内部バッファの境界値テスト
    {
        "012345678901234567890123456789012345678901234567890123456789",
        1,
        "\x1c\xed\x81\x1a\xf4\x7e\xad\x37\x48\x72\xfc\xca\x9d\x73\xdd\x71",
    },
    {
        "012345678901234567890123456789012345678901234567890123456789012",
        1,
        "\xc5\xe2\x56\x43\x7e\x75\x80\x92\xdb\xfe\x06\x28\x3e\x48\x90\x19",
    },
    {
        "0123456789012345678901234567890123456789012345678901234567890123",
        1,
        "\x7f\x7b\xfd\x34\x87\x09\xde\xea\xac\xe1\x9e\x3f\x53\x5f\x8c\x54",
    },
    {
        "01234567890123456789012345678901234567890123456789012345678901234",
        1,
        "\xbe\xb9\xf4\x8b\xc8\x02\xca\x5c\xa0\x43\xbc\xc1\x5e\x21\x9a\x5a",
    }
};


void HexDump(const char* array, size_t size)
{
    for(size_t i = 0; i < size; i++)
    {
        NN_LOG("%02x", array[i] & 0xFF);
    }
}

void HashDump(const char* expected, const char* result)
{
    NN_LOG("Expected:");
    HexDump(expected, nn::crypto::Md5Generator::HashSize);
    NN_LOG("\n");
    NN_LOG("Result:  ");
    HexDump(result, nn::crypto::Md5Generator::HashSize);
    NN_LOG("\n");

}

const int TestVectorCount = sizeof(md5TestVectors) / sizeof(md5TestVectors[0]);

void Md5BasicTest(const Md5TestVector& testVector)
{
    nn::crypto::Md5Generator context;
    nn::Bit8                 hash[nn::crypto::Md5Generator::HashSize];

    context.Initialize();
    for (int j = 0; j < static_cast<int>(testVector.repeat); j++)
    {
        context.Update(testVector.data, strlen(testVector.data));
    }
    context.GetHash(hash, sizeof(hash));

    EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Md5Generator::HashSize);

    // HashDump(testVector.expectedHash, reinterpret_cast<const char*>(hash));

}

void Md5FunctionTest(const Md5TestVector& testVector)
{
    nn::Bit8  hash[nn::crypto::Md5Generator::HashSize];

    if (testVector.repeat == 1)
    {
        nn::crypto::GenerateMd5Hash(hash,
                                     sizeof(hash),
                                     testVector.data,
                                     strlen(testVector.data));

        EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Md5Generator::HashSize);
    }
}

/**
  @brief   Md5Generator クラスによるハッシュ計算をテストします。

  @details
  RFC1321 の A.5 Test suiteを用いて、
  正しいハッシュ値が計算されることを確認します。
 */
TEST(Md5Test, Basic)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        Md5BasicTest(md5TestVectors[i]);
    }
}

/**
  @brief   GenerateMd5Hash 関数によるハッシュ計算をテストします。

  @details
  RFC1321 の A.5 Test suiteを用いて、
  正しいハッシュ値が計算されることを確認します。
 */
TEST(Md5Test, FunctionInterface)
{
    // 短いテストベクトルについてのみテストする
    for (int i = 0; i < TestVectorCount; i++)
    {
        Md5FunctionTest(md5TestVectors[i]);
    }
}

/**
  @brief   Md5Generator クラスの状態遷移をテストします。
 */
TEST(Md5Test, StateTransition)
{
    nn::crypto::Md5Generator md5;
    nn::Bit8                  hash[nn::crypto::Md5Generator::HashSize];
    nn::Bit8                  hash2[nn::crypto::Md5Generator::HashSize];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化せずに Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(md5.Update(md5TestVectors[0].data, strlen(md5TestVectors[0].data)), "");

    // 初期化せずに GetHash が呼ばれても NG
    EXPECT_DEATH_IF_SUPPORTED(md5.GetHash(hash, sizeof(hash)), "");
#endif

    // 初期化
    md5.Initialize();

    // 初期化の後に GetHash が呼ばれても大丈夫
    md5.GetHash(hash, sizeof(hash));

    // GetHash は連続で呼んでも大丈夫で、同じ値が出力されるはず
    md5.GetHash(hash2, sizeof(hash2));

    EXPECT_ARRAY_EQ(hash, hash2, nn::crypto::Md5Generator::HashSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetHash の後に Update が呼ばれたら NG (Release ビルドでは ASSERT が無効のため停止しない)
    EXPECT_DEATH_IF_SUPPORTED(md5.Update(md5TestVectors[0].data, strlen(md5TestVectors[0].data)), "");
#endif
}

/**
  @brief   非 32bit アラインメントの入出力バッファに対する挙動をテストします。
 */
TEST(Md5Test, UnalignedIoBuffer)
{
    nn::crypto::Md5Generator md5;
    nn::Bit8                  hash[nn::crypto::Md5Generator::HashSize + sizeof(nn::Bit64)] = {0};
    nn::Bit8                  data[nn::crypto::Md5Generator::BlockSize + sizeof(nn::Bit64)] = {0};

    Md5TestVector testVector =
    {
        "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
        1,
        "\x82\x15\xef\x07\x96\xa2\x0b\xca\xaa\xe1\x16\xd3\x87\x6c\x66\x4a",
    };

    // 非 32bit アラインメントの入力バッファに対して問題ないことをテストする
    for (int i = 0; i < static_cast<int>(sizeof(nn::Bit64)); i++)
    {
        nn::Bit8* head = data + i;

        std::memcpy(head, testVector.data, strlen(testVector.data));
        md5.Initialize();
        md5.Update(head, strlen(testVector.data));
        md5.GetHash(hash, nn::crypto::Md5Generator::HashSize);

        EXPECT_ARRAY_EQ(hash, testVector.expectedHash, nn::crypto::Md5Generator::HashSize);
    }

    // 非 32bit アラインメントの出力バッファに対して問題ないことをテストする
    for (int i = 0; i < static_cast<int>(sizeof(nn::Bit64)); i++)
    {
        nn::Bit8* head = hash + i;

        md5.Initialize();
        md5.Update(testVector.data, strlen(testVector.data));
        md5.GetHash(head, nn::crypto::Md5Generator::HashSize);

        EXPECT_ARRAY_EQ(head, testVector.expectedHash, nn::crypto::Md5Generator::HashSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
TEST(Md5Test, Destructor)
{
    nn::crypto::Md5Generator md5;
    md5.Initialize();

    // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
    EXPECT_ARRAY_NONZERO(&md5, sizeof(md5));
    md5.~Md5Generator();
    EXPECT_ARRAY_ZERO(&md5, sizeof(md5));
}
