﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/crypto/crypto_Compare.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Tick.h>
#include <cstring>

#include "testCrypto_Util.h"

namespace {

template <typename Function>
void MeasureTime(Function function, const void* pBuffer1, const void* pBuffer2, size_t size)
{
    const auto trialCount = 1000000;

    auto start = nn::os::GetSystemTick();
    for (auto i = 0; i < trialCount; ++i)
    {
        function(pBuffer1, pBuffer2, size);
    }
    auto end = nn::os::GetSystemTick();

    auto time = nn::os::ConvertToTimeSpan(end - start);
    NN_LOG("time: %lld us\n", time.GetMicroSeconds());
}

}

/**
  @brief   一致をテストします。
 */
TEST(CompareTest, SameBytes)
{
    const nn::Bit8 buffer1[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    const nn::Bit8 buffer2[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10
    };
    EXPECT_EQ(nn::crypto::IsSameBytes(buffer1, buffer2, sizeof(buffer1)), true);
}

/**
  @brief   不一致をテストします。
 */
TEST(CompareTest, DifferentBytes)
{
    const nn::Bit8 buffer1[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    const nn::Bit8 buffer2[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0E
    };
    EXPECT_EQ(nn::crypto::IsSameBytes(buffer1, buffer2, sizeof(buffer1)), false);
}

/**
  @brief   一致のときと不一致のときの処理時間を計測します。
 */
TEST(CompareTest, DISABLED_TimeMeasurement)
{
    const nn::Bit8 buffer1[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    const nn::Bit8 buffer2[] =
    {
        0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    const nn::Bit8 buffer3[] =
    {
        0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
    };

    NN_LOG("IsSameBytes same\n");
    MeasureTime(nn::crypto::IsSameBytes, buffer1, buffer2, sizeof(buffer1));
    NN_LOG("IsSameBytes different\n");
    MeasureTime(nn::crypto::IsSameBytes, buffer1, buffer3, sizeof(buffer1));
}
