﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/nn_Common.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/crypto/crypto_Aes128XtsEncryptor.h>
#include <nn/crypto/crypto_Aes128XtsDecryptor.h>

#include "testCrypto_Util.h"

struct AesXtsTestVector
{
    nn::Bit8  key1[32];
    nn::Bit8  key2[32];
    size_t    keySize;
    nn::Bit8  tweak[16];
    size_t    tweakSize;
    nn::Bit8  plainText[512];
    size_t    textSize;
    nn::Bit8  cipherText[512];
};

const AesXtsTestVector aesXtsTestVectors[] =
{
    // #1) シンプルなケース
    {
        {0},
        {0},
        16,
        {0},
        16,
        {0},
        32,
        {0x91, 0x7c, 0xf6, 0x9e, 0xbd, 0x68, 0xb2, 0xec, 0x9b, 0x9f, 0xe9, 0xa3, 0xea, 0xdd, 0xa6, 0x92,
         0xcd, 0x43, 0xd2, 0xf5, 0x95, 0x98, 0xed, 0x85, 0x8c, 0x02, 0xc2, 0x65, 0x2f, 0xbf, 0x92, 0x2e},
    },

    // #2) Tweak != 0
    {
        {0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11},
        {0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22},
        16,
        {0x33, 0x33, 0x33, 0x33, 0x33},
        16,
        {0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44,
         0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44},
        32,
        {0xc4, 0x54, 0x18, 0x5e, 0x6a, 0x16, 0x93, 0x6e, 0x39, 0x33, 0x40, 0x38, 0xac, 0xef, 0x83, 0x8b,
         0xfb, 0x18, 0x6f, 0xff, 0x74, 0x80, 0xad, 0xc4, 0x28, 0x93, 0x82, 0xec, 0xd6, 0xd3, 0x94, 0xf0},
    },

    // #3) 比較的長めのメッセージ
    {
        {0x27, 0x18, 0x28, 0x18, 0x28, 0x45, 0x90, 0x45, 0x23, 0x53, 0x60, 0x28, 0x74, 0x71, 0x35, 0x26},
        {0x31, 0x41, 0x59, 0x26, 0x53, 0x58, 0x97, 0x93, 0x23, 0x84, 0x62, 0x64, 0x33, 0x83, 0x27, 0x95},
        16,
        {0},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
         0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
         0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
         0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
         0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
         0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
         0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
         0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
         0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
         0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
         0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
         0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
         0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
         0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        512,
        {0x27, 0xa7, 0x47, 0x9b, 0xef, 0xa1, 0xd4, 0x76, 0x48, 0x9f, 0x30, 0x8c, 0xd4, 0xcf, 0xa6, 0xe2,
         0xa9, 0x6e, 0x4b, 0xbe, 0x32, 0x08, 0xff, 0x25, 0x28, 0x7d, 0xd3, 0x81, 0x96, 0x16, 0xe8, 0x9c,
         0xc7, 0x8c, 0xf7, 0xf5, 0xe5, 0x43, 0x44, 0x5f, 0x83, 0x33, 0xd8, 0xfa, 0x7f, 0x56, 0x00, 0x00,
         0x05, 0x27, 0x9f, 0xa5, 0xd8, 0xb5, 0xe4, 0xad, 0x40, 0xe7, 0x36, 0xdd, 0xb4, 0xd3, 0x54, 0x12,
         0x32, 0x80, 0x63, 0xfd, 0x2a, 0xab, 0x53, 0xe5, 0xea, 0x1e, 0x0a, 0x9f, 0x33, 0x25, 0x00, 0xa5,
         0xdf, 0x94, 0x87, 0xd0, 0x7a, 0x5c, 0x92, 0xcc, 0x51, 0x2c, 0x88, 0x66, 0xc7, 0xe8, 0x60, 0xce,
         0x93, 0xfd, 0xf1, 0x66, 0xa2, 0x49, 0x12, 0xb4, 0x22, 0x97, 0x61, 0x46, 0xae, 0x20, 0xce, 0x84,
         0x6b, 0xb7, 0xdc, 0x9b, 0xa9, 0x4a, 0x76, 0x7a, 0xae, 0xf2, 0x0c, 0x0d, 0x61, 0xad, 0x02, 0x65,
         0x5e, 0xa9, 0x2d, 0xc4, 0xc4, 0xe4, 0x1a, 0x89, 0x52, 0xc6, 0x51, 0xd3, 0x31, 0x74, 0xbe, 0x51,
         0xa1, 0x0c, 0x42, 0x11, 0x10, 0xe6, 0xd8, 0x15, 0x88, 0xed, 0xe8, 0x21, 0x03, 0xa2, 0x52, 0xd8,
         0xa7, 0x50, 0xe8, 0x76, 0x8d, 0xef, 0xff, 0xed, 0x91, 0x22, 0x81, 0x0a, 0xae, 0xb9, 0x9f, 0x91,
         0x72, 0xaf, 0x82, 0xb6, 0x04, 0xdc, 0x4b, 0x8e, 0x51, 0xbc, 0xb0, 0x82, 0x35, 0xa6, 0xf4, 0x34,
         0x13, 0x32, 0xe4, 0xca, 0x60, 0x48, 0x2a, 0x4b, 0xa1, 0xa0, 0x3b, 0x3e, 0x65, 0x00, 0x8f, 0xc5,
         0xda, 0x76, 0xb7, 0x0b, 0xf1, 0x69, 0x0d, 0xb4, 0xea, 0xe2, 0x9c, 0x5f, 0x1b, 0xad, 0xd0, 0x3c,
         0x5c, 0xcf, 0x2a, 0x55, 0xd7, 0x05, 0xdd, 0xcd, 0x86, 0xd4, 0x49, 0x51, 0x1c, 0xeb, 0x7e, 0xc3,
         0x0b, 0xf1, 0x2b, 0x1f, 0xa3, 0x5b, 0x91, 0x3f, 0x9f, 0x74, 0x7a, 0x8a, 0xfd, 0x1b, 0x13, 0x0e,
         0x94, 0xbf, 0xf9, 0x4e, 0xff, 0xd0, 0x1a, 0x91, 0x73, 0x5c, 0xa1, 0x72, 0x6a, 0xcd, 0x0b, 0x19,
         0x7c, 0x4e, 0x5b, 0x03, 0x39, 0x36, 0x97, 0xe1, 0x26, 0x82, 0x6f, 0xb6, 0xbb, 0xde, 0x8e, 0xcc,
         0x1e, 0x08, 0x29, 0x85, 0x16, 0xe2, 0xc9, 0xed, 0x03, 0xff, 0x3c, 0x1b, 0x78, 0x60, 0xf6, 0xde,
         0x76, 0xd4, 0xce, 0xcd, 0x94, 0xc8, 0x11, 0x98, 0x55, 0xef, 0x52, 0x97, 0xca, 0x67, 0xe9, 0xf3,
         0xe7, 0xff, 0x72, 0xb1, 0xe9, 0x97, 0x85, 0xca, 0x0a, 0x7e, 0x77, 0x20, 0xc5, 0xb3, 0x6d, 0xc6,
         0xd7, 0x2c, 0xac, 0x95, 0x74, 0xc8, 0xcb, 0xbc, 0x2f, 0x80, 0x1e, 0x23, 0xe5, 0x6f, 0xd3, 0x44,
         0xb0, 0x7f, 0x22, 0x15, 0x4b, 0xeb, 0xa0, 0xf0, 0x8c, 0xe8, 0x89, 0x1e, 0x64, 0x3e, 0xd9, 0x95,
         0xc9, 0x4d, 0x9a, 0x69, 0xc9, 0xf1, 0xb5, 0xf4, 0x99, 0x02, 0x7a, 0x78, 0x57, 0x2a, 0xee, 0xbd,
         0x74, 0xd2, 0x0c, 0xc3, 0x98, 0x81, 0xc2, 0x13, 0xee, 0x77, 0x0b, 0x10, 0x10, 0xe4, 0xbe, 0xa7,
         0x18, 0x84, 0x69, 0x77, 0xae, 0x11, 0x9f, 0x7a, 0x02, 0x3a, 0xb5, 0x8c, 0xca, 0x0a, 0xd7, 0x52,
         0xaf, 0xe6, 0x56, 0xbb, 0x3c, 0x17, 0x25, 0x6a, 0x9f, 0x6e, 0x9b, 0xf1, 0x9f, 0xdd, 0x5a, 0x38,
         0xfc, 0x82, 0xbb, 0xe8, 0x72, 0xc5, 0x53, 0x9e, 0xdb, 0x60, 0x9e, 0xf4, 0xf7, 0x9c, 0x20, 0x3e,
         0xbb, 0x14, 0x0f, 0x2e, 0x58, 0x3c, 0xb2, 0xad, 0x15, 0xb4, 0xaa, 0x5b, 0x65, 0x50, 0x16, 0xa8,
         0x44, 0x92, 0x77, 0xdb, 0xd4, 0x77, 0xef, 0x2c, 0x8d, 0x6c, 0x01, 0x7d, 0xb7, 0x38, 0xb1, 0x8d,
         0xeb, 0x4a, 0x42, 0x7d, 0x19, 0x23, 0xce, 0x3f, 0xf2, 0x62, 0x73, 0x57, 0x79, 0xa4, 0x18, 0xf2,
         0x0a, 0x28, 0x2d, 0xf9, 0x20, 0x14, 0x7b, 0xea, 0xbe, 0x42, 0x1e, 0xe5, 0x31, 0x9d, 0x05, 0x68},
    },

    // #4) TextSize がブロックサイズの倍数ではないパターン 1
    {
        {0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0},
        {0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8, 0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0},
        16,
        {0x9a, 0x78, 0x56, 0x34, 0x12},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10},
        17,
        {0x6c, 0x16, 0x25, 0xdb, 0x46, 0x71, 0x52, 0x2d, 0x3d, 0x75, 0x99, 0x60, 0x1d, 0xe7, 0xca, 0x09,
         0xed},
    },

    // #5) TextSize がブロックサイズの倍数ではないパターン 2
    {
        {0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0},
        {0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8, 0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0},
        16,
        {0x9a, 0x78, 0x56, 0x34, 0x12},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11},
        18,
        {0xd0, 0x69, 0x44, 0x4b, 0x7a, 0x7e, 0x0c, 0xab, 0x09, 0xe2, 0x44, 0x47, 0xd2, 0x4d, 0xeb, 0x1f,
         0xed, 0xbf},
    },

    // #6) TextSize がブロックサイズの倍数ではないパターン 3
    {
        {0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0},
        {0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8, 0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0},
        16,
        {0x9a, 0x78, 0x56, 0x34, 0x12},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12},
        19,
        {0xe5, 0xdf, 0x13, 0x51, 0xc0, 0x54, 0x4b, 0xa1, 0x35, 0x0b, 0x33, 0x63, 0xcd, 0x8e, 0xf4, 0xbe,
         0xed, 0xbf, 0x9d},
    },

    // #7) TextSize がブロックサイズの倍数ではないパターン 4
    {
        {0xff, 0xfe, 0xfd, 0xfc, 0xfb, 0xfa, 0xf9, 0xf8, 0xf7, 0xf6, 0xf5, 0xf4, 0xf3, 0xf2, 0xf1, 0xf0},
        {0xbf, 0xbe, 0xbd, 0xbc, 0xbb, 0xba, 0xb9, 0xb8, 0xb7, 0xb6, 0xb5, 0xb4, 0xb3, 0xb2, 0xb1, 0xb0},
        16,
        {0x9a, 0x78, 0x56, 0x34, 0x12},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13},
        20,
        {0x9d, 0x84, 0xc8, 0x13, 0xf7, 0x19, 0xaa, 0x2c, 0x7b, 0xe3, 0xf6, 0x61, 0x71, 0xc7, 0xc5, 0xc2,
         0xed, 0xbf, 0x9d, 0xac},
    },

    // #8) 比較的長めのメッセージその 2
    {
        {0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef},
        {0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf},
        16,
        {0x21, 0x43, 0x65, 0x87, 0xa9},
        16,
        {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
         0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
         0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
         0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
         0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
         0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
         0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
         0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
         0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
         0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
         0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
         0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
         0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
         0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
         0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
         0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
         0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
         0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
         0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
         0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7, 0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
         0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
         0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff},
        512,
        {0x38, 0xb4, 0x58, 0x12, 0xef, 0x43, 0xa0, 0x5b, 0xd9, 0x57, 0xe5, 0x45, 0x90, 0x7e, 0x22, 0x3b,
         0x95, 0x4a, 0xb4, 0xaa, 0xf0, 0x88, 0x30, 0x3a, 0xd9, 0x10, 0xea, 0xdf, 0x14, 0xb4, 0x2b, 0xe6,
         0x8b, 0x24, 0x61, 0x14, 0x9d, 0x8c, 0x8b, 0xa8, 0x5f, 0x99, 0x2b, 0xe9, 0x70, 0xbc, 0x62, 0x1f,
         0x1b, 0x06, 0x57, 0x3f, 0x63, 0xe8, 0x67, 0xbf, 0x58, 0x75, 0xac, 0xaf, 0xa0, 0x4e, 0x42, 0xcc,
         0xbd, 0x7b, 0xd3, 0xc2, 0xa0, 0xfb, 0x1f, 0xff, 0x79, 0x1e, 0xc5, 0xec, 0x36, 0xc6, 0x6a, 0xe4,
         0xac, 0x1e, 0x80, 0x6d, 0x81, 0xfb, 0xf7, 0x09, 0xdb, 0xe2, 0x9e, 0x47, 0x1f, 0xad, 0x38, 0x54,
         0x9c, 0x8e, 0x66, 0xf5, 0x34, 0x5d, 0x7c, 0x1e, 0xb9, 0x4f, 0x40, 0x5d, 0x1e, 0xc7, 0x85, 0xcc,
         0x6f, 0x6a, 0x68, 0xf6, 0x25, 0x4d, 0xd8, 0x33, 0x9f, 0x9d, 0x84, 0x05, 0x7e, 0x01, 0xa1, 0x77,
         0x41, 0x99, 0x04, 0x82, 0x99, 0x95, 0x16, 0xb5, 0x61, 0x1a, 0x38, 0xf4, 0x1b, 0xb6, 0x47, 0x8e,
         0x6f, 0x17, 0x3f, 0x32, 0x08, 0x05, 0xdd, 0x71, 0xb1, 0x93, 0x2f, 0xc3, 0x33, 0xcb, 0x9e, 0xe3,
         0x99, 0x36, 0xbe, 0xea, 0x9a, 0xd9, 0x6f, 0xa1, 0x0f, 0xb4, 0x11, 0x2b, 0x90, 0x17, 0x34, 0xdd,
         0xad, 0x40, 0xbc, 0x18, 0x78, 0x99, 0x5f, 0x8e, 0x11, 0xae, 0xe7, 0xd1, 0x41, 0xa2, 0xf5, 0xd4,
         0x8b, 0x7a, 0x4e, 0x1e, 0x7f, 0x0b, 0x2c, 0x04, 0x83, 0x0e, 0x69, 0xa4, 0xfd, 0x13, 0x78, 0x41,
         0x1c, 0x2f, 0x28, 0x7e, 0xdf, 0x48, 0xc6, 0xc4, 0xe5, 0xc2, 0x47, 0xa1, 0x96, 0x80, 0xf7, 0xfe,
         0x41, 0xce, 0xfb, 0xd4, 0x9b, 0x58, 0x21, 0x06, 0xe3, 0x61, 0x6c, 0xbb, 0xe4, 0xdf, 0xb2, 0x34,
         0x4b, 0x2a, 0xe9, 0x51, 0x93, 0x91, 0xf3, 0xe0, 0xfb, 0x49, 0x22, 0x25, 0x4b, 0x1d, 0x6d, 0x2d,
         0x19, 0xc6, 0xd4, 0xd5, 0x37, 0xb3, 0xa2, 0x6f, 0x3b, 0xcc, 0x51, 0x58, 0x8b, 0x32, 0xf3, 0xec,
         0xa0, 0x82, 0x9b, 0x6a, 0x5a, 0xc7, 0x25, 0x78, 0xfb, 0x81, 0x4f, 0xb4, 0x3c, 0xf8, 0x0d, 0x64,
         0xa2, 0x33, 0xe3, 0xf9, 0x97, 0xa3, 0xf0, 0x26, 0x83, 0x34, 0x2f, 0x2b, 0x33, 0xd2, 0x5b, 0x49,
         0x25, 0x36, 0xb9, 0x3b, 0xec, 0xb2, 0xf5, 0xe1, 0xa8, 0xb8, 0x2f, 0x5b, 0x88, 0x33, 0x42, 0x72,
         0x9e, 0x8a, 0xe0, 0x9d, 0x16, 0x93, 0x88, 0x41, 0xa2, 0x1a, 0x97, 0xfb, 0x54, 0x3e, 0xea, 0x3b,
         0xbf, 0xf5, 0x9f, 0x13, 0xc1, 0xa1, 0x84, 0x49, 0xe3, 0x98, 0x70, 0x1c, 0x1a, 0xd5, 0x16, 0x48,
         0x34, 0x6c, 0xbc, 0x04, 0xc2, 0x7b, 0xb2, 0xda, 0x3b, 0x93, 0xa1, 0x37, 0x2c, 0xca, 0xe5, 0x48,
         0xfb, 0x53, 0xbe, 0xe4, 0x76, 0xf9, 0xe9, 0xc9, 0x17, 0x73, 0xb1, 0xbb, 0x19, 0x82, 0x83, 0x94,
         0xd5, 0x5d, 0x3e, 0x1a, 0x20, 0xed, 0x69, 0x11, 0x3a, 0x86, 0x0b, 0x68, 0x29, 0xff, 0xa8, 0x47,
         0x22, 0x46, 0x04, 0x43, 0x50, 0x70, 0x22, 0x1b, 0x25, 0x7e, 0x8d, 0xff, 0x78, 0x36, 0x15, 0xd2,
         0xca, 0xe4, 0x80, 0x3a, 0x93, 0xaa, 0x43, 0x34, 0xab, 0x48, 0x2a, 0x0a, 0xfa, 0xc9, 0xc0, 0xae,
         0xda, 0x70, 0xb4, 0x5a, 0x48, 0x1d, 0xf5, 0xde, 0xc5, 0xdf, 0x8c, 0xc0, 0xf4, 0x23, 0xc7, 0x7a,
         0x5f, 0xd4, 0x6c, 0xd3, 0x12, 0x02, 0x1d, 0x4b, 0x43, 0x88, 0x62, 0x41, 0x9a, 0x79, 0x1b, 0xe0,
         0x3b, 0xb4, 0xd9, 0x7c, 0x0e, 0x59, 0x57, 0x85, 0x42, 0x53, 0x1b, 0xa4, 0x66, 0xa8, 0x3b, 0xaf,
         0x92, 0xce, 0xfc, 0x15, 0x1b, 0x5c, 0xc1, 0x61, 0x1a, 0x16, 0x78, 0x93, 0x81, 0x9b, 0x63, 0xfb,
         0x8a, 0x6b, 0x18, 0xe8, 0x6d, 0xe6, 0x02, 0x90, 0xfa, 0x72, 0xb7, 0x97, 0xb0, 0xce, 0x59, 0xf3},
    },
};

const int TestVectorCount = sizeof(aesXtsTestVectors) / sizeof(aesXtsTestVectors[0]);

TEST(AesXtsTest, Encryption)
{
    nn::crypto::Aes128XtsEncryptor aesXts;
    nn::Bit8 output[512] = {0};
    size_t ret;

    for (int i = 0; i < TestVectorCount; i++)
    {
        aesXts.Initialize(aesXtsTestVectors[i].key1, aesXtsTestVectors[i].key2, aesXtsTestVectors[i].keySize,
                          aesXtsTestVectors[i].tweak, aesXtsTestVectors[i].tweakSize);

        size_t remaining = (aesXtsTestVectors[i].textSize % 16 + 16);

        ret = aesXts.Update(output, sizeof(output), aesXtsTestVectors[i].plainText, aesXtsTestVectors[i].textSize);
        EXPECT_TRUE(ret == aesXtsTestVectors[i].textSize - remaining); // Update では 1 ブロック分＋端数が処理されない

        ret = aesXts.Finalize(output + ret, sizeof(output) - ret);
        EXPECT_TRUE(ret == remaining);

        EXPECT_ARRAY_EQ(output, aesXtsTestVectors[i].cipherText, aesXtsTestVectors[i].textSize);
    }
}

TEST(AesXtsTest, Decryption)
{
    nn::crypto::Aes128XtsDecryptor aesXts;
    nn::Bit8 output[512] = {0};
    size_t ret;

    for (int i = 0; i < TestVectorCount; i++)
    {
        aesXts.Initialize(aesXtsTestVectors[i].key1, aesXtsTestVectors[i].key2, aesXtsTestVectors[i].keySize,
                          aesXtsTestVectors[i].tweak, aesXtsTestVectors[i].tweakSize);

        size_t remaining = (aesXtsTestVectors[i].textSize % 16 + 16);

        ret = aesXts.Update(output, sizeof(output), aesXtsTestVectors[i].cipherText, aesXtsTestVectors[i].textSize);
        EXPECT_TRUE(ret == aesXtsTestVectors[i].textSize - remaining); // Update では 1 ブロック分＋端数が処理されない

        ret = aesXts.Finalize(output + ret, sizeof(output) - ret);
        EXPECT_TRUE(ret == remaining);

        EXPECT_ARRAY_EQ(output, aesXtsTestVectors[i].plainText, aesXtsTestVectors[i].textSize);
    }
}

void EncryptDataBufferingTest(const AesXtsTestVector& testVector)
{
    nn::crypto::Aes128XtsEncryptor aesXts;
    nn::Bit8                       output[512] = {0};
    const size_t                   BlockSize = nn::crypto::Aes128XtsEncryptor::BlockSize;
    const size_t                   MaxInputSize = std::min((BlockSize * 4), testVector.textSize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesXts.Initialize(testVector.key1, testVector.key2, testVector.keySize,
                          testVector.tweak, testVector.tweakSize);

        const nn::Bit8* pInput = testVector.plainText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesXts.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        aesXts.Finalize(pOutput, outBufSize);

        EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);
    }
}

TEST(AesXtsTest, EncryptDataBufferingTest)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        EncryptDataBufferingTest(aesXtsTestVectors[i]);
    }
}

void DecryptDataBufferingTest(const AesXtsTestVector& testVector)
{
    nn::crypto::Aes128XtsDecryptor aesXts;
    nn::Bit8                       output[512] = {0};
    const size_t                   BlockSize = nn::crypto::Aes128XtsDecryptor::BlockSize;
    const size_t                   MaxInputSize = std::min((BlockSize * 4), testVector.textSize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesXts.Initialize(testVector.key1, testVector.key2, testVector.keySize,
                          testVector.tweak, testVector.tweakSize);

        const nn::Bit8* pInput = testVector.cipherText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesXts.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        aesXts.Finalize(pOutput, outBufSize);

        EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);
    }
}

TEST(AesXtsTest, DecryptDataBufferingTest)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        DecryptDataBufferingTest(aesXtsTestVectors[i]);
    }
}

TEST(AesXtsTest, UtilityFunctionInterface)
{
    nn::Bit8 output[512];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        nn::crypto::EncryptAes128Xts(output, sizeof(output),
                                     aesXtsTestVectors[i].key1, aesXtsTestVectors[i].key2, aesXtsTestVectors[i].keySize,
                                     aesXtsTestVectors[i].tweak, aesXtsTestVectors[i].tweakSize,
                                     aesXtsTestVectors[i].plainText, aesXtsTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesXtsTestVectors[i].cipherText, aesXtsTestVectors[i].textSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        nn::crypto::DecryptAes128Xts(output, sizeof(output),
                                     aesXtsTestVectors[i].key1, aesXtsTestVectors[i].key2, aesXtsTestVectors[i].keySize,
                                     aesXtsTestVectors[i].tweak, aesXtsTestVectors[i].tweakSize,
                                     aesXtsTestVectors[i].cipherText, aesXtsTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesXtsTestVectors[i].plainText, aesXtsTestVectors[i].textSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void XtsDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };
    const uint8_t iv[16] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    };

    // Encryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        // Tweak 暗号化用
        nn::crypto::AesEncryptor<KeySize> aes2;
        aes2.Initialize(key, KeySize);

        nn::crypto::XtsEncryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, &aes2, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~XtsEncryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
        EXPECT_ARRAY_NONZERO(&aes2, sizeof(aes2));
    }

    // Decryptor test
    {
        nn::crypto::AesDecryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        // Tweak 暗号化用
        nn::crypto::AesEncryptor<KeySize> aes2;
        aes2.Initialize(key, KeySize);

        nn::crypto::XtsDecryptor<nn::crypto::AesDecryptor<KeySize>> mode;
        mode.Initialize(&aes, &aes2, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~XtsDecryptor<nn::crypto::AesDecryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードがクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
        EXPECT_ARRAY_NONZERO(&aes2, sizeof(aes2));
    }
}

TEST(AesXtsTest, Destructor)
{
    XtsDestructorTest<16>();
    XtsDestructorTest<24>();
    XtsDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesXtsTest, ZeroByteEncryption)
{
    nn::crypto::AesEncryptor128                           aes1;
    nn::crypto::AesEncryptor128                           aes2;
    nn::crypto::XtsEncryptor<nn::crypto::AesEncryptor128> aesXts;
    nn::Bit8                                              output[32];

    // 初期化
    aes1.Initialize(aesXtsTestVectors[0].key1, aesXtsTestVectors[0].keySize);
    aes2.Initialize(aesXtsTestVectors[0].key2, aesXtsTestVectors[0].keySize);
    aesXts.Initialize(&aes1, &aes2, aesXtsTestVectors[0].tweak, aesXtsTestVectors[0].tweakSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    size_t ret = aesXts.Update(output, sizeof(output), nullptr, 0);
    EXPECT_EQ(ret, 0);

    ret = aesXts.Update(output, sizeof(output),
                        aesXtsTestVectors[0].plainText, aesXtsTestVectors[0].textSize);
    EXPECT_EQ(ret, aesXtsTestVectors[0].textSize - 16);

    ret = aesXts.Finalize(output + ret, sizeof(output) - ret);
    EXPECT_EQ(ret, 16);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesXtsTestVectors[0].cipherText, aesXtsTestVectors[0].textSize);
}

TEST(AesXtsTest, ZeroByteDecryption)
{
    nn::crypto::AesDecryptor128                           aes1;
    nn::crypto::AesEncryptor128                           aes2;
    nn::crypto::XtsDecryptor<nn::crypto::AesDecryptor128> aesXts;
    nn::Bit8                                              output[32];

    // 初期化
    aes1.Initialize(aesXtsTestVectors[0].key1, aesXtsTestVectors[0].keySize);
    aes2.Initialize(aesXtsTestVectors[0].key2, aesXtsTestVectors[0].keySize);
    aesXts.Initialize(&aes1, &aes2, aesXtsTestVectors[0].tweak, aesXtsTestVectors[0].tweakSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    size_t ret = aesXts.Update(output, sizeof(output), nullptr, 0);
    EXPECT_EQ(ret, 0);

    ret = aesXts.Update(output, sizeof(output),
                        aesXtsTestVectors[0].cipherText ,aesXtsTestVectors[0].textSize);
    EXPECT_EQ(ret, aesXtsTestVectors[0].textSize - 16);

    ret = aesXts.Finalize(output + ret, sizeof(output) - ret);
    EXPECT_EQ(ret, 16);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesXtsTestVectors[0].plainText, aesXtsTestVectors[0].textSize);
}
