﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/crypto/crypto_Aes128CmacGenerator.h>

#include "testCrypto_Util.h"

struct AesCmacTestVector
{
    nn::Bit8  key[32];
    size_t    keySize;
    nn::Bit8  plainText[512];
    size_t    textSize;
    nn::Bit8  mac[16];
};

const AesCmacTestVector aesCmacTestVectors[] =
{
    // #1) Message is NULL string
    {
        {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        16,
        {0},
        0,
        {0xbb, 0x1d, 0x69, 0x29, 0xe9, 0x59, 0x37, 0x28, 0x7f, 0xa3, 0x7d, 0x12, 0x9b, 0x75, 0x67, 0x46},
    },

    // #2) Message size is equal to one block size
    {
        {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a},
        16,
        {0x07, 0x0a, 0x16, 0xb4, 0x6b, 0x4d, 0x41, 0x44, 0xf7, 0x9b, 0xdd, 0x9d, 0xd0, 0x4a, 0x28, 0x7c},
    },

    // #3) Message size is not multiple of block size
    {
        {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
         0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
         0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11},
        40,
        {0xdf, 0xa6, 0x67, 0x47, 0xde, 0x9a, 0xe6, 0x30, 0x30, 0xca, 0x32, 0x61, 0x14, 0x97, 0xc8, 0x27},
    },

    // #4) Message size is mulitiple of block size
    {
        {0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c},
        16,
        {0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
         0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c, 0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51,
         0x30, 0xc8, 0x1c, 0x46, 0xa3, 0x5c, 0xe4, 0x11, 0xe5, 0xfb, 0xc1, 0x19, 0x1a, 0x0a, 0x52, 0xef,
         0xf6, 0x9f, 0x24, 0x45, 0xdf, 0x4f, 0x9b, 0x17, 0xad, 0x2b, 0x41, 0x7b, 0xe6, 0x6c, 0x37, 0x10},
        64,
        {0x51, 0xf0, 0xbe, 0xbf, 0x7e, 0x3b, 0x9d, 0x92, 0xfc, 0x49, 0x74, 0x17, 0x79, 0x36, 0x3c, 0xfe},
    },
};

const int TestVectorCount = sizeof(aesCmacTestVectors) / sizeof(aesCmacTestVectors[0]);

TEST(AesCmacTest, Basic)
{
    nn::crypto::Aes128CmacGenerator aesCmac;

    for (int i = 0; i < TestVectorCount; i++)
    {
        nn::Bit8 mac[16];

        EXPECT_GE(sizeof(mac), nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);

        aesCmac.Initialize(aesCmacTestVectors[i].key, aesCmacTestVectors[i].keySize);
        aesCmac.Update(aesCmacTestVectors[i].plainText, aesCmacTestVectors[i].textSize);
        aesCmac.GetMac(mac, sizeof(mac));

        EXPECT_ARRAY_EQ(mac, aesCmacTestVectors[i].mac, 16);
    }
}

TEST(AesCmacTest, DataBuffering)
{
    nn::crypto::Aes128CmacGenerator aesCmac;

    for (int i = 0; i < TestVectorCount; i++)
    {
        const size_t MaxInputSize = std::min((nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize * 4), aesCmacTestVectors[i].textSize);

        for (size_t inputSize = 1; inputSize < MaxInputSize; ++inputSize)
        {
            nn::Bit8 mac[16];

            EXPECT_GE(sizeof(mac), nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128>::MacSize);

            aesCmac.Initialize(aesCmacTestVectors[i].key, aesCmacTestVectors[i].keySize);

            const nn::Bit8* pInput = aesCmacTestVectors[i].plainText;
            size_t remaining = aesCmacTestVectors[i].textSize;
            while (remaining)
            {
                size_t processSize = std::min(inputSize, remaining);
                aesCmac.Update(pInput, processSize);
                pInput += processSize;
                remaining -= processSize;
            }

            aesCmac.GetMac(mac, sizeof(mac));

            EXPECT_ARRAY_EQ(mac, aesCmacTestVectors[i].mac, 16);
        }
    }
}

TEST(AesCmacTest, StateTransition)
{
    nn::crypto::Aes128CmacGenerator aesCmac;
    nn::Bit8                        mac[nn::crypto::Aes128CmacGenerator::MacSize];
    nn::Bit8                        mac2[nn::crypto::Aes128CmacGenerator::MacSize];

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // 初期化せずに Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(aesCmac.Update(aesCmacTestVectors[0].plainText, aesCmacTestVectors[0].textSize), "");

    // 初期化せずに GetMac が呼ばれても NG
    EXPECT_DEATH_IF_SUPPORTED(aesCmac.GetMac(mac, sizeof(mac)), "");
#endif

    // 初期化
    aesCmac.Initialize(aesCmacTestVectors[0].key, aesCmacTestVectors[0].keySize);

    // Update を呼ばずに GetMac を呼ぶのは大丈夫
    aesCmac.GetMac(mac, sizeof(mac));

    // GetMac は連続で呼んでも大丈夫で、同じ値が出力されるはず
    aesCmac.GetMac(mac2, sizeof(mac2));

    EXPECT_ARRAY_EQ(mac, mac2, nn::crypto::Aes128CmacGenerator::MacSize);

#if defined(NN_SDK_BUILD_DEBUG) || defined(NN_SDK_BUILD_DEVELOP)
    // GetMac の後に Update が呼ばれたら NG
    EXPECT_DEATH_IF_SUPPORTED(aesCmac.Update(aesCmacTestVectors[0].plainText, aesCmacTestVectors[0].textSize), "");
#endif
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void CmacDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };

    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CmacGenerator<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CmacGenerator<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }
}

TEST(AesCmacTest, Destructor)
{
    CmacDestructorTest<16>();
    CmacDestructorTest<24>();
    CmacDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesCmacTest, ZeroByteInput)
{
    nn::crypto::AesEncryptor128                            aes;
    nn::crypto::CmacGenerator<nn::crypto::AesEncryptor128> aesCmac;
    nn::Bit8                                               mac[16];

    // 初期化
    aes.Initialize(aesCmacTestVectors[0].key, aesCmacTestVectors[0].keySize);
    aesCmac.Initialize(&aes);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    aesCmac.Update(nullptr, 0);
    aesCmac.Update(aesCmacTestVectors[0].plainText, aesCmacTestVectors[0].textSize);

    // MAC の取得
    aesCmac.GetMac(mac, sizeof(mac));

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(mac, aesCmacTestVectors[0].mac, 16);
}

