﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/crypto/crypto_CbcEncryptor.h>
#include <nn/crypto/crypto_CbcDecryptor.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/crypto/crypto_Aes128CbcDecryptor.h>

#include "testCrypto_Util.h"

struct AesCbcTestVector
{
    nn::Bit8  key[32];
    size_t    keySize;
    nn::Bit8  iv[16];
    size_t    ivSize;
    nn::Bit8  plainText[160];
    size_t    textSize;
    nn::Bit8  cipherText[160];
};

/* http://csrc.nist.gov/groups/STM/cavp/documents/aes/aesmmt.zip
   に掲載されているテストベクトルから抜粋 */
const AesCbcTestVector aesCbcTestVectors[] =
{
    // #1) 鍵サイズ = 128bit、平文 = 1 ブロック
    {
        {0x1f, 0x8e, 0x49, 0x73, 0x95, 0x3f, 0x3f, 0xb0, 0xbd, 0x6b, 0x16, 0x66, 0x2e, 0x9a, 0x3c, 0x17},
        16,
        {0x2f, 0xe2, 0xb3, 0x33, 0xce, 0xda, 0x8f, 0x98, 0xf4, 0xa9, 0x9b, 0x40, 0xd2, 0xcd, 0x34, 0xa8},
        16,
        {0x45, 0xcf, 0x12, 0x96, 0x4f, 0xc8, 0x24, 0xab, 0x76, 0x61, 0x6a, 0xe2, 0xf4, 0xbf, 0x08, 0x22},
        16,
        {0x0f, 0x61, 0xc4, 0xd4, 0x4c, 0x51, 0x47, 0xc0, 0x3c, 0x19, 0x5a, 0xd7, 0xe2, 0xcc, 0x12, 0xb2},
    },

    // #2) 鍵サイズ = 128bit、平文 = 4 ブロック
    {
        {0xb7, 0xf3, 0xc9, 0x57, 0x6e, 0x12, 0xdd, 0x0d, 0xb6, 0x3e, 0x8f, 0x8f, 0xac, 0x2b, 0x9a, 0x39},
        16,
        {0xc8, 0x0f, 0x09, 0x5d, 0x8b, 0xb1, 0xa0, 0x60, 0x69, 0x9f, 0x7c, 0x19, 0x97, 0x4a, 0x1a, 0xa0},
        16,
        {0x9a, 0xc1, 0x99, 0x54, 0xce, 0x13, 0x19, 0xb3, 0x54, 0xd3, 0x22, 0x04, 0x60, 0xf7, 0x1c, 0x1e,
         0x37, 0x3f, 0x1c, 0xd3, 0x36, 0x24, 0x08, 0x81, 0x16, 0x0c, 0xfd, 0xe4, 0x6e, 0xbf, 0xed, 0x2e,
         0x79, 0x1e, 0x8d, 0x5a, 0x1a, 0x13, 0x6e, 0xbd, 0x1d, 0xc4, 0x69, 0xde, 0xc0, 0x0c, 0x41, 0x87,
         0x72, 0x2b, 0x84, 0x1c, 0xda, 0xbc, 0xb2, 0x2c, 0x1b, 0xe8, 0xa1, 0x46, 0x57, 0xda, 0x20, 0x0e},
        64,
        {0x19, 0xb9, 0x60, 0x97, 0x72, 0xc6, 0x3f, 0x33, 0x86, 0x08, 0xbf, 0x6e, 0xb5, 0x2c, 0xa1, 0x0b,
         0xe6, 0x50, 0x97, 0xf8, 0x9c, 0x1e, 0x09, 0x05, 0xc4, 0x24, 0x01, 0xfd, 0x47, 0x79, 0x1a, 0xe2,
         0xc5, 0x44, 0x0b, 0x2d, 0x47, 0x31, 0x16, 0xca, 0x78, 0xbd, 0x9f, 0xf2, 0xfb, 0x60, 0x15, 0xcf,
         0xd3, 0x16, 0x52, 0x4e, 0xae, 0x7d, 0xcb, 0x95, 0xae, 0x73, 0x8e, 0xbe, 0xae, 0x84, 0xa4, 0x67},
    },

    // #3) 鍵サイズ = 128bit、平文 = 10 ブロック
    {
        {0x2c, 0x14, 0x41, 0x37, 0x51, 0xc3, 0x1e, 0x27, 0x30, 0x57, 0x0b, 0xa3, 0x36, 0x1c, 0x78, 0x6b},
        16,
        {0x1d, 0xbb, 0xeb, 0x2f, 0x19, 0xab, 0xb4, 0x48, 0xaf, 0x84, 0x97, 0x96, 0x24, 0x4a, 0x19, 0xd7},
        16,
        {0x40, 0xd9, 0x30, 0xf9, 0xa0, 0x53, 0x34, 0xd9, 0x81, 0x6f, 0xe2, 0x04, 0x99, 0x9c, 0x3f, 0x82,
         0xa0, 0x3f, 0x6a, 0x04, 0x57, 0xa8, 0xc4, 0x75, 0xc9, 0x45, 0x53, 0xd1, 0xd1, 0x16, 0x69, 0x3a,
         0xdc, 0x61, 0x80, 0x49, 0xf0, 0xa7, 0x69, 0xa2, 0xee, 0xd6, 0xa6, 0xcb, 0x14, 0xc0, 0x14, 0x3e,
         0xc5, 0xcc, 0xcd, 0xbc, 0x8d, 0xec, 0x4c, 0xe5, 0x60, 0xcf, 0xd2, 0x06, 0x22, 0x57, 0x09, 0x32,
         0x6d, 0x4d, 0xe7, 0x94, 0x8e, 0x54, 0xd6, 0x03, 0xd0, 0x1b, 0x12, 0xd7, 0xfe, 0xd7, 0x52, 0xfb,
         0x23, 0xf1, 0xaa, 0x44, 0x94, 0xfb, 0xb0, 0x01, 0x30, 0xe9, 0xde, 0xd4, 0xe7, 0x7e, 0x37, 0xc0,
         0x79, 0x04, 0x2d, 0x82, 0x80, 0x40, 0xc3, 0x25, 0xb1, 0xa5, 0xef, 0xd1, 0x5f, 0xc8, 0x42, 0xe4,
         0x40, 0x14, 0xca, 0x43, 0x74, 0xbf, 0x38, 0xf3, 0xc3, 0xfc, 0x3e, 0xe3, 0x27, 0x73, 0x3b, 0x0c,
         0x8a, 0xee, 0x1a, 0xbc, 0xd0, 0x55, 0x77, 0x2f, 0x18, 0xdc, 0x04, 0x60, 0x3f, 0x7b, 0x2c, 0x1e,
         0xa6, 0x9f, 0xf6, 0x62, 0x36, 0x1f, 0x2b, 0xe0, 0xa1, 0x71, 0xbb, 0xdc, 0xea, 0x1e, 0x5d, 0x3f},
        160,
        {0x6b, 0xe8, 0xa1, 0x28, 0x00, 0x45, 0x5a, 0x32, 0x05, 0x38, 0x85, 0x3e, 0x0c, 0xba, 0x31, 0xbd,
         0x2d, 0x80, 0xea, 0x0c, 0x85, 0x16, 0x4a, 0x4c, 0x5c, 0x26, 0x1a, 0xe4, 0x85, 0x41, 0x7d, 0x93,
         0xef, 0xfe, 0x2e, 0xbc, 0x0d, 0x0a, 0x0b, 0x51, 0xd6, 0xea, 0x18, 0x63, 0x3d, 0x21, 0x0c, 0xf6,
         0x3c, 0x0c, 0x4d, 0xdb, 0xc2, 0x76, 0x07, 0xf2, 0xe8, 0x1e, 0xd9, 0x11, 0x31, 0x91, 0xef, 0x86,
         0xd5, 0x6f, 0x3b, 0x99, 0xbe, 0x6c, 0x41, 0x5a, 0x41, 0x50, 0x29, 0x9f, 0xb8, 0x46, 0xce, 0x71,
         0x60, 0xb4, 0x0b, 0x63, 0xba, 0xf1, 0x17, 0x9d, 0x19, 0x27, 0x5a, 0x2e, 0x83, 0x69, 0x83, 0x76,
         0xd2, 0x8b, 0x92, 0x54, 0x8c, 0x68, 0xe0, 0x6e, 0x6d, 0x99, 0x4e, 0x2c, 0x15, 0x01, 0xed, 0x29,
         0x70, 0x14, 0xe7, 0x02, 0xcd, 0xef, 0xee, 0x2f, 0x65, 0x64, 0x47, 0x70, 0x60, 0x09, 0x61, 0x4d,
         0x80, 0x1d, 0xe1, 0xca, 0xaf, 0x73, 0xf8, 0xb7, 0xfa, 0x56, 0xcf, 0x1b, 0xa9, 0x4b, 0x63, 0x19,
         0x33, 0xbb, 0xe5, 0x77, 0x62, 0x43, 0x80, 0x85, 0x0f, 0x11, 0x74, 0x35, 0xa0, 0x35, 0x5b, 0x2b},
    },

    // #4) 鍵サイズ = 192bit、平文 = 1 ブロック
    {
        {0xba, 0x75, 0xf4, 0xd1, 0xd9, 0xd7, 0xcf, 0x7f, 0x55, 0x14, 0x45, 0xd5, 0x6c, 0xc1, 0xa8, 0xab,
         0x2a, 0x07, 0x8e, 0x15, 0xe0, 0x49, 0xdc, 0x2c},
        24,
        {0x53, 0x1c, 0xe7, 0x81, 0x76, 0x40, 0x16, 0x66, 0xaa, 0x30, 0xdb, 0x94, 0xec, 0x4a, 0x30, 0xeb},
        16,
        {0xc5, 0x1f, 0xc2, 0x76, 0x77, 0x4d, 0xad, 0x94, 0xbc, 0xdc, 0x1d, 0x28, 0x91, 0xec, 0x86, 0x68},
        16,
        {0x70, 0xdd, 0x95, 0xa1, 0x4e, 0xe9, 0x75, 0xe2, 0x39, 0xdf, 0x36, 0xff, 0x4a, 0xee, 0x1d, 0x5d},
    },

    // #5) 鍵サイズ = 192bit、平文 = 4 ブロック
    {
        {0x06, 0x7b, 0xb1, 0x7b, 0x4d, 0xf7, 0x85, 0x69, 0x7e, 0xac, 0xcf, 0x96, 0x1f, 0x98, 0xe2, 0x12,
         0xcb, 0x75, 0xe6, 0x79, 0x7c, 0xe9, 0x35, 0xcb},
        24,
        {0x8b, 0x59, 0xc9, 0x20, 0x9c, 0x52, 0x9c, 0xa8, 0x39, 0x1c, 0x9f, 0xc0, 0xce, 0x03, 0x3c, 0x38},
        16,
        {0xdb, 0x37, 0x85, 0xa8, 0x89, 0xb4, 0xbd, 0x38, 0x77, 0x54, 0xda, 0x22, 0x2f, 0x0e, 0x4c, 0x2d,
         0x2b, 0xfe, 0x0d, 0x79, 0xe0, 0x5b, 0xc9, 0x10, 0xfb, 0xa9, 0x41, 0xbe, 0xea, 0x30, 0xf1, 0x23,
         0x9e, 0xac, 0xf0, 0x06, 0x8f, 0x46, 0x19, 0xec, 0x01, 0xc3, 0x68, 0xe9, 0x86, 0xfc, 0xa6, 0xb7,
         0xc5, 0x8e, 0x49, 0x05, 0x79, 0xd2, 0x96, 0x11, 0xbd, 0x10, 0x08, 0x79, 0x86, 0xef, 0xf5, 0x4f},
        64,
        {0xd5, 0xf5, 0x58, 0x97, 0x60, 0xbf, 0x9c, 0x76, 0x22, 0x28, 0xfd, 0xe2, 0x36, 0xde, 0x1f, 0xa2,
         0xdd, 0x2d, 0xad, 0x44, 0x8d, 0xb3, 0xfa, 0x9b, 0xe0, 0xc4, 0x19, 0x6e, 0xfd, 0x46, 0xa3, 0x5c,
         0x84, 0xdd, 0x1a, 0xc7, 0x7d, 0x9d, 0xb5, 0x8c, 0x95, 0x91, 0x8c, 0xb3, 0x17, 0xa6, 0x43, 0x0a,
         0x08, 0xd2, 0xfb, 0x6a, 0x8e, 0x8b, 0x0f, 0x1c, 0x9b, 0x72, 0xc7, 0xa3, 0x44, 0xdc, 0x34, 0x9f},
    },

    // #6) 鍵サイズ = 192bit、平文 = 10 ブロック
    {
        {0x16, 0x2a, 0xd5, 0x0e, 0xe6, 0x4a, 0x07, 0x02, 0xaa, 0x55, 0x1f, 0x57, 0x1d, 0xed, 0xc1, 0x6b,
         0x2c, 0x1b, 0x6a, 0x1e, 0x4d, 0x4b, 0x5e, 0xee},
        24,
        {0x24, 0x40, 0x80, 0x38, 0x16, 0x1a, 0x2c, 0xca, 0xe0, 0x7b, 0x02, 0x9b, 0xb6, 0x63, 0x55, 0xc1},
        16,
        {0xbe, 0x8a, 0xbf, 0x00, 0x90, 0x13, 0x63, 0x98, 0x7a, 0x82, 0xcc, 0x77, 0xd0, 0xec, 0x91, 0x69,
         0x7b, 0xa3, 0x85, 0x7f, 0x9e, 0x4f, 0x84, 0xbd, 0x79, 0x40, 0x6c, 0x13, 0x8d, 0x02, 0x69, 0x8f,
         0x00, 0x32, 0x76, 0xd0, 0x44, 0x91, 0x20, 0xbe, 0xf4, 0x57, 0x8d, 0x78, 0xfe, 0xca, 0xbe, 0x8e,
         0x07, 0x0e, 0x11, 0x71, 0x0b, 0x3f, 0x0a, 0x27, 0x44, 0xbd, 0x52, 0x43, 0x4e, 0xc7, 0x00, 0x15,
         0x88, 0x4c, 0x18, 0x1e, 0xbd, 0xfd, 0x51, 0xc6, 0x04, 0xa7, 0x1c, 0x52, 0xe4, 0xc0, 0xe1, 0x10,
         0xbc, 0x40, 0x8c, 0xd4, 0x62, 0xb2, 0x48, 0xa8, 0x0b, 0x8a, 0x8a, 0xc0, 0x6b, 0xb9, 0x52, 0xac,
         0x1d, 0x7f, 0xae, 0xd1, 0x44, 0x80, 0x7f, 0x1a, 0x73, 0x1b, 0x7f, 0xeb, 0xca, 0xf7, 0x83, 0x57,
         0x62, 0xde, 0xfe, 0x92, 0xec, 0xcf, 0xc7, 0xa9, 0x94, 0x4e, 0x1c, 0x70, 0x2c, 0xff, 0xe6, 0xbc,
         0x86, 0x73, 0x3e, 0xd3, 0x21, 0x42, 0x31, 0x21, 0x08, 0x5a, 0xc0, 0x2d, 0xf8, 0x96, 0x2b, 0xcb,
         0xc1, 0x93, 0x70, 0x92, 0xee, 0xbf, 0x0e, 0x90, 0xa8, 0xb2, 0x0e, 0x3d, 0xd8, 0xc2, 0x44, 0xae},
        160,
        {0xc8, 0x2c, 0xf2, 0xc4, 0x76, 0xde, 0xa8, 0xcb, 0x6a, 0x6e, 0x60, 0x7a, 0x40, 0xd2, 0xf0, 0x39,
         0x1b, 0xe8, 0x2e, 0xa9, 0xec, 0x84, 0xa5, 0x37, 0xa6, 0x82, 0x0f, 0x9a, 0xfb, 0x99, 0x7b, 0x76,
         0x39, 0x7d, 0x00, 0x54, 0x24, 0xfa, 0xa6, 0xa7, 0x4d, 0xc4, 0xe8, 0xc7, 0xaa, 0x4a, 0x89, 0x00,
         0x69, 0x0f, 0x89, 0x4b, 0x6d, 0x1d, 0xca, 0x80, 0x67, 0x53, 0x93, 0xd2, 0x24, 0x3a, 0xda, 0xc7,
         0x62, 0xf1, 0x59, 0x30, 0x1e, 0x35, 0x7e, 0x98, 0xb7, 0x24, 0x76, 0x23, 0x10, 0xcd, 0x5a, 0x7b,
         0xaf, 0xe1, 0xc2, 0xa0, 0x30, 0xdb, 0xa4, 0x6f, 0xd9, 0x3a, 0x9f, 0xdb, 0x89, 0xcc, 0x13, 0x2c,
         0xa9, 0xc1, 0x7d, 0xc7, 0x20, 0x31, 0xec, 0x68, 0x22, 0xee, 0x5a, 0x9d, 0x99, 0xdb, 0xca, 0x66,
         0xc7, 0x84, 0xc0, 0x1b, 0x08, 0x85, 0xcb, 0xb6, 0x2e, 0x29, 0xd9, 0x78, 0x01, 0x92, 0x7e, 0xc4,
         0x15, 0xa5, 0xd2, 0x15, 0x15, 0x8d, 0x32, 0x5f, 0x9e, 0xe6, 0x89, 0x43, 0x7a, 0xd1, 0xb7, 0x68,
         0x4a, 0xd3, 0x3c, 0x0d, 0x92, 0x73, 0x94, 0x51, 0xac, 0x87, 0xf3, 0x9f, 0xf8, 0xc3, 0x1b, 0x84},
    },

    // #7) 鍵サイズ = 256bit、平文 = 1 ブロック
    {
        {0x6e, 0xd7, 0x6d, 0x2d, 0x97, 0xc6, 0x9f, 0xd1, 0x33, 0x95, 0x89, 0x52, 0x39, 0x31, 0xf2, 0xa6,
         0xcf, 0xf5, 0x54, 0xb1, 0x5f, 0x73, 0x8f, 0x21, 0xec, 0x72, 0xdd, 0x97, 0xa7, 0x33, 0x09, 0x07},
        32,
        {0x85, 0x1e, 0x87, 0x64, 0x77, 0x6e, 0x67, 0x96, 0xaa, 0xb7, 0x22, 0xdb, 0xb6, 0x44, 0xac, 0xe8},
        16,
        {0x62, 0x82, 0xb8, 0xc0, 0x5c, 0x5c, 0x15, 0x30, 0xb9, 0x7d, 0x48, 0x16, 0xca, 0x43, 0x47, 0x62},
        16,
        {0x6a, 0xcc, 0x04, 0x14, 0x2e, 0x10, 0x0a, 0x65, 0xf5, 0x1b, 0x97, 0xad, 0xf5, 0x17, 0x2c, 0x41},
    },

    // #8) 鍵サイズ = 256bit、平文 = 4 ブロック
    {
        {0x04, 0x93, 0xff, 0x63, 0x71, 0x08, 0xaf, 0x6a, 0x5b, 0x8e, 0x90, 0xac, 0x1f, 0xdf, 0x03, 0x5a,
         0x3d, 0x4b, 0xaf, 0xd1, 0xaf, 0xb5, 0x73, 0xbe, 0x7a, 0xde, 0x9e, 0x86, 0x82, 0xe6, 0x63, 0xe5},
        32,
        {0xc0, 0xcd, 0x2b, 0xeb, 0xcc, 0xbb, 0x6c, 0x49, 0x92, 0x0b, 0xd5, 0x48, 0x2a, 0xc7, 0x56, 0xe8},
        16,
        {0x8b, 0x37, 0xf9, 0x14, 0x8d, 0xf4, 0xbb, 0x25, 0x95, 0x6b, 0xe6, 0x31, 0x0c, 0x73, 0xc8, 0xdc,
         0x58, 0xea, 0x97, 0x14, 0xff, 0x49, 0xb6, 0x43, 0x10, 0x7b, 0x34, 0xc9, 0xbf, 0xf0, 0x96, 0xa9,
         0x4f, 0xed, 0xd6, 0x82, 0x35, 0x26, 0xab, 0xc2, 0x7a, 0x8e, 0x0b, 0x16, 0x61, 0x6e, 0xee, 0x25,
         0x4a, 0xb4, 0x56, 0x7d, 0xd6, 0x8e, 0x8c, 0xcd, 0x4c, 0x38, 0xac, 0x56, 0x3b, 0x13, 0x63, 0x9c},
        64,
        {0x05, 0xd5, 0xc7, 0x77, 0x29, 0x42, 0x1b, 0x08, 0xb7, 0x37, 0xe4, 0x11, 0x19, 0xfa, 0x44, 0x38,
         0xd1, 0xf5, 0x70, 0xcc, 0x77, 0x2a, 0x4d, 0x6c, 0x3d, 0xf7, 0xff, 0xed, 0xa0, 0x38, 0x4e, 0xf8,
         0x42, 0x88, 0xce, 0x37, 0xfc, 0x4c, 0x4c, 0x7d, 0x11, 0x25, 0xa4, 0x99, 0xb0, 0x51, 0x36, 0x4c,
         0x38, 0x9f, 0xd6, 0x39, 0xbd, 0xda, 0x64, 0x7d, 0xaa, 0x3b, 0xda, 0xda, 0xb2, 0xeb, 0x55, 0x94},
    },

    // #9) 鍵サイズ = 256bit、平文 = 10 ブロック
    {
        {0x48, 0xbe, 0x59, 0x7e, 0x63, 0x2c, 0x16, 0x77, 0x23, 0x24, 0xc8, 0xd3, 0xfa, 0x1d, 0x9c, 0x5a,
         0x9e, 0xcd, 0x01, 0x0f, 0x14, 0xec, 0x5d, 0x11, 0x0d, 0x3b, 0xfe, 0xc3, 0x76, 0xc5, 0x53, 0x2b},
        32,
        {0xd6, 0xd5, 0x81, 0xb8, 0xcf, 0x04, 0xeb, 0xd3, 0xb6, 0xea, 0xa1, 0xb5, 0x3f, 0x04, 0x7e, 0xe1},
        16,
        {0x0c, 0x63, 0xd4, 0x13, 0xd3, 0x86, 0x45, 0x70, 0xe7, 0x0b, 0xb6, 0x61, 0x8b, 0xf8, 0xa4, 0xb9,
         0x58, 0x55, 0x86, 0x68, 0x8c, 0x32, 0xbb, 0xa0, 0xa5, 0xec, 0xc1, 0x36, 0x2f, 0xad, 0xa7, 0x4a,
         0xda, 0x32, 0xc5, 0x2a, 0xcf, 0xd1, 0xaa, 0x74, 0x44, 0xba, 0x56, 0x7b, 0x4e, 0x7d, 0xaa, 0xec,
         0xf7, 0xcc, 0x1c, 0xb2, 0x91, 0x82, 0xaf, 0x16, 0x4a, 0xe5, 0x23, 0x2b, 0x00, 0x28, 0x68, 0x69,
         0x56, 0x35, 0x59, 0x98, 0x07, 0xa9, 0xa7, 0xf0, 0x7a, 0x1f, 0x13, 0x7e, 0x97, 0xb1, 0xe1, 0xc9,
         0xda, 0xbc, 0x89, 0xb6, 0xa5, 0xe4, 0xaf, 0xa9, 0xdb, 0x58, 0x55, 0xed, 0xaa, 0x57, 0x50, 0x56,
         0xa8, 0xf4, 0xf8, 0x24, 0x22, 0x16, 0x24, 0x2b, 0xb0, 0xc2, 0x56, 0x31, 0x0d, 0x9d, 0x32, 0x98,
         0x26, 0xac, 0x35, 0x3d, 0x71, 0x5f, 0xa3, 0x9f, 0x80, 0xce, 0xc1, 0x44, 0xd6, 0x42, 0x45, 0x58,
         0xf9, 0xf7, 0x0b, 0x98, 0xc9, 0x20, 0x09, 0x6e, 0x0f, 0x2c, 0x85, 0x5d, 0x59, 0x48, 0x85, 0xa0,
         0x06, 0x25, 0x88, 0x0e, 0x9d, 0xfb, 0x73, 0x41, 0x63, 0xce, 0xce, 0xf7, 0x2c, 0xf0, 0x30, 0xb8},
        160,
        {0xfc, 0x58, 0x73, 0xe5, 0x0d, 0xe8, 0xfa, 0xf4, 0xc6, 0xb8, 0x4b, 0xa7, 0x07, 0xb0, 0x85, 0x4e,
         0x9d, 0xb9, 0xab, 0x2e, 0x9f, 0x7d, 0x70, 0x7f, 0xbb, 0xa3, 0x38, 0xc6, 0x84, 0x3a, 0x18, 0xfc,
         0x6f, 0xac, 0xeb, 0xaf, 0x66, 0x3d, 0x26, 0x29, 0x6f, 0xb3, 0x29, 0xb4, 0xd2, 0x6f, 0x18, 0x49,
         0x4c, 0x79, 0xe0, 0x9e, 0x77, 0x96, 0x47, 0xf9, 0xba, 0xfa, 0x87, 0x48, 0x96, 0x30, 0xd7, 0x9f,
         0x43, 0x01, 0x61, 0x0c, 0x23, 0x00, 0xc1, 0x9d, 0xbf, 0x31, 0x48, 0xb7, 0xca, 0xc8, 0xc4, 0xf4,
         0x94, 0x41, 0x02, 0x75, 0x4f, 0x33, 0x2e, 0x92, 0xb6, 0xf7, 0xc5, 0xe7, 0x5b, 0xc6, 0x17, 0x9e,
         0xb8, 0x77, 0xa0, 0x78, 0xd4, 0x71, 0x90, 0x09, 0x02, 0x17, 0x44, 0xc1, 0x4f, 0x13, 0xfd, 0x2a,
         0x55, 0xa2, 0xb9, 0xc4, 0x4d, 0x18, 0x00, 0x06, 0x85, 0xa8, 0x45, 0xa4, 0xf6, 0x32, 0xc7, 0xc5,
         0x6a, 0x77, 0x30, 0x6e, 0xfa, 0x66, 0xa2, 0x4d, 0x05, 0xd0, 0x88, 0xdc, 0xd7, 0xc1, 0x3f, 0xe2,
         0x4f, 0xc4, 0x47, 0x27, 0x59, 0x65, 0xdb, 0x9e, 0x4d, 0x37, 0xfb, 0xc9, 0x30, 0x44, 0x48, 0xcd},
    },
};

const int TestVectorCount = sizeof(aesCbcTestVectors) / sizeof(aesCbcTestVectors[0]);

template <typename T>
void EncryptionTest(const AesCbcTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CbcEncryptor<T> aesCbc;
    nn::Bit8                    output[160];

    EXPECT_GE(sizeof(output), nn::crypto::CbcEncryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCbc.Initialize(&aes, testVector.iv, testVector.ivSize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.plainText, testVector.textSize);
        aesCbc.Update(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aesCbc.Update(output, sizeof(output), testVector.plainText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);
}

template <typename T>
void DecryptionTest(const AesCbcTestVector& testVector, bool isInPlace)
{
    T                           aes;
    nn::crypto::CbcDecryptor<T> aesCbc;
    nn::Bit8                    output[160];

    EXPECT_GE(sizeof(output), nn::crypto::CbcDecryptor<T>::BlockSize);

    aes.Initialize(testVector.key, testVector.keySize);
    aesCbc.Initialize(&aes, testVector.iv, testVector.ivSize);
    if (isInPlace)
    {
        std::memcpy(output, testVector.cipherText, testVector.textSize);
        aesCbc.Update(output, sizeof(output), output, testVector.textSize);
    }
    else
    {
        aesCbc.Update(output, sizeof(output), testVector.cipherText, testVector.textSize);
    }
    EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);
}

template <typename T>
void EncryptDataBufferingTest(const AesCbcTestVector& testVector)
{
    T                           aes;
    nn::crypto::CbcEncryptor<T> aesCbc;
    nn::Bit8                    output[160];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesCbc.Initialize(&aes, testVector.iv, testVector.ivSize);

        const nn::Bit8* pInput = testVector.plainText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCbc.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        EXPECT_ARRAY_EQ(output, testVector.cipherText, testVector.textSize);
    }
}

template <typename T>
void DecryptDataBufferingTest(const AesCbcTestVector& testVector)
{
    T                           aes;
    nn::crypto::CbcDecryptor<T> aesCbc;
    nn::Bit8                    output[160];
    const size_t                MaxInputSize = std::min((T::BlockSize * 4), testVector.textSize);

    aes.Initialize(testVector.key, testVector.keySize);

    // 複数ブロック＋端数の処理までテストするため、適当なブロックサイズ分までの各入力サイズについてテストする
    for (size_t inputSize = 1; inputSize < MaxInputSize; inputSize++)
    {
        aesCbc.Initialize(&aes, testVector.iv, testVector.ivSize);

        const nn::Bit8* pInput = testVector.cipherText;
        nn::Bit8*       pOutput = output;
        size_t          outBufSize = sizeof(output);
        size_t          remaining = testVector.textSize;
        while (remaining)
        {
            size_t processSize = std::min(inputSize, remaining);
            size_t ret = aesCbc.Update(pOutput, outBufSize, pInput, processSize);
            pInput += processSize;
            remaining -= processSize;
            // 処理された分だけ出力バッファを進める
            pOutput += ret;
            outBufSize -= ret;
        }

        EXPECT_ARRAY_EQ(output, testVector.plainText, testVector.textSize);
    }
}

/**
  @brief   AES-CBC による暗号化をテストします。

  @details
  NIST が公表しているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCbcTest, Encryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            EncryptionTest<nn::crypto::AesEncryptor128>(aesCbcTestVectors[i], false);
            break;
        case 24:
            EncryptionTest<nn::crypto::AesEncryptor192>(aesCbcTestVectors[i], false);
            break;
        case 32:
            EncryptionTest<nn::crypto::AesEncryptor256>(aesCbcTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CBC による復号化をテストします。

  @details
  NIST が公表しているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCbcTest, Decryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            DecryptionTest<nn::crypto::AesDecryptor128>(aesCbcTestVectors[i], false);
            break;
        case 24:
            DecryptionTest<nn::crypto::AesDecryptor192>(aesCbcTestVectors[i], false);
            break;
        case 32:
            DecryptionTest<nn::crypto::AesDecryptor256>(aesCbcTestVectors[i], false);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CBC による同一入出力バッファ上での暗号化をテストします。

  @details
  NIST が公表しているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCbcTest, InPlaceEncryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            EncryptionTest<nn::crypto::AesEncryptor128>(aesCbcTestVectors[i], true);
            break;
        case 24:
            EncryptionTest<nn::crypto::AesEncryptor192>(aesCbcTestVectors[i], true);
            break;
        case 32:
            EncryptionTest<nn::crypto::AesEncryptor256>(aesCbcTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CBC による同一入出力バッファ上での復号化をテストします。

  @details
  NIST が公表しているテストベクトルを用いて 128bit、192bit、256bit
  のそれぞれの鍵サイズで正しく暗号化できることをテストします。
 */
TEST(AesCbcTest, InPlaceDecryption)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            DecryptionTest<nn::crypto::AesDecryptor128>(aesCbcTestVectors[i], true);
            break;
        case 24:
            DecryptionTest<nn::crypto::AesDecryptor192>(aesCbcTestVectors[i], true);
            break;
        case 32:
            DecryptionTest<nn::crypto::AesDecryptor256>(aesCbcTestVectors[i], true);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CBC による暗号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく暗号化できることをテストします。
 */
TEST(AesCbcTest, EncryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor128>(aesCbcTestVectors[i]);
            break;
        case 24:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor192>(aesCbcTestVectors[i]);
            break;
        case 32:
            EncryptDataBufferingTest<nn::crypto::AesEncryptor256>(aesCbcTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   AES-CBC による復号化を入力サイズを変えてテストします。

  @details
  適当な大きさまで入力サイズを変えて、正しく復号化できることをテストします。
 */
TEST(AesCbcTest, DecryptDataBuffering)
{
    for (int i = 0; i < TestVectorCount; i++)
    {
        switch (aesCbcTestVectors[i].keySize)
        {
        case 16:
            DecryptDataBufferingTest<nn::crypto::AesDecryptor128>(aesCbcTestVectors[i]);
            break;
        case 24:
            DecryptDataBufferingTest<nn::crypto::AesDecryptor192>(aesCbcTestVectors[i]);
            break;
        case 32:
            DecryptDataBufferingTest<nn::crypto::AesDecryptor256>(aesCbcTestVectors[i]);
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

/**
  @brief   Aes128CbcEncryptor/Decryptor による暗号化・復号化をテストします。
 */
TEST(AesCbcTest, UtilityClassInterface)
{
    nn::Bit8 output[160];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCbcTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CbcEncryptor aes128CbcEnc;

        aes128CbcEnc.Initialize(aesCbcTestVectors[i].key, aesCbcTestVectors[i].keySize,
                                aesCbcTestVectors[i].iv, aesCbcTestVectors[i].ivSize);
        aes128CbcEnc.Update(output, sizeof(output),
                            aesCbcTestVectors[i].plainText, aesCbcTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCbcTestVectors[i].cipherText, aesCbcTestVectors[i].textSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCbcTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::Aes128CbcDecryptor aes128CbcDec;

        aes128CbcDec.Initialize(aesCbcTestVectors[i].key, aesCbcTestVectors[i].keySize,
                                aesCbcTestVectors[i].iv, aesCbcTestVectors[i].ivSize);
        aes128CbcDec.Update(output, sizeof(output),
                            aesCbcTestVectors[i].cipherText, aesCbcTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCbcTestVectors[i].plainText, aesCbcTestVectors[i].textSize);
    }
}

/**
  @brief   ユーティリティ関数による暗号化・復号化をテストします。
 */
TEST(AesCbcTest, UtilityFunctionInterface)
{
    nn::Bit8 output[160];

    // 暗号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCbcTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::EncryptAes128Cbc(output, sizeof(output),
                                     aesCbcTestVectors[i].key, aesCbcTestVectors[i].keySize,
                                     aesCbcTestVectors[i].iv, aesCbcTestVectors[i].ivSize,
                                     aesCbcTestVectors[i].plainText, aesCbcTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCbcTestVectors[i].cipherText, aesCbcTestVectors[i].textSize);
    }

    // 復号化のテスト
    for (int i = 0; i < TestVectorCount; i++)
    {
        // 128bit 鍵のみテスト
        if (aesCbcTestVectors[i].keySize != 16)
        {
            continue;
        }

        nn::crypto::DecryptAes128Cbc(output, sizeof(output),
                                     aesCbcTestVectors[i].key, aesCbcTestVectors[i].keySize,
                                     aesCbcTestVectors[i].iv, aesCbcTestVectors[i].ivSize,
                                     aesCbcTestVectors[i].cipherText, aesCbcTestVectors[i].textSize);

        EXPECT_ARRAY_EQ(output, aesCbcTestVectors[i].plainText, aesCbcTestVectors[i].textSize);
    }
}

/**
  @brief   デストラクタで内部データがクリアされることをテストします。
 */
template <size_t KeySize>
void CbcDestructorTest()
{
    // Dummy key and iv
    const uint8_t key[32] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
    };
    const uint8_t iv[16] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
    };

    // Encryptor test
    {
        nn::crypto::AesEncryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CbcEncryptor<nn::crypto::AesEncryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CbcEncryptor<nn::crypto::AesEncryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードのほうだけクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }

    // Decryptor test
    {
        nn::crypto::AesDecryptor<KeySize> aes;
        aes.Initialize(key, KeySize);

        nn::crypto::CbcDecryptor<nn::crypto::AesDecryptor<KeySize>> mode;
        mode.Initialize(&aes, iv, mode.IvSize);

        // 明示的にデストラクタを呼んで呼び出し前後でのメモリクリアを確認する
        EXPECT_ARRAY_NONZERO(&mode, sizeof(mode));
        mode.~CbcDecryptor<nn::crypto::AesDecryptor<KeySize>>();
        EXPECT_ARRAY_ZERO(&mode, sizeof(mode));

        // AES 本体は暗号利用モードがクリアされてもクリアされていない
        EXPECT_ARRAY_NONZERO(&aes, sizeof(aes));
    }
}

TEST(AesCbcTest, Destructor)
{
    CbcDestructorTest<16>();
    CbcDestructorTest<24>();
    CbcDestructorTest<32>();
}

/**
  @brief   Update に 0 バイトが渡されても正常に計算できることをチェックします。
 */
TEST(AesCbcTest, ZeroByteEncryption)
{
    nn::crypto::AesEncryptor128                           aes;
    nn::crypto::CbcEncryptor<nn::crypto::AesEncryptor128> aesCbc;
    nn::Bit8                                              output[32];

    // 初期化
    aes.Initialize(aesCbcTestVectors[0].key, aesCbcTestVectors[0].keySize);
    aesCbc.Initialize(&aes, aesCbcTestVectors[0].iv, aesCbcTestVectors[0].ivSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    EXPECT_EQ(aesCbc.Update(output, sizeof(output), nullptr, 0), 0);

    aesCbc.Update(output, sizeof(output),
                  aesCbcTestVectors[0].plainText, aesCbcTestVectors[0].textSize);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesCbcTestVectors[0].cipherText, aesCbcTestVectors[0].textSize);
}

TEST(AesCbcTest, ZeroByteDecryption)
{
    nn::crypto::AesDecryptor128                           aes;
    nn::crypto::CbcDecryptor<nn::crypto::AesDecryptor128> aesCbc;
    nn::Bit8                                              output[32];

    // 初期化
    aes.Initialize(aesCbcTestVectors[0].key, aesCbcTestVectors[0].keySize);
    aesCbc.Initialize(&aes, aesCbcTestVectors[0].iv, aesCbcTestVectors[0].ivSize);

    // 平文の入力前に 0 バイトの Update を呼ぶ
    EXPECT_EQ(aesCbc.Update(output, sizeof(output), nullptr, 0), 0);

    aesCbc.Update(output, sizeof(output),
                  aesCbcTestVectors[0].cipherText, aesCbcTestVectors[0].textSize);

    // 出力結果が正しいことを確認
    EXPECT_ARRAY_EQ(output, aesCbcTestVectors[0].plainText, aesCbcTestVectors[0].textSize);
}
